#******************************************************************************
#
#       Copyright:      2005-2025 Paul Obermeier (obermeier@tcl3d.org)
#
#                       See the file "Tcl3D_License.txt" for information on
#                       usage and redistribution of this file, and for a
#                       DISCLAIMER OF ALL WARRANTIES.
#
#       Module:         Tcl3D extension package
#       Filename:       img2pkg.tcl
#
#       Author:         Paul Obermeier
#
#       Description:    This script expects PNG and GIF image filenames or directory
#                       names as parameters and writes a namespaced Tcl package
#                       to stdout.
#                       If a directory name is given, all files with suffix
#                       ".png" and ".gif" are included in the generated package.
#
#                       Assume the package name is "::gaugeImg" and one of the
#                       image files is called "PwrdByTcl.png".
#                       The generated package then provides functions of the
#                       following form:
#                           proc ::gaugeImg::PwrdByTcl {} {
#                               return <Base64 encoded PNG or GIF image data>
#                           }
#
#                       You may change the name of the namespace/package
#                       by editing the marked lines below.
#
#                       To re-generate the file tclfiles/gaugeImgs.tcl call
#                       this script as follows:
#                       > tclsh img2pkg.tcl imgs > tclfiles/gaugeImgs.tcl
#
#******************************************************************************

if { $argc < 1 } {
    puts stderr "No input files given for img2pkg.tcl"
    exit 1
}

# Parse arguments and generate a list of all files to be converted.
set fileList {}
foreach val $argv {
    if { [file isdirectory $val] } {
        foreach f [lsort [glob -nocomplain -directory $val *.png]] {
            lappend fileList $f
        }
        foreach f [lsort [glob -nocomplain -directory $val *.gif]] {
            lappend fileList $f
        }
    } elseif { [file isfile $val] } {
        lappend fileList $val
    }
}


set packageName   "tcl3dgauges"
set namespaceName "::gaugeImg"

fconfigure stdout -translation binary

# Write namespace header.
puts "namespace eval $namespaceName {"
foreach fileName $fileList {
    set imgName  [file rootname [file tail $fileName]]
    if { [file readable $fileName] } {
        puts "    namespace export $imgName"
    }
}
puts "}"
puts ""

# Generate namespace procedures.
foreach fileName $fileList {
    set imgName  [file rootname [file tail $fileName]]
    set procName [format "%s::%s" $namespaceName $imgName]

    if { [file readable $fileName] } {
        # puts stderr "Generating proc $procName (from $fileName)"
        set fp [open $fileName r]
        fconfigure $fp -translation binary
        set imgData [read $fp]
        close $fp
        set uuImg [binary encode base64 $imgData]
        puts "proc $procName \{\} \{"
        puts "return \{"
        puts -nonewline $uuImg
        puts "\}"
        puts "\} \; \# End of proc $procName"
        puts ""
    } else {
        puts stderr "File $fileName does not exist."
        exit 1
    }
}

puts ""
exit 0
