/******************************************************************************
 *{@C
 *      Copyright:      2011-2025 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOgl
 *      Filename:       tcl3dGeoMath.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    C functions for handling geographic coordinate
 *                      conversions.
 *
 *****************************************************************************/

#ifndef __TCL3D_GEOMATH_H__
#define __TCL3D_GEOMATH_H__

#ifdef __cplusplus
   extern "C" {
#endif

/* Reference Ellipsoids, data taken from Wikipedia and DoD, 
 * WGS84, DMA TR 8350.2-B,1 Sept. 1991
 */

typedef enum 
{
    Airy,
    Airy_Modified,
    Australian_National,
    Bessel_1841,
    Bessel_1841_Namibia,
    Clarke_1866,
    Clarke_1880,
    Everest_Sabah_Sarawak,
    Everest_1830,
    Everest_1948,
    Everest_1956,
    Everest_1969,
    Fischer_1960,
    Fischer_1960_Modified,
    Fischer_1968,
    GRS_1980,
    Helmert_1906,
    Hough,
    International_1924,
    Karsovsky_1940,
    SGS_1985,
    South_American_1969,
    Sphere_6371km,
    WGS_1960,
    WGS_1966,
    WGS_1972,
    WGS_1984
} RefEllipsoid;

#ifndef SWIG

/* These are the definitions as seen by the C compiler. */

void tcl3dGeodetic2Geocentric (double lat, double lon, double alt,
                               RefEllipsoid refEll,
                               double *x, double *y, double *z);
void tcl3dGeocentric2Geodetic (double x, double y, double z, 
                               RefEllipsoid refEll,
                               double *lat, double *lon, double *alt);

void tcl3dLocal2GeoOrient (double head, double pitch, double roll,
                           double lat, double lon, 
                           double *psi, double *theta, double *phi);
void tcl3dGeo2LocalOrient (double psi, double theta, double phi,
                           double lat, double lon,
                           double *head, double *pitch, double *roll);

#else

/* These are the definitions as seen by SWIG. Here the 3 output parameters
   are returned as a Tcl list with the help of the OUTPUT keyword. */

void tcl3dGeodetic2Geocentric (double lat, double lon, double alt,
                               RefEllipsoid refEll,
                               double *OUTPUT, double *OUTPUT, double *OUTPUT);
void tcl3dGeocentric2Geodetic (double x, double y, double z,
                               RefEllipsoid refEll,
                               double *OUTPUT, double *OUTPUT, double *OUTPUT);

void tcl3dLocal2GeoOrient (double head, double pitch, double roll,
                           double lat, double lon,
                           double *OUTPUT, double *OUTPUT, double *OUTPUT);
void tcl3dGeo2LocalOrient (double psi, double theta, double phi,
                           double lat, double lon,
                           double *OUTPUT, double *OUTPUT, double *OUTPUT);
#endif

#ifdef __cplusplus
   }
#endif

#endif /* __TCL3D_GEOMATH_H__ */
