/******************************************************************************
 *{@C
 *      Copyright:      2009-2025 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOsg
 *      Filename:       tcl3dOsgNodeVisitor.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping the Open Scene Graph library.
 *                      This file is responsible to supply Tcl callback
 *                      functions to be used as OSG node visitors.
 *
 *                      The macro USE_OSG_IF must be defined at compilation
 *                      time to include the functionality of the OSG library
 *                      into the Tcl3D package.
 *
 *****************************************************************************/

%include tclinterp.i

%{
    #include <osg/NodeVisitor>
    #include <osg/Node>

    #include <iostream>
    #include <vector>

    class tcl3dOsgNodeVisitor : public osg::NodeVisitor {
    public: 

        tcl3dOsgNodeVisitor (Tcl_Interp *interp);

        tcl3dOsgNodeVisitor (Tcl_Interp *interp, TraversalMode tm);

        tcl3dOsgNodeVisitor (Tcl_Interp *interp, VisitorType type, TraversalMode tm);

        tcl3dOsgNodeVisitor (Tcl_Interp *interp, const std::string &tclProc,
                             const std::string &tclArgs="");

        virtual void apply (osg::Node &node);

        void setVisitorProc (const std::string &tclProc,
                             const std::string &tclArgs="");

        const std::string& getVisitorProc ();

        const std::string& getVisitorArgs ();

    private: 
        std::string _tclProc;
        std::string _tclArgs;
        Tcl_Interp  *_interp;
    };
%}

class tcl3dOsgNodeVisitor : public osg::NodeVisitor {
public: 

    tcl3dOsgNodeVisitor (Tcl_Interp *interp);

    tcl3dOsgNodeVisitor (Tcl_Interp *interp, TraversalMode tm);

    tcl3dOsgNodeVisitor (Tcl_Interp *interp, VisitorType type, TraversalMode tm);

    tcl3dOsgNodeVisitor (Tcl_Interp *interp, const std::string &tclProc,
                         const std::string &tclArgs="");

    virtual void apply (osg::Node &node);

    void setVisitorProc (const std::string &tclProc, const std::string &tclArgs = "");

    const std::string& getVisitorProc ();

    const std::string& getVisitorArgs ();

private: 
    std::string _tclProc;
    std::string _tclArgs;
    Tcl_Interp  *_interp;
};

%{
tcl3dOsgNodeVisitor::tcl3dOsgNodeVisitor (Tcl_Interp *interp) :
    osg::NodeVisitor (TRAVERSE_ALL_CHILDREN), 
    _tclProc ("unknown"),
    _tclArgs (""),
    _interp (interp) 
{
#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor ()\n"); fflush (stdout);
#endif
}

tcl3dOsgNodeVisitor::tcl3dOsgNodeVisitor (Tcl_Interp *interp, VisitorType type, TraversalMode tm) : 
    osg::NodeVisitor (type, tm),
    _tclProc ("unknown"),
    _tclArgs (""),
    _interp (interp) 
{
#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor (type=%d, tm=%d)\n", type, tm); fflush (stdout);
#endif
}

tcl3dOsgNodeVisitor::tcl3dOsgNodeVisitor (Tcl_Interp *interp, TraversalMode tm) : 
    osg::NodeVisitor (tm), 
    _tclProc ("unknown"),
    _tclArgs (""),
    _interp (interp) 
{
#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor (tm=%d)\n", tm); fflush (stdout);
#endif
}

tcl3dOsgNodeVisitor::tcl3dOsgNodeVisitor (Tcl_Interp *interp, 
        const std::string& tclProc, const std::string& tclArgs) : 
    osg::NodeVisitor (TRAVERSE_ALL_CHILDREN),
    _tclProc (tclProc),
    _tclArgs (tclArgs),
    _interp (interp)
{
#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor (tclProc=%s tclArgs=%s)\n", 
            tclProc.c_str(), tclArgs.c_str()); fflush (stdout);
#endif
}

void tcl3dOsgNodeVisitor::apply (osg::Node &node)
{
    char cmd[256];
    char buf[30];
    char *b = buf;
    osg::Node *nodePtr = &node;

    memset (buf, 0, sizeof (buf));

#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor::apply (node=%s)\n", node.getName().c_str());
    fflush (stdout);
#endif

    /* OPA TODO
     * This way of creating a SWIG compatible pointer is a hack and should
     * be improved.
     */
    b = SWIG_PackData (b, &nodePtr, sizeof (void *));
    sprintf (cmd, "%.*s _%s_p_%s__%s \"%s\"",
             200, _tclProc.c_str(), buf, node.libraryName(), node.className(),
             _tclArgs.c_str());
    if (Tcl_Eval (_interp, cmd) != TCL_OK) {
        Tcl_BackgroundError (_interp);
    }
    traverse(node);
}

void tcl3dOsgNodeVisitor::setVisitorProc (const std::string &tclProc, 
                                          const std::string &tclArgs)
{
#ifdef DEBUG_NODE_VISITOR
    printf ("tcl3dOsgNodeVisitor::setVisitorProc (tclProc=%s tclArgs=%s)\n",
            tclProc.c_str(), tclArgs.c_str()); fflush (stdout);
#endif
    _tclProc = tclProc;
    _tclArgs = tclArgs;
}

const std::string& tcl3dOsgNodeVisitor::getVisitorProc ()
{
    return _tclProc;
}

const std::string& tcl3dOsgNodeVisitor::getVisitorArgs ()
{
    return _tclArgs;
}
%}
