/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImageOpts.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Miscellaneous utility functions to retreive 
 *                      values from an option string.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      FF_ImgGetOptBool
 *                      FF_ImgGetOptVerbose
 *                      FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression
 *                      FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth
 *                      FF_ImgGetOptHeight
 *                      FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder
 *                      FF_ImgGetOptPixelType
 *                      FF_ImgGetOptSkipBytes
 *
 **************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Misc.h"

#include "FF_Image.h"
#include "FF_ImagePrivate.h"

static const char * GetOptValue (const char *opts, const char *key)
{
    static char result[FF_ImgMaxNameLen];
    char *optPos = NULL;
    int  resIndex = 0;

    result[0] = '\0';

    if (!opts) {
        return result;
    }
    if (!key) {
        return result;
    }

    /* Search key in opts. */
    optPos = strstr (opts, key);
    if (!optPos) {
        return result;
    }

    /* Key was found. Search for space to find the start of the value. */
    while (*optPos != ' ') {
        optPos++;
    }
    optPos++;

    /* Start of value was found.
       Copy all characters until next space or end of option string.
    */
    while (*optPos != ' ' && *optPos != '\0') {
        result[resIndex] = *optPos;
        resIndex++;
        optPos++;
    }
    result[resIndex] = '\0';
    return result;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptBool
 *
 *      Usage:          Get boolean flag value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptBool
 *                              (const char *opts,
 *                               const char *key,
 *                               UT_Bool *value)
 *
 *      Description:    Get the value of key "key" in option string "opts".
 *                      Valid string values for the boolean value are:
 *                      "true"  "on"  "1"  - Use boolean flag
 *                      "false" "off" "0"  - Do not use boolean flag
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "value" and UT_True returned.
 *                      Otherwise "value" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptWithAlpha FF_ImgGetOptCompression
 *                      FF_ImgGetOptScanOrder FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptBool (const char *opts, const char *key, UT_Bool *value)
{
    const char * result;
    
    result = GetOptValue (opts, key);
    if (UT_StringEqual (result, "true") || UT_StringEqual (result, "on") || UT_StringEqual (result, "1")) {
        *value = UT_True;
        return UT_True;
    } else if (UT_StringEqual (result, "false") || UT_StringEqual (result, "off") || UT_StringEqual (result, "0")) {
        *value = UT_False;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptVerbose
 *
 *      Usage:          Get verbose flag value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptVerbose
 *                              (const char *opts,
 *                               UT_Bool *verbose)
 *
 *      Description:    Get the value of key "-verbose" in option string "opts".
 *                      Valid string values for the boolean value are:
 *                      "true" "on" "1"    - Use verbose flag
 *                      "false" "off" "0"  - Do not use verbose flag
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "verbose" and UT_True returned.
 *                      Otherwise "verbose" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptWithAlpha FF_ImgGetOptCompression
 *                      FF_ImgGetOptScanOrder FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptVerbose (const char *opts, UT_Bool *verbose)
{
    const char * result;
    
    result = GetOptValue (opts, "-verbose");
    if (UT_StringEqual (result, "true") || UT_StringEqual (result, "on") || UT_StringEqual (result, "1")) {
        *verbose = UT_True;
        return UT_True;
    } else if (UT_StringEqual (result, "false") || UT_StringEqual (result, "off") || UT_StringEqual (result, "0")) {
        *verbose = UT_False;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptScanOrder
 *
 *      Usage:          Get scan order value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptScanOrder
 *                              (const char *opts,
 *                               FF_ImgScanOrderType *scanOrder)
 *
 *      Description:    Get the value of key "-scanorder" in option string "opts".
 *                      Valid string values for the scan order are:
 *                      "TopDown"
 *                      "BottomUp"
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "scanOrder" and UT_True returned.
 *                      Otherwise "scanOrder" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptScanOrder (const char *opts, FF_ImgScanOrderType *scanOrder)
{
    const char * result;
    
    result = GetOptValue (opts, "-scanorder");
    if (UT_StringEqual (result, "TopDown")) {
        *scanOrder = FF_ImgScanOrderTypeTopDown;
        return UT_True;
    } else if (UT_StringEqual (result, "BottomUp")) {
        *scanOrder = FF_ImgScanOrderTypeBottomUp;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptByteOrder
 *
 *      Usage:          Get byte order value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptByteOrder
 *                              (const char *opts,
 *                               FF_ImgByteOrderType *byteOrder)
 *
 *      Description:    Get the value of key "-byteorder" in option string "opts".
 *                      Valid string values for the byte order are:
 *                      "Intel"         - Little-endian
 *                      "Motorola"      - Big-endian
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "byteOrder" and UT_True returned.
 *                      Otherwise "byteOrder" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptWidth
 *                      FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptByteOrder (const char *opts, FF_ImgByteOrderType *byteOrder)
{
    const char * result;
    
    result = GetOptValue (opts, "-byteorder");
    if (UT_StringEqual (result, "Intel")) {
        *byteOrder = FF_ImgByteOrderTypeIntel;
        return UT_True;
    } else if (UT_StringEqual (result, "Motorola")) {
        *byteOrder = FF_ImgByteOrderTypeMotorola;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptCompression
 *
 *      Usage:          Get compression type value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptCompression
 *                              (const char *opts,
 *                               FF_ImgComprType *compression)
 *
 *      Description:    Get the value of key "-compression" in option string "opts".
 *                      Valid string values for the compression type are:
 *                      "None" - No compression
 *                      "Rle"  - Run length encoded compression
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "compression" and UT_True returned.
 *                      Otherwise "compression" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptScanOrder FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptCompression (const char *opts, FF_ImgComprType *compression)
{
    const char * result;
    
    result = GetOptValue (opts, "-compression");
    if (UT_StringEqual (result, "none")) {
        *compression = FF_ImgComprTypeNone;
        return UT_True;
    } else if (UT_StringEqual (result, "rle")) {
        *compression = FF_ImgComprTypeRle;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptWithAlpha
 *
 *      Usage:          Get alpha flag value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptWithAlpha
 *                              (const char *opts,
 *                               UT_Bool *withAlpha)
 *
 *      Description:    Get the value of key "-withalpha" in option string "opts".
 *                      Valid string values for the boolean value are:
 *                      "true" "on" "1"    - Use alpha value
 *                      "false" "off" "0"  - Do not use alpha value
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "withAlpha" and UT_True returned.
 *                      Otherwise "withAlpha" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptCompression
 *                      FF_ImgGetOptScanOrder FF_ImgGetOptUseHeader
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptWithAlpha (const char *opts, UT_Bool *withAlpha)
{
    const char * result;
    
    result = GetOptValue (opts, "-withalpha");
    if (UT_StringEqual (result, "true") || UT_StringEqual (result, "on") || UT_StringEqual (result, "1")) {
        *withAlpha = UT_True;
        return UT_True;
    } else if (UT_StringEqual (result, "false") || UT_StringEqual (result, "off") || UT_StringEqual (result, "0")) {
        *withAlpha = UT_False;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptUseHeader
 *
 *      Usage:          Get header flag value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptUseHeader
 *                              (const char *opts,
 *                               UT_Bool *useheader)
 *
 *      Description:    Get the value of key "-useheader" in option string "opts".
 *                      Valid string values for the boolean value are:
 *                      "true" "on" "1"    - Write RAW header information
 *                      "false" "off" "0"  - Do not write RAW header information
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "useHeader" and UT_True returned.
 *                      Otherwise "useHeader" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptWidth FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptUseHeader (const char *opts, UT_Bool *useHeader)
{
    const char * result;
    
    result = GetOptValue (opts, "-useheader");
    if (UT_StringEqual (result, "true") || UT_StringEqual (result, "on") || UT_StringEqual (result, "1")) {
        *useHeader = UT_True;
        return UT_True;
    } else if (UT_StringEqual (result, "false") || UT_StringEqual (result, "off") || UT_StringEqual (result, "0")) {
        *useHeader = UT_False;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptWidth
 *
 *      Usage:          Get width value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptWidth
 *                              (const char *opts,
 *                               Int32 *width)
 *
 *      Description:    Get the integer value of key "-width" in option string "opts".
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "width" and UT_True returned.
 *                      Otherwise "width" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptWidth (const char *opts, Int32 *width)
{
    const char * result;
    Int32 temp;
    
    result = GetOptValue (opts, "-width");
    if (1 == sscanf (result, "%d", &temp)) {
        *width = temp;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptHeight
 *
 *      Usage:          Get height value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptHeight
 *                              (const char *opts,
 *                               Int32 *height)
 *
 *      Description:    Get the integer value of key "-height" in option string "opts".
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "height" and UT_True returned.
 *                      Otherwise "height" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptWidth FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptHeight (const char *opts, Int32 *height)
{
    const char * result;
    Int32 temp;
    
    result = GetOptValue (opts, "-height");
    if (1 == sscanf (result, "%d", &temp)) {
        *height = temp;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptNumChans
 *
 *      Usage:          Get number of channels from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptNumChans
 *                              (const char *opts,
 *                               Int32 *numChans)
 *
 *      Description:    Get the integer value of key "-numchan" in option string "opts".
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "numChans" and UT_True returned.
 *                      Otherwise "numChans" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptWidth FF_ImgGetOptHeight
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptNumChans (const char *opts, Int32 *numChans)
{
    const char * result;
    Int32 temp;
    
    result = GetOptValue (opts, "-numchan");
    if (1 == sscanf (result, "%d", &temp)) {
        *numChans = temp;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptPixelType
 *
 *      Usage:          Get pixel type value from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptPixelType
 *                              (const char *opts,
 *                               FF_ImgRawPixelType *pixelType)
 *
 *      Description:    Get the value of key "-pixeltype" in option string "opts".
 *                      Valid string values for the pixel type are:
 *                      double - 64-bit floating point numbers
 *                      float  - 32-bit floating point numbers
 *                      int    - Unsigned 32-bit integers
 *                      short  - Unsigned 16-bit integers
 *                      byte   - Unsigned 8-bit integers
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "pixelType" and UT_True returned.
 *                      Otherwise "pixelType" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptWidth
 *                      FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptSkipBytes
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptPixelType (const char *opts, FF_ImgRawPixelType *pixelType)
{
    const char * result;
    
    result = GetOptValue (opts, "-pixeltype");
    if (UT_StringEqual (result, FF_ImgRawPixelTypeStrDouble)) {
        *pixelType = FF_ImgRawPixelTypeDouble;
        return UT_True;
    } else if (UT_StringEqual (result, FF_ImgRawPixelTypeStrFloat)) {
        *pixelType = FF_ImgRawPixelTypeFloat;
        return UT_True;
    } else if (UT_StringEqual (result, FF_ImgRawPixelTypeStrInt)) {
        *pixelType = FF_ImgRawPixelTypeInt;
        return UT_True;
    } else if (UT_StringEqual (result, FF_ImgRawPixelTypeStrShort)) {
        *pixelType = FF_ImgRawPixelTypeShort;
        return UT_True;
    } else if (UT_StringEqual (result, FF_ImgRawPixelTypeStrByte)) {
        *pixelType = FF_ImgRawPixelTypeByte;
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgGetOptSkipBytes
 *
 *      Usage:          Get number of bytes to skip from the option string.
 *
 *      Synopsis:       UT_Bool FF_ImgGetOptSkipBytes
 *                              (const char *opts,
 *                               Int32 *skipBytes)
 *
 *      Description:    Get the integer value of key "-skipbytes" in option string "opts".
 *
 *      Return value:   If a value for the key is contained in the option string,
 *                      the value is set in "skipBytes" and UT_True returned.
 *                      Otherwise "skipBytes" is not touched and UT_False returned.
 *
 *      See also:       FF_ImgGetOptVerbose FF_ImgGetOptWithAlpha
 *                      FF_ImgGetOptCompression FF_ImgGetOptScanOrder
 *                      FF_ImgGetOptUseHeader FF_ImgGetOptWidth
 *                      FF_ImgGetOptHeight FF_ImgGetOptNumChans
 *                      FF_ImgGetOptByteOrder FF_ImgGetOptPixelType
 *
 ***************************************************************************/

UT_Bool FF_ImgGetOptSkipBytes (const char *opts, Int32 *skipBytes)
{
    const char * result;
    Int32 temp;
    
    result = GetOptValue (opts, "-skipbytes");
    if (1 == sscanf (result, "%d", &temp)) {
        *skipBytes = temp;
        return UT_True;
    }
    return UT_False;
}
