/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImagePoSoft.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions for reading and writing image files
 *                      in poSoft's file format.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      FF_ImgOpenReadPoSoft
 *                      FF_ImgOpenWritePoSoft
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Portable.h"
#include "UT_FileHeader.h"
#include "UT_FileIO.h"
#include "UT_Memory.h"

#include "FF_Image.h"
#include "FF_ImagePrivate.h"
#include "FF_ImagePoSoft.h"

#define MINRUN   3   /* Minimum length of a run-length encoded sequence of bytes */
#define MAXRUN 127   /* Maximum length of a run-length encoded sequence of bytes */

/* Format dependent pixel file descriptor */

typedef struct {
    FILE  *fp;          /* C standard I/O file descriptor */
    Int32 offoff;       /* Position of the first byte of the
                           scan line offset table in the file */
    Int32 *offsets;     /* Positions of the first bytes of the
                           scan lines in the file */
    UInt8 *tmpbuf;      /* Intermediate buffer for various data conversions */
    UT_Bool withAlpha;  /* Use alpha channel for reading or writing */
    FF_ImgComprType compression[FF_NumImgChanTypes];
} POIFILE;

static void printImgInfo (FF_ImgFileDesc * px_f, FF_ImgComprType compression,
                          const char *fileName, const char *msg)
{
    Int32 chan, numChans = 0;

    printf ("%s %s\n", msg, fileName);
    printf ("\tSize in pixel     : %d x %d\n", px_f->desc.width, px_f->desc.height);
    printf ("\tCompression       : %s\n", FF_ImgGetCompressionTypeName (compression));
    for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
        if (px_f->channel[chan] != FF_ImgFmtTypeNone) {
            numChans++;
        }
    }
    printf ("\tNumber of channels: %d\n", numChans);
    for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
        if (px_f->channel[chan] != FF_ImgFmtTypeNone) {
            printf ("\tChannel %-12s: %s\n",
                    FF_ImgGetChannelTypeName (chan),
                    FF_ImgGetFormatTypeName (px_f->channel[chan]));
        }
    }
    fflush (stdout);
}

/* Reset all pointers in an "POIFILE" to NULL. */

static UT_Bool clearImgFile (POIFILE * poif)
{
    poif->offsets = NULL;
    poif->tmpbuf  = NULL;
    return UT_True;
}

/* Allocate and initialize "POIFILE". No memory is allocated for the scan
line offset table and for the data conversion and compression buffers. */

static UT_Bool allocImgFile (FF_ImgFileDesc * px_f)
{
    POIFILE *poif;

    if (!(poif = UT_MemPerm (POIFILE))) {
        return UT_False;
    }
    clearImgFile (poif);
    ((pixfile *) px_f->pixprivate)->fmtprivate = poif;
    return UT_True;
}

/* Allocate memory for an POIFILE's scan line offset table and for the
   data conversion and compression buffers:
   The number of entries in the scan line offset table, "offsets", is obviously
   equal to the number of scan lines in the file.
   The data conversion buffer, "tmpbuf", is allocated big enough to hold either
   the scan line offset table in machine-independent format, one scan line of
   floating point data in machine-independent format, or "QmN" scan lines of
   UInt8 data, whatever is largest. */

static UT_Bool allocTables (FF_ImgFileDesc * px_f)
{
    POIFILE *poif;
    UT_MemState memstate;
    Int32 offsize, tmpsize;

    memstate = UT_MemRemember ();
    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    offsize = px_f->desc.height;
    tmpsize = 4 * UT_MAX (px_f->desc.width, px_f->desc.height);
    if (!(poif->offsets = UT_MemTempArray (offsize, Int32)) ||
        !(poif->tmpbuf =  UT_MemTempArray (tmpsize, UInt8))) {
        UT_MemRestore (memstate);
        clearImgFile (poif);
        return UT_False;
    }
    UT_MemSave (memstate);
    return UT_True;
}

/* Free all memory used for an "POIFILE" and for the data
   buffers linked to it. */

static void freeImgFile (POIFILE * poif)
{
    if (poif->tmpbuf) {
        UT_MemFree (poif->tmpbuf);
    }
    if (poif->offsets) {
        UT_MemFree (poif->offsets);
    }
    UT_MemFree (poif);
    return;
}

/***************************************************************************
 *[
 *      Function Name:  rleCompress
 *
 *      Description:    Compress an array of bytes, using run-length
 *                      encoding.
 *
 *      Usage:          Int32 rleCompress
 *                              (Int32 inLen, const UInt8 in[], UInt8 out[]);
 *
 *                      "In" must be an array of "inLen" data bytes. The
 *                      data are compressed, and the result is stored in
 *                      array "out". The compression scheme is very basic. It
 *                      looks for repeats of a value that can be encoded into
 *                      a repeat count and a repeat value. The encoded data
 *                      consist of uncompressed and compressed data runs. Data
 *                      are not compressed if the repeat count is two or less.
 *                      Both, uncompressed and compressed runs begin with a
 *                      repeat count. A negative count indicates an
 *                      uncompressed run; the count byte is followed by -count
 *                      data bytes. A non-negative count indicates a compressed
 *                      run; the count byte is followed by a data byte which
 *                      is repeated (count + 1) times in the original
 *                      uncompressed data.
 *
 *      Return Value:   The length of the compressed data in bytes.
 *]
 ***************************************************************************/

static Int32 rleCompress (Int32 inLen, const UInt8 in[], UInt8 out[])
{
    UInt8 *outWrite;
    const UInt8 *inEnd, *runStart, *runEnd;

    inEnd    = in + inLen;
    runStart = in;
    runEnd   = in + 1;
    outWrite = out;
    while (runStart < inEnd) {
        while (runEnd < inEnd &&
               *runStart == *runEnd &&
               runEnd - runStart - 1 < MAXRUN) {
            ++runEnd;
        }
        if (runEnd - runStart >= MINRUN) {
            /* Found a run of compressable data */
            *outWrite++ = (runEnd - runStart) - 1;
            *outWrite++ = *(Int8 *) runStart;
            runStart = runEnd;
        } else {
            /* Found a run of uncompressable data */
            while (runEnd < inEnd &&
                   ((runEnd + 1 >= inEnd ||
                     *runEnd != *(runEnd + 1)) ||
                    (runEnd + 2 >= inEnd ||
                     *(runEnd + 1) != *(runEnd + 2))) &&
                   runEnd - runStart < MAXRUN) {
                ++runEnd;
            }
            *outWrite++ = (Int8)(runStart - runEnd);
            while (runStart < runEnd) {
                *outWrite++ = (UInt8)*runStart;
                ++runStart;
            }
        }
        ++runEnd;
    }
    return outWrite - out;
}

/***************************************************************************
 *[
 *      Function Name:  rleUncompress
 *
 *      Description:    Uncompress run-length encoded data.
 *
 *      Usage:          UT_Bool rleUncompress
 *                              (Int32 inLen, Int32 outLen,
 *                               const UInt8 in[], UInt8 out[])
 *
 *                      "In" must be an array of "inLen" run-length encoded
 *                      data bytes. The data are decompressed, and the result
 *                      is stored in array "out".  This routine is the inverse
 *                      of function rleCompress.
 *
 *      Return Value:   UT_True if the length of the uncompressed data is
 *                      exactly "outLen" bytes, UT_False otherwise.
 *]
 ***************************************************************************/

static UT_Bool rleUncompress
        (Int32 inLen, Int32 outLen, const UInt8 in[], UInt8 out[])
{
    int count;
    Int8 *inPtr = (Int8 *)in;

    while (inLen > 0) {
        if (*inPtr < 0) {
            count = -((int)*inPtr++);
            inLen -= count + 1;
            if (0 > (outLen -= count)) {
                UT_ErrSetNum (FF_ImgErrSyntax, str_long_scan);
                return UT_False;
            }
            while (count-- > 0) {
                *out++ = *(UInt8 *) (inPtr++);
            }
        } else {
            count = *inPtr++;
            inLen -= 2;
            if (0 > (outLen -= count + 1)) {
                UT_ErrSetNum (FF_ImgErrSyntax, str_long_scan);
                return UT_False;
            }
            while (count-- >= 0) {
                *out++ = *(UInt8 *) inPtr;
            }
            inPtr++;
        }
    }
    if (outLen) {
        UT_ErrSetNum (FF_ImgErrSyntax, str_short_scan);
        return UT_False;
    }
    return UT_True;
}

/* Convert an integer into a machine-independent format (an array of 4 bytes)
   and write the array to a file. */

static UT_Bool writeInt32 (FILE * fp, Int32 i)
{
    UInt8 buf[4];

    UT_PortInt32ToByte (i, buf);
    if (!fwrite (buf, sizeof (buf), 1, fp)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
        return UT_False;
    }
    return UT_True;
}

/* Convert a floating point number into a machine-independent format (an array
   of 4 bytes) and write the array to a file. */

static UT_Bool writeFloat32 (FILE * fp, Float32 f)
{
    UInt8 buf[4];

    UT_PortFloat32ToByte (f, buf);
    if (!fwrite (buf, sizeof (buf), 1, fp)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
        return UT_False;
    }
    return UT_True;
}

/* Write the file format version to a file. */

static UT_Bool writeVersion (FILE * fp, char buf[])
{
    if (!fwrite (buf, FF_ImgMaxNameLen, 1, fp)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
        return UT_False;
    }
    return UT_True;
}

/* All channels of scan line number y for pixel file "px_f" are stored in the file.
   Note that the poSoft file format allows the scan lines to be written in any order.
   The position of the first byte of the scan line data in the file is entered
   into the scan line offset table (the offset table will be copied into the
   file after all scan lines have been writen.
   The scan line data written start with the scan line number, which is
   redundant, but helps in checking a pixel file for consistency.
   It can also be useful when an attempt is made to reconstruct the scan line
   offset table of an incomplete pixel file. The scan line number is followed
   by the pixel data for the channels in the scan line.
   Since the pixel data may be compressed, the block of pixel data for every
   channel is preceded by its length in bytes. */

static UT_Bool poiWriteScan (FF_ImgFileDesc * px_f, Int32 y)
{
    pixfile *pixf;
    POIFILE *poif;
    Int32 chn, len = 0;
    UInt8 *buf = NULL;
    Float32 *fbuf = NULL;

    pixf = px_f->pixprivate;
    poif = pixf->fmtprivate;

    /* Check if the parameters supplied are reasonable. */
    if (poif->offsets[y]) {
        UT_ErrFatal ("poiWriteScan", "scan line already exists in the file");
    }

    /* Enter the new scan line into the scan line offset table. */

    if (0 > (poif->offsets[y] = ftell (poif->fp))) {
        UT_ErrSetNum (UT_ErrFromOs (), "ftell");
        return UT_False;
    }

    /* Write the scan line data. */

    if (!writeInt32 (poif->fp, y)) {
        return UT_False;
    }
    for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
        if ((chn == FF_ImgChanTypeMatte) && !poif->withAlpha) {
            continue;
        }
        if (px_f->channel[chn]) {
            switch (px_f->channel[chn]) {
                case FF_ImgFmtTypeUByte: {
                    switch (poif->compression[chn]) {
                        case FF_ImgComprTypeNone: {
                            buf = pixf->buf[chn].ubytes;
                            len = px_f->desc.width;
                            break;
                        }
                        case FF_ImgComprTypeRle: {
                            buf = poif->tmpbuf;
                            len = rleCompress
                                (px_f->desc.width,
                                 pixf->buf[chn].ubytes, buf);
                            if (len >= px_f->desc.width) {
                                buf = pixf->buf[chn].ubytes;
                                len = px_f->desc.width;
                            }
                            break;
                        }
                        default: {
                            UT_ErrFatal ("poiWriteScan", "unknown compression scheme");
                        }
                    }
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    fbuf = pixf->buf[chn].floats;
                    buf = poif->tmpbuf;
                    len = px_f->desc.width;
                    while (len--) {
                        UT_PortFloat32ToByte (*fbuf++, buf);
                        buf += 4;
                    }
                    buf = poif->tmpbuf;
                    len = 4 * px_f->desc.width;
                    break;
                }
                default: {
                    UT_ErrFatal ("poiWriteScan", "unknown pixel format");
                }
            }
            if (len && !writeInt32 (poif->fp, len)) {
                return UT_False;
            }
            if (len && !fwrite (buf, 1, len, poif->fp)) {
                UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
                return UT_False;
            }
        }
    }
    return UT_True;
}

/* Convert the scan line offset table of a pixel file into a machine-
   independent format and write it to the file. */

static UT_Bool writeOffsets (FF_ImgFileDesc * px_f)
{
    Int32   i;
    POIFILE *poif;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    for (i = 0; i < px_f->desc.height; ++i) {
        UT_PortInt32ToByte (poif->offsets[i], poif->tmpbuf + (4 * i));
    }
    if (0 > fseek (poif->fp, poif->offoff, SEEK_SET)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fseek");
        return UT_False;
    }
    if (!fwrite (poif->tmpbuf, 4, px_f->desc.height, poif->fp)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
        return UT_False;
    }
    return UT_True;
}

/* Write the beginning of an image file: First write the poSoft file header
   which identifies the file as an image file.
   Then write the image file specific header containing the size, aspect ratio
   etc. of the image stored in the file.
   Finally write a dummy scan line offset table which will later be replaced
   with the true data. */

static UT_Bool writeHeader (FF_ImgFileDesc * px_f)
{
    Int32   i, chn;
    Int32   chanFmt, comprType;
    POIFILE *poif;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    if (!UT_FileHeaderWrite (poif->fp, UT_FileTypeImage, UT_FileSubTypePoImage,
                             UT_FileCodingTypePortableBinary) ||
        !writeInt32   (poif->fp, px_f->desc.width) ||
        !writeInt32   (poif->fp, px_f->desc.height) ||
        !writeFloat32 (poif->fp, px_f->desc.aspect) ||
        !writeFloat32 (poif->fp, px_f->desc.red.x) ||
        !writeFloat32 (poif->fp, px_f->desc.red.y) ||
        !writeFloat32 (poif->fp, px_f->desc.green.x) ||
        !writeFloat32 (poif->fp, px_f->desc.green.y) ||
        !writeFloat32 (poif->fp, px_f->desc.blue.x) ||
        !writeFloat32 (poif->fp, px_f->desc.blue.y) ||
        !writeFloat32 (poif->fp, px_f->desc.white.x) ||
        !writeFloat32 (poif->fp, px_f->desc.white.y) ||
        !writeFloat32 (poif->fp, px_f->desc.gamma) ||
        !writeInt32   (poif->fp, FF_NumImgChanTypes) ||
        !writeVersion (poif->fp, px_f->version)) {
        return UT_False;
    }
    for (chn = 0; chn < (Int32) FF_NumImgChanTypes; ++chn) {
        chanFmt   = (Int32) px_f->channel[chn];
        comprType = (Int32) poif->compression[chn];
        if ((chn == FF_ImgChanTypeMatte) && !poif->withAlpha) {
            chanFmt   = (Int32) FF_ImgFmtTypeNone;
            comprType = (Int32) FF_ImgComprTypeNone;
        }
        if (!writeInt32 (poif->fp, chanFmt) ||
            !writeInt32 (poif->fp, comprType)) {
            return UT_False;
        }
    }
    for (i = 0; i < px_f->desc.height; ++i) {
        poif->offsets[i] = 0;
    }
    if (0 > (poif->offoff = ftell (poif->fp))) {
        UT_ErrSetNum (UT_ErrFromOs (), "ftell");
        return UT_False;
    }
    if (!writeOffsets (px_f)) {
        return UT_False;
    }
    return UT_True;
}

/* Close a pixel file previously opened with FF_ImgOpenWritePoSoft: 
   Write the scan line lookup table. Close the standard I/O library
   file associated with the pixel file. Then deallocate the memory 
   used to hold information about the file. */

static void poiCloseWrite (FF_ImgFileDesc * px_f)
{
    POIFILE *poif;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    writeOffsets (px_f);
    UT_FileClose (poif->fp);
    freeImgFile (poif);
    return;
}

/***************************************************************************
 *
 *      Functions for reading pixel files
 *
 ***************************************************************************/

/* Read 4 bytes, representing an integer number in a machine-independent
   format, from a file and convert them into the current machine's format. */

static UT_Bool readInt32 (FILE * fp, Int32 * i)
{
    UInt8 buf[4];

    if (!fread (buf, sizeof (buf), 1, fp)) {
        if (feof (fp)) {
            UT_ErrSetNum (UT_ErrUnexpEof, "fread");
        } else {
            UT_ErrSetNum (UT_ErrFromOs (), "fread");
        }
        return UT_False;
    }
    *i = UT_PortByteToInt32 (buf);
    return UT_True;
}

/* Read 4 bytes, representing a floating point number in a machine-independent
   format, from a file and convert them into the current machine's format. */

static UT_Bool readFloat32 (FILE * fp, Float32 * f)
{
    UInt8 buf[4];

    if (!fread (buf, sizeof (buf), 1, fp)) {
        if (feof (fp)) {
            UT_ErrSetNum (UT_ErrUnexpEof, "fread");
        } else {
            UT_ErrSetNum (UT_ErrFromOs (), "fread");
        }
        return UT_False;
    }
    *f = UT_PortByteToFloat32 (buf);
    return UT_True;
}

/* Read the file format version from a file. */

static UT_Bool readVersion (FILE * fp, char buf[])
{
    if (!fread (buf, FF_ImgMaxNameLen, 1, fp)) {
        if (feof (fp)) {
            UT_ErrSetNum (UT_ErrUnexpEof, "fread");
        } else {
            UT_ErrSetNum (UT_ErrFromOs (), "fread");
        }
        return UT_False;
    }
    return UT_True;
}

/* All channels of scan line number y are read from pixel file "px_f".
   Note that the poSoft file format allows the scan lines to be read
   in any order. */

static UT_Bool poiReadScan (FF_ImgFileDesc * px_f, Int32 y)
{
    pixfile *pixf;
    POIFILE *poif;
    Int32 chn, len, f_y;
    UInt8 *buf = NULL;
    Float32 *fbuf = NULL;

    pixf = px_f->pixprivate;
    poif = pixf->fmtprivate;

    /* Find out where in the file the scan line is. */

    if (!poif->offsets[y]) {
        UT_ErrSetNum (FF_ImgErrSyntax, str_no_scan);
        return UT_False;
    }
    if (0 > fseek (poif->fp, poif->offsets[y], SEEK_SET)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fseek");
        return UT_False;
    }

    /* Read the scan line data and perform some consistency checks. */

    if (!readInt32 (poif->fp, &f_y)) {
        return UT_False;
    }
    if (y != f_y) {
        UT_ErrSetNum (FF_ImgErrSyntax, str_scan_number);
        return UT_False;
    }
    for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
        if (px_f->channel[chn]) {
            switch (px_f->channel[chn]) {
                case FF_ImgFmtTypeUByte: {
                    switch (poif->compression[chn]) {
                        case FF_ImgComprTypeNone: {
                            if (!readInt32 (poif->fp, &len)) {
                                return UT_False;
                            }
                            if (len != px_f->desc.width) {
                                UT_ErrSetNum (FF_ImgErrSyntax, str_scan_len);
                                return UT_False;
                            }
                            buf = pixf->buf[chn].ubytes;
                            break;
                        }
                        case FF_ImgComprTypeRle: {
                            if (!readInt32 (poif->fp, &len)) {
                                return UT_False;
                            }
                            if ((len <= 0) || (len > px_f->desc.width)) {
                                UT_ErrSetNum (FF_ImgErrSyntax, str_scan_len);
                                return UT_False;
                            }
                            if (len < px_f->desc.width) {
                                buf = poif->tmpbuf;
                            } else {
                                buf = pixf->buf[chn].ubytes;
                            }
                            break;
                        }
                        default: {
                            UT_ErrFatal ("poiReadScan", "unknown compression type");
                        }
                    }
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    if (!readInt32 (poif->fp, &len)) {
                        return UT_False;
                    }
                    if (len != 4 * px_f->desc.width) {
                        UT_ErrSetNum (FF_ImgErrSyntax, str_scan_len);
                        return UT_False;
                    }
                    buf = poif->tmpbuf;
                    break;
                }
                default: {
                    UT_ErrFatal ("poiReadScan", "unknown pixel format");
                }
            }
            if (len && !fread (buf, 1, len, poif->fp)) {
                if (feof (poif->fp)) {
                    UT_ErrSetNum (UT_ErrUnexpEof, "fread");
                } else {
                    UT_ErrSetNum (UT_ErrFromOs (), "fread");
                }
                return UT_False;
            }
            switch (px_f->channel[chn]) {
                case FF_ImgFmtTypeUByte: {
                    switch (poif->compression[chn]) {
                        case FF_ImgComprTypeNone: {
                            break;
                        }
                        case FF_ImgComprTypeRle: {
                            if (len < px_f->desc.width &&
                                !rleUncompress (len, px_f->desc.width,
                                                buf, pixf->buf[chn].ubytes)) {
                                return UT_False;
                            }
                            break;
                        }
                        default: {
                            UT_ErrFatal ("poiReadScan", "unknown compression scheme");
                        }
                    }
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    fbuf = pixf->buf[chn].floats;
                    while (len) {
                        *fbuf++ = UT_PortByteToFloat32 (buf);
                        buf += 4;
                        len -= 4;
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("poiReadScan", "unknown pixel format");
                }
            }
        }
    }
    return UT_True;
}

/* Read the scan line offset table from a pixel file. */

static UT_Bool readOffsets (FF_ImgFileDesc * px_f)
{
    Int32   i;
    POIFILE *poif;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    if (!fread (poif->tmpbuf, 4, px_f->desc.height, poif->fp)) {
        if (feof (poif->fp)) {
            UT_ErrSetNum (UT_ErrUnexpEof, "fread");
        } else {
            UT_ErrSetNum (UT_ErrFromOs (), "fread");
        }
        return UT_False;
    }
    for (i = 0; i < px_f->desc.height; ++i) {
        poif->offsets[i] = UT_PortByteToInt32 (poif->tmpbuf + (4 * i));
    }
    return UT_True;
}

/* Read the beginning of a pixel file: First read the file header which tells
   us if the file is actually a pixel file or not. Then read the pixel file
   header containing the size, aspect ratio etc. of the image stored in the
   file.
   Allocate memory for the scan line offset table and for the data conversion
   buffer. Finally read the scan line offset table from the file. */

static UT_Bool readHeader (FF_ImgFileDesc * px_f, FF_ImgComprType *compression)
{
    Int32 chn, nchan, tmp;
    POIFILE *poif;
    UT_FileHeader fileHdr;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    if (!UT_FileHeaderRead (poif->fp, &fileHdr)) {
        return UT_False;
    }
    if (fileHdr.fileType != UT_FileTypeImage) {
        UT_ErrSetNum (FF_ImgErrNoImgFile, str_FileType, 
                     UT_GetFileTypeName (fileHdr.fileType));
        return UT_False;
    }
    if (fileHdr.fileSubType != UT_FileSubTypePoImage) {
        UT_ErrSetNum (FF_ImgErrNoImgFile, str_FileSubType, 
                     UT_GetFileSubTypeName (fileHdr.fileSubType));
        return UT_False;
    }
    if (!readInt32   (poif->fp, &px_f->desc.width) ||
        !readInt32   (poif->fp, &px_f->desc.height) ||
        !readFloat32 (poif->fp, &px_f->desc.aspect) ||
        !readFloat32 (poif->fp, &px_f->desc.red.x) ||
        !readFloat32 (poif->fp, &px_f->desc.red.y) ||
        !readFloat32 (poif->fp, &px_f->desc.green.x) ||
        !readFloat32 (poif->fp, &px_f->desc.green.y) ||
        !readFloat32 (poif->fp, &px_f->desc.blue.x) ||
        !readFloat32 (poif->fp, &px_f->desc.blue.y) ||
        !readFloat32 (poif->fp, &px_f->desc.white.x) ||
        !readFloat32 (poif->fp, &px_f->desc.white.y) ||
        !readFloat32 (poif->fp, &px_f->desc.gamma) ||
        !readInt32   (poif->fp, &nchan) ||
        !readVersion (poif->fp, px_f->version)) {
        return UT_False;
    }
    if (px_f->desc.width < 1 ||
        px_f->desc.height < 1 ||
        px_f->desc.aspect == 0.0 ||
        nchan > FF_NumImgChanTypes) {
        UT_ErrSetNum (FF_ImgErrSyntax, str_header);
        return UT_False;
    }
    for (chn = 0; chn < FF_NumImgChanTypes; chn++) {
        px_f->channel[chn] = FF_ImgFmtTypeNone;
    }
    for (chn = 0; chn < nchan; chn++) {
        if (!readInt32 (poif->fp, &tmp)) {
            return UT_False;
        }
        if (tmp < (Int32) FF_ImgFmtTypeNone || 
            tmp > (Int32) FF_NumImgFmtTypes) {
            UT_ErrSetNum (FF_ImgErrSyntax, str_pixformat);
            return UT_False;
        }
        px_f->channel[chn] = (FF_ImgFmtType) tmp;
        if ((chn == FF_ImgChanTypeMatte) && !poif->withAlpha) {
            px_f->channel[chn] = FF_ImgFmtTypeNone;
        }
        if (!readInt32 (poif->fp, &tmp)) {
            return UT_False;
        }
        if (!px_f->channel[chn]) {
            tmp = (Int32) FF_ImgComprTypeNone;
        }
        if (tmp < (Int32) FF_ImgComprTypeNone ||
            tmp > (Int32) FF_NumImgComprTypes) {
            UT_ErrSetNum (FF_ImgErrSyntax, str_compression);
            return UT_False;
        }
        poif->compression[chn] = (FF_ImgComprType) tmp;
        *compression = (FF_ImgComprType) tmp;
    }
    if (!allocTables (px_f) || !readOffsets (px_f)) {
        return UT_False;
    }
    return UT_True;
}

/* Close a pixel file previously opened with FF_ImgOpenReadPoSoft: Close the
   standard I/O library file associated with the pixel file. Then deallocate
   the memory used to hold information about the file. */

static void poiCloseRead (FF_ImgFileDesc * px_f)
{
    POIFILE *poif;

    poif = ((pixfile *) px_f->pixprivate)->fmtprivate;
    UT_FileClose (poif->fp);
    freeImgFile (poif);
    return;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenWritePoSoft
 *
 *      Usage:          Open a poSoft image file for writing.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenWritePoSoft
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f),
 *                               const char *opts)
 *
 *      Description:    Image file "name" is opened for writing.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenReadPoSoft
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenWritePoSoft (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    Int32 i;
    FILE *fp;
    pixfile *pixf;
    POIFILE *poif;
    FF_ImgComprType optCompression;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;

    if (!(fp = UT_FileOpen (name, "wb"))) {
        return UT_False;
    }

    if (!allocImgFile (px_f)) {
        UT_FileClose (fp);
        return UT_False;
    }

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default compression value and check options of -compression key. */
    optCompression = FF_ImgComprTypeNone;
    FF_ImgGetOptCompression (opts, &optCompression);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);

    pixf = px_f->pixprivate;
    pixf->readscan = NULL;
    pixf->writescan = poiWriteScan;
    pixf->close = poiCloseWrite;
    poif = pixf->fmtprivate;
    poif->fp = fp;
    poif->withAlpha = optWithAlpha;
    for (i=0; i<FF_NumImgChanTypes; i++) {
        poif->compression[i] = optCompression;
    }
    if (!allocTables (px_f) ||
        !writeHeader (px_f)) {
        UT_FileClose (poif->fp);
        freeImgFile (poif);
        return UT_False;
    }
    px_f->scanorder = FF_ImgScanOrderTypeTopDown;

    if (optVerbose) {
         printImgInfo (px_f, optCompression, name, "Saving image:");
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenReadPoSoft
 *
 *      Usage:          Open a poSoft image file for reading.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenReadPoSoft
 *                              (const char *name,
 *                              FF_ImgFileDesc *px_f,
 *                              const char *opts)
 *
 *      Description:    Image file "name" is opened for reading.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenWritePoSoft
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenReadPoSoft (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    FILE *fp;
    pixfile *pixf;
    POIFILE *poif;
    FF_ImgComprType compression = FF_ImgComprTypeNone;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;

    if (!(fp = UT_FileOpen (name, "rb"))) {
        return UT_False;
    }

    if (!allocImgFile (px_f)) {
        UT_FileClose (fp);
        return UT_False;
    }

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);

    pixf = px_f->pixprivate;
    pixf->readscan = poiReadScan;
    pixf->writescan = NULL;
    pixf->close = poiCloseRead;

    poif = pixf->fmtprivate;
    poif->fp = fp;
    poif->withAlpha = optWithAlpha;

    if (!readHeader (px_f, &compression)) {
        UT_FileClose (poif->fp);
        freeImgFile (poif);
        return UT_False;
    }
    px_f->scanorder = FF_ImgScanOrderTypeTopDown;

    if (optVerbose) {
         printImgInfo (px_f, compression, name, "Reading image:");
    }

    return UT_True;
}
