/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImagePrivate.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions for reading and writing image files.
 *
 *      Additional documentation:
 *                      None.
 *
 **************************************************************************/

#ifndef __FF_IMAGEPRIVATE_H__
#define __FF_IMAGEPRIVATE_H__

#include "UT_Compat.h"
#include "FF_Image.h"

/* Message strings */

#define str_open_read    "cannot open image file \"%s\" for reading"
#define str_open_write   "cannot open image file \"%s\" for writing"
#define str_wr_header    "cannot write header of image file \"%s\""
#define str_rd_header    "cannot read header of image file \"%s\""
#define str_long_scan    "scan line is too long"
#define str_short_scan   "scan line is too short"
#define str_scan_len     "scan line length is incorrect"
#define str_scan_number  "scan line number in the file is incorrect"
#define str_no_scan      "scan line is missing in the file"
#define str_compat_num   "compatibility number is %x"
#define str_FileType     "File type is \"%s\""
#define str_FileSubType  "File subtype is \"%s\""
#define str_header       "file header is inconsistent"
#define str_pixformat    "unknown image data format"
#define str_compression  "unknown data compression method"
#define str_magic_num    "Invalid magic number (0X%X) in file \"%s\""
#define str_no_memory    "Cannot allocate enough memory"
#define str_SizeOverflow "Overflow using width %d and height %d"
#define str_InvalidChan  "Invalid image channel \"%s\""

/* An element of a scaling lookup table */

typedef struct {
    Int32   src,            /* Index of source pixel */
            dest;           /* Index of destination pixel */
    Float32 weight;         /* Weighting factor */
} scaletab;

/* The private part of the image file header, not visible to the user */

typedef struct {
    void    *fmtprivate;    /* Private data maintained by the format-
                               dependent layer of the image file library */
    UT_Bool readfile;       /* True if the file has been opened for
                               reading, false if the file has been opened
                               for writing */
    UT_Bool (*readscan) (FF_ImgFileDesc *, Int32),
            (*writescan) (FF_ImgFileDesc *, Int32);
    void    (*close) (FF_ImgFileDesc *);
    FF_ImgDesc *image;      /* The image associated with the image file */
    UT_Bool scale,          /* Change the resolution of the image
                               while reading the file. */
            gamma,          /* Perform gamma correction on the image
                               while reading or writing the file. */
            color,          /* Perform color correction on the image
                               while reading or writing the file. */
            nocopy;         /* Indicates that copying pixel data into
                               an intermediate buffer may be eliminated. */
    scaletab *xtable;       /* Lookup table for scaling the image in
                               x direction while reading it */
    Int32   xtabsize;       /* Number of entries in xtable */
    Float32 my,             /* Factor used for sscaling the image in
                               y direction while reading it */
            pxarea;         /* Area of a single destination pixel, in
                               source pixels, used for scaliing, too. */
    Int16   *gammatable;    /* Lookup table for gamma correction */
    Float32 cm[3][3];       /* Matrix for color correction */
    union   {               /* Buffer for one scan line of pixel data */
            UInt8   *ubytes;
            Float32 *floats;
            } buf [FF_NumImgChanTypes],
              rbuf[FF_NumImgChanTypes];
    Float32 *sbuf[FF_NumImgChanTypes]; /* Buffer for scaling UInt8 pixel data */
    Int32   ybuf;                      /* Y coordinate of scan line in buf */
} pixfile;

/* Functions defined in FF_ImageUtil.c */

extern UT_Bool FF_ImgGenScaleTable     (FF_ImgFileDesc *, FF_ImgDesc *);
extern UT_Bool FF_ImgGenGammaTable     (FF_ImgFileDesc *, FF_ImgDesc *);
extern void    FF_ImgApplyGammaUByte   (FF_ImgFileDesc *, UInt8 *);
extern void    FF_ImgApplyGammaFloat   (FF_ImgFileDesc *, Float32 *);
extern UT_Bool FF_ImgGenColorMatrix    (FF_ImgFileDesc *, FF_ImgDesc *);
extern void    FF_ImgApplyColorCorrect (FF_ImgFileDesc *);

#endif /* __FF_IMAGEPRIVATE_H__ */
