/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImageRaw.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The file format dependent layer of the image file
 *                      format library: Functions to read and write
 *                      files in a simple raw image file format.
 *
 * The following image types are currently supported:
 *
 * Grayscale image:  1 channel  of 32-bit floating point values.
 * True-color image: 3 channels of 32-bit floating point values.
 *
 * There are 2 supported file formats:
 * One with the binary raw data only, the other with a 7 line ASCII
 * header of the following form:
 *
 *     Magic=RAW                File format identifier. Fixed value.
 *     Width=172                Image width in pixels.
 *     Height=181               Image height in pixels.
 *     NumChan=1                Possible values: 1.
 *     ByteOrder=Intel          Possible values: "Intel" or "Motorola".
 *     ScanOrder=TopDown        Possible values: "TopDown" or "BottomUp".
 *     PixelType=float          Possible values: "float", "short", "byte".
 *
 *     Each ASCII header line is exactly 20 bytes long (including the newline).
 *     If necessary, spaces are added between the value and the newline 
 *     character. Thus these files can easily be read by external tools like
 *     MatLab by skipping 140 bytes and then read in the pure data.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      FF_ImgOpenWriteRaw
 *                      FF_ImgOpenReadRaw
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Error.h"
#include "UT_Portable.h"
#include "UT_Memory.h"
#include "UT_FileIO.h"

#include "FF_Image.h"
#include "FF_ImagePrivate.h"
#include "FF_ImageRaw.h"

/* Format dependent pixel file descriptor */

typedef struct {
    FILE  *fp;                     /* File descriptor */
    long  fpPos;                   /* File position after reading header */
    Int32 numChans;                /* Number of channels in file */
    UT_Bool withAlpha;             /* Read/Write alpha channel */
    UT_Bool swapBytes;             /* Swap bytes during read or write */
    Int32 pixelSize;               /* Size of a pixel in bytes */
    FF_ImgRawPixelType pixelType;  /* Numeric value of pixel type */
    FF_ImgByteOrderType byteOrder; /* Byte ordering of pixel values */
    FF_ImgChanType *chanMap;       /* Channel mapping for writing */
    UInt8 *rowbuf;                 /* Scanline buffer for reading raw bytes */
} rawImgFile;

static FF_ImgChanType sDefMap1[] = { FF_ImgChanTypeBrightness, FF_ImgChanTypeNone,  FF_ImgChanTypeNone, FF_ImgChanTypeNone };
static FF_ImgChanType sDefMap2[] = { FF_ImgChanTypeBrightness, FF_ImgChanTypeNone,  FF_ImgChanTypeNone, FF_ImgChanTypeMatte };
static FF_ImgChanType sDefMap3[] = { FF_ImgChanTypeRed,        FF_ImgChanTypeGreen, FF_ImgChanTypeBlue, FF_ImgChanTypeNone };
static FF_ImgChanType sDefMap4[] = { FF_ImgChanTypeRed,        FF_ImgChanTypeGreen, FF_ImgChanTypeBlue, FF_ImgChanTypeMatte };

/* Header fields. */
#define strMagic     "Magic=%s"
#define strWidth     "Width=%d"
#define strHeight    "Height=%d"
#define strNumChan   "NumChan=%d"
#define strByteOrder "ByteOrder=%s"
#define strScanOrder "ScanOrder=%s"
#define strPixelType "PixelType=%s"

/* Header fields possible values. */
#define strIntel    "Intel"
#define strMotorola "Motorola"
#define strTopDown  "TopDown"
#define strBottomUp "BottomUp"

#define strUnknown  "Unknown"

#define HEADLEN     20
#define MAXSIZE    100

static void printImgInfo (Int32 width, Int32 height,
                          Int32 nChans, UT_Bool isIntel, 
                          UT_Bool isTopDown, const char *pixelType,
                          const char *fileName, const char *msg)
{
    printf ("%s %s\n", msg, fileName);
    printf ("\tSize in pixel     : %d x %d\n", width, height);
    printf ("\tNumber of channels: %d\n",      nChans);
    printf ("\tPixel type        : %s\n",      pixelType);
    printf ("\tVertical encoding : %s\n",      isTopDown? strTopDown: strBottomUp);
    printf ("\tHost byte order   : %s\n",      UT_PortIsIntel()? strIntel: strMotorola);
    printf ("\tFile byte order   : %s\n",      isIntel?          strIntel: strMotorola);
    fflush (stdout);
}

static UT_Bool checkMagic (const char *magic)
{
    if (strncmp (magic, "RAW", 3) != 0) {
        UT_ErrSetNum (UT_ErrParamInvalid,
            "Invalid value for header field Magic: \"%s\". Must be \"RAW\"", magic);
        return UT_False;
    }
    return UT_True;
}

static UT_Bool checkPixelType (const char *pixelType)
{
    if (!(strncmp (pixelType, FF_ImgRawPixelTypeStrDouble, strlen (FF_ImgRawPixelTypeStrDouble)) == 0) &&
        !(strncmp (pixelType, FF_ImgRawPixelTypeStrFloat,  strlen (FF_ImgRawPixelTypeStrFloat)) == 0) &&
         (strncmp (pixelType, FF_ImgRawPixelTypeStrInt,    strlen (FF_ImgRawPixelTypeStrInt)) == 0) &&
         (strncmp (pixelType, FF_ImgRawPixelTypeStrShort,  strlen (FF_ImgRawPixelTypeStrShort)) == 0) &&
         (strncmp (pixelType, FF_ImgRawPixelTypeStrByte,   strlen (FF_ImgRawPixelTypeStrByte))  == 0)) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Invalid value for header field PixelType: \"%s\"."
                      "Must be \"%s\", \"%s\", \"%s\", \"%s\" or \"%s\"",
                      pixelType, FF_ImgRawPixelTypeStrDouble, FF_ImgRawPixelTypeStrFloat,
                      FF_ImgRawPixelTypeStrInt, FF_ImgRawPixelTypeStrShort, FF_ImgRawPixelTypeStrByte);
        return UT_False;
    }
    return UT_True;
}

static UT_Bool checkNumChans (UInt32 nChans)
{
    if (nChans < 1 || nChans > 4) {
        UT_ErrSetNum (UT_ErrParamInvalid, 
            "Invalid value for header field NumChan: %d. Must be 1, 2, 3 or 4", nChans);
        return UT_False;
    }
    return UT_True;
}

static UT_Bool checkHeight (UInt32 height)
{
    if (height < 1) {
        UT_ErrSetNum (UT_ErrParamInvalid,
            "Invalid value for header field Height: %d. Must be greater than zero", height);
        return UT_False;
    }
    return UT_True;
}

static UT_Bool checkWidth (UInt32 width)
{
    if (width < 1) {
        UT_ErrSetNum (UT_ErrParamInvalid,
            "Invalid value for header field Width: %d. Must be greater than zero", width);
        return UT_False;
    }
    return UT_True;
}

static UT_Bool readHeaderLine (FILE *fp, char *buf)
{
    int c;
    char *bufPtr, *bufEndPtr;
    UT_Bool success = UT_False;
 
    buf[0]    = '\0';
    bufPtr    = buf;
    bufEndPtr = buf + MAXSIZE;

    while (((c = fgetc (fp)) != EOF) && bufPtr < bufEndPtr) {
        if (c == '\n') {
            *bufPtr = '\0';
            success = UT_True;
            break;
        }
        *bufPtr = c;
        bufPtr++;
    }
    return success;
}

static UT_Bool readHeader (FILE *fp, Int32 *width, Int32 *height, Int32 *nChans,
                           FF_ImgByteOrderType *byteOrder, FF_ImgScanOrderType *scanOrder,
                           FF_ImgRawPixelType *pixelType, const char *fileName)
{
    char buf[MAXSIZE];
    char tmpStr[MAXSIZE];

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strMagic, tmpStr))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field Magic");
        return UT_False;
    }
    if (!checkMagic (tmpStr)) {
        return UT_False;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strWidth, width))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field Width");
        return UT_False;
    }
    if (!checkWidth (*width)) {
        return UT_False;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strHeight, height))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field Height");
        return UT_False;
    }
    if (!checkHeight (*height)) {
        return UT_False;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strNumChan, nChans))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field NumChan");
        return UT_False;
    }
    if (!checkNumChans (*nChans)) {
        return UT_False;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strByteOrder, tmpStr))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field ByteOrder");
        return UT_False;
    }
    *byteOrder = FF_ImgByteOrderTypeMotorola;
    if (strcmp (tmpStr, strIntel) == 0) {
        *byteOrder = FF_ImgByteOrderTypeIntel;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strScanOrder, tmpStr))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field ScanOrder");
        return UT_False;
    }
    *scanOrder = FF_ImgScanOrderTypeBottomUp;
    if (strcmp (tmpStr, strTopDown) == 0) {
        *scanOrder = FF_ImgScanOrderTypeTopDown;
    }

    if (!readHeaderLine (fp, buf) || (1 != sscanf (buf, strPixelType, tmpStr))) {
        UT_ErrSetNum (UT_ErrParamInvalid, "Unable to parse header field PixelType");
        return UT_False;
    }
    if (!checkPixelType (tmpStr)) {
        return UT_False;
    }
    if (strncmp (tmpStr, FF_ImgRawPixelTypeStrDouble, strlen (FF_ImgRawPixelTypeStrDouble)) == 0) {
        *pixelType = FF_ImgRawPixelTypeDouble;
    } else if (strncmp (tmpStr, FF_ImgRawPixelTypeStrFloat, strlen (FF_ImgRawPixelTypeStrFloat)) == 0) {
        *pixelType = FF_ImgRawPixelTypeFloat;
    } else if (strncmp (tmpStr, FF_ImgRawPixelTypeStrInt, strlen (FF_ImgRawPixelTypeStrInt)) == 0) {
        *pixelType = FF_ImgRawPixelTypeInt;
    } else if (strncmp (tmpStr, FF_ImgRawPixelTypeStrShort, strlen (FF_ImgRawPixelTypeStrShort)) == 0) {
        *pixelType = FF_ImgRawPixelTypeShort;
    } else if (strncmp (tmpStr, FF_ImgRawPixelTypeStrByte, strlen (FF_ImgRawPixelTypeStrByte)) == 0) {
        *pixelType = FF_ImgRawPixelTypeByte;
    }
    if (fileName) {
        printImgInfo (*width, *height, *nChans,
                      *byteOrder == FF_ImgByteOrderTypeIntel? UT_True: UT_False,
                      *scanOrder == FF_ImgScanOrderTypeTopDown? UT_True: UT_False,
                      tmpStr, fileName, "Reading image:");
    }
    return UT_True;
}

static void writeIntLine (FILE *fp, const char *fmt, Int32 val)
{
    char str[HEADLEN+1];

    sprintf (str, fmt, val);
    while (strlen (str) < HEADLEN -1) {
        strcat (str, " ");
    }
    strcat (str, "\n");
    fprintf (fp, "%s", str);
}

static void writeStringLine (FILE *fp, const char *fmt, const char *val)
{
    char str[HEADLEN+1];

    sprintf (str, fmt, val);
    while (strlen (str) < HEADLEN -1) {
        strcat (str, " ");
    }
    strcat (str, "\n");
    fprintf (fp, "%s", str);
}

static UT_Bool writeHeader (FILE *fp, Int32 width, Int32 height,
                            Int32 nChans, UT_Bool isIntel, 
                            UT_Bool isTopDown, const char *pixelType, const char *fileName)
{
    if (!checkWidth (width) ||
        !checkHeight (height) ||
        !checkNumChans (nChans) ||
        !checkPixelType (pixelType)) {
        return UT_False;
    }
    writeStringLine (fp, strMagic,     "RAW");
    writeIntLine    (fp, strWidth,     width);
    writeIntLine    (fp, strHeight,    height);
    writeIntLine    (fp, strNumChan,   nChans);
    writeStringLine (fp, strByteOrder, isIntel?   strIntel:   strMotorola);
    writeStringLine (fp, strScanOrder, isTopDown? strTopDown: strBottomUp);
    writeStringLine (fp, strPixelType, pixelType);
    if (fileName) {
        printImgInfo (width, height, nChans, isIntel, isTopDown, pixelType, fileName, "Writing image:");
    }
    return UT_True;
}

static void byteSwap (UInt8 *buf, Int32 numPixels, Int32 numBytesToSwap)
{
    Int32 i;
    UInt8 *bufPtr = buf;
    UInt8 tmp[8];

    switch (numBytesToSwap) {
        case 2: {
            for (i=0; i<numPixels; i++) {
                memcpy (tmp, bufPtr, 2);
                bufPtr[0] = tmp[1];
                bufPtr[1] = tmp[0];
                bufPtr += 2;
            }
            break;
        }
        case 4: {
            for (i=0; i<numPixels; i++) {
                memcpy (tmp, bufPtr, 4);
                bufPtr[0] = tmp[3];
                bufPtr[1] = tmp[2];
                bufPtr[2] = tmp[1];
                bufPtr[3] = tmp[0];
                bufPtr += 4;
            }
            break;
        }
        case 8: {
            for (i=0; i<numPixels; i++) {
                memcpy (tmp, bufPtr, 8);
                bufPtr[0] = tmp[7];
                bufPtr[1] = tmp[6];
                bufPtr[2] = tmp[5];
                bufPtr[3] = tmp[4];
                bufPtr[4] = tmp[3];
                bufPtr[5] = tmp[2];
                bufPtr[6] = tmp[1];
                bufPtr[7] = tmp[0];
                bufPtr += 8;
            }
            break;
        }
        default: {
            UT_ErrFatal ("byteSwap", "Invalid number of bytes to swap");
        }
    }
}

static void rawClose (FF_ImgFileDesc * px_f)
{
    rawImgFile *rawf;

    rawf = ((pixfile *) px_f->pixprivate)->fmtprivate;
    if (rawf) {
        UT_FileClose (rawf->fp);
        if (rawf->rowbuf) {
            UT_MemFree (rawf->rowbuf);
        }
        UT_MemFree (rawf);
    }
    return;
}

static UT_Bool writeError (FILE *fp)
{
    (void) fp;
    UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
    return UT_False;
}

static UT_Bool rawWriteScan (FF_ImgFileDesc *px_f, Int32 y)
{
    FF_ImgChanType *chanMap;
    Int32 i, numChans;
    Int32 stopOffset;
    size_t bytesPerLine;
    pixfile *pixf;
    rawImgFile *rawf;
    UT_Bool withAlpha;

    pixf      = px_f->pixprivate;
    rawf      = pixf->fmtprivate;
    chanMap   = rawf->chanMap;
    numChans  = rawf->numChans;
    withAlpha = rawf->withAlpha;

    if ((numChans == 2 || numChans == 4) && withAlpha == UT_False) {
        bytesPerLine = px_f->desc.width * rawf->pixelSize * (numChans - 1);
        stopOffset   = px_f->desc.width * (numChans - 1);
    } else {
        bytesPerLine = px_f->desc.width * rawf->pixelSize * numChans;
        stopOffset   = px_f->desc.width * numChans;
    }

    switch (rawf->pixelSize) {
        case 1: {
            /* All channels are in Byte format. */
            UInt8 *dest = rawf->rowbuf;
            UInt8 *stop = dest + stopOffset;
            UInt8 *src[4] = { NULL };

            for( i=0; i < 4; i++) {
                if (chanMap[i] != FF_ImgChanTypeNone) {
                    if ((chanMap[i] == FF_ImgChanTypeMatte) && (withAlpha == UT_False)) {
                        continue;
                    }
                    src[i] = pixf->buf[chanMap[i]].ubytes;
                }
            }
            while (dest < stop) {
                if (src[0]) {
                    *(dest++) = *(src[0]++);
                }
                if (src[1]) {
                    *(dest++) = *(src[1]++);
                }
                if (src[2]) {
                    *(dest++) = *(src[2]++);
                }
                if (src[3]) {
                    *(dest++) = *(src[3]++);
                }
            }
            break;
        }
        case 4: {
            /* All channels are in Float format. */
            Float32 *dest = (Float32 *)rawf->rowbuf;
            Float32 *stop = dest + stopOffset;
            Float32 *src[4] = { NULL };

            for( i=0; i < 4; i++) {
                if (chanMap[i] != FF_ImgChanTypeNone) {
                    if ((chanMap[i] == FF_ImgChanTypeMatte) && (withAlpha == UT_False)) {
                        continue;
                    }
                    src[i] = pixf->buf[chanMap[i]].floats;
                }
            }
            while (dest < stop) {
                if (src[0]) {
                    *(dest++) = *(src[0]++);
                }
                if (src[1]) {
                    *(dest++) = *(src[1]++);
                }
                if (src[2]) {
                    *(dest++) = *(src[2]++);
                }
                if (src[3]) {
                    *(dest++) = *(src[3]++);
                }
            }
            break;
        }
        default: {
            UT_ErrFatal ("rawWriteScan", "unknown pixel format");
            break;
        }
    }

    if (rawf->swapBytes) {
        byteSwap (rawf->rowbuf, stopOffset, rawf->pixelSize);
    }

    if (bytesPerLine != fwrite (rawf->rowbuf, 1, bytesPerLine, rawf->fp)) {
        return writeError (rawf->fp);
    }
    return UT_True;
}

static UT_Bool readError (FILE *fp)
{
    if (feof (fp)) {
        UT_ErrSetNum (UT_ErrUnexpEof, "fread");
    } else {
        UT_ErrSetNum (UT_ErrFromOs (), "fread");
    }
    return UT_False;
}

static UT_Bool rawReadScan (FF_ImgFileDesc *px_f, Int32 y)
{
    pixfile *pixf;
    rawImgFile *rawf;
    long totalOffset, lineOffset;
    size_t bytesPerLine;
    Int32 numChans;
    UT_Bool withAlpha;

    pixf = px_f->pixprivate;
    rawf = pixf->fmtprivate;

    withAlpha = rawf->withAlpha;
    numChans  = rawf->numChans;
    if (px_f->scanorder == FF_ImgScanOrderTypeTopDown) {
        y = px_f->desc.height - 1 - y;
    }

    bytesPerLine = px_f->desc.width * rawf->pixelSize * numChans;
    lineOffset   = y * bytesPerLine;
    totalOffset  = rawf->fpPos + lineOffset;

    if (0 > fseek (rawf->fp, totalOffset, SEEK_SET)) {
        UT_ErrSetNum (UT_ErrFromOs (), "fseek");
        return UT_False;
    }
    if (bytesPerLine != fread (rawf->rowbuf, 1, bytesPerLine, rawf->fp)) {
        return readError (rawf->fp);
    }
    if (rawf->swapBytes) {
        byteSwap (rawf->rowbuf, px_f->desc.width * numChans, rawf->pixelSize);
    }

    switch (rawf->pixelType) {
        case FF_ImgRawPixelTypeByte: {
            /* All channels are in Byte format. */
            UInt8 *src  = rawf->rowbuf;
            UInt8 *stop = src + px_f->desc.width * numChans;
            UInt8 *dest1 = NULL, *dest2 = NULL, *dest3 = NULL, *dest4 = NULL;
            switch (numChans) {
                case 1: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].ubytes;
                    while (src < stop) {
                        *(dest1++) = *(src++);
                    }
                    break;
                }
                case 2: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].ubytes;
                    dest2 = pixf->buf[FF_ImgChanTypeMatte].ubytes;
                    while (src < stop) {
                        *(dest1++) = *(src++);
                        if (withAlpha) {
                            *(dest2++) = *(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                case 3: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].ubytes;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].ubytes;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].ubytes;
                    while (src < stop) {
                        *(dest1++) = *(src++);
                        *(dest2++) = *(src++);
                        *(dest3++) = *(src++);
                    }
                    break;
                }
                case 4: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].ubytes;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].ubytes;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].ubytes;
                    if (withAlpha) {
                        dest4 = pixf->buf[FF_ImgChanTypeMatte].ubytes;
                    }
                    while (src < stop) {
                        *(dest1++) = *(src++);
                        *(dest2++) = *(src++);
                        *(dest3++) = *(src++);
                        if (withAlpha) {
                            *(dest4++) = *(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("rawReadScan", "Illegal number of channels");
                    break;
                }
            }
            break;
        }
        case FF_ImgRawPixelTypeShort: {
            /* All channels are in Short format. */
            UInt16 *src  = (UInt16 *)rawf->rowbuf;
            UInt16 *stop = src + px_f->desc.width * numChans;
            Float32 *dest1 = NULL, *dest2 = NULL, *dest3 = NULL, *dest4 = NULL;
            switch (numChans) {
                case 1: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                    }
                    break;
                }
                case 2: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest2++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                case 3: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                    }
                    break;
                }
                case 4: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    if (withAlpha) {
                        dest4 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    }
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest4++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("rawReadScan", "Illegal number of channels");
                    break;
                }
            }
            break;
        }
        case FF_ImgRawPixelTypeInt: {
            /* All channels are in Int format. */
            UInt32 *src  = (UInt32 *)rawf->rowbuf;
            UInt32 *stop = src + px_f->desc.width * numChans;
            Float32 *dest1 = NULL, *dest2 = NULL, *dest3 = NULL, *dest4 = NULL;
            switch (numChans) {
                case 1: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                    }
                    break;
                }
                case 2: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest2++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                case 3: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                    }
                    break;
                }
                case 4: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    if (withAlpha) {
                        dest4 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    }
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest4++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("rawReadScan", "Illegal number of channels");
                    break;
                }
            }
            break;
        }
        case FF_ImgRawPixelTypeFloat: {
            /* All channels are in Float32 format. */
            Float32 *src  = (Float32 *)rawf->rowbuf;
            Float32 *stop = src + px_f->desc.width * numChans;
            Float32 *dest1 = NULL, *dest2 = NULL, *dest3 = NULL, *dest4 = NULL;
            switch (numChans) {
                case 1: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                    }
                    break;
                }
                case 2: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest2++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                case 3: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                    }
                    break;
                }
                case 4: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    if (withAlpha) {
                        dest4 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    }
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest4++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("rawReadScan", "Illegal number of channels");
                    break;
                }
            }
            break;
        }
        case FF_ImgRawPixelTypeDouble: {
            /* All channels are in Float64 format. */
            Float64 *src  = (Float64 *)rawf->rowbuf;
            Float64 *stop = src + px_f->desc.width * numChans;
            Float32 *dest1 = NULL, *dest2 = NULL, *dest3 = NULL, *dest4 = NULL;
            switch (numChans) {
                case 1: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                    }
                    break;
                }
                case 2: {
                    dest1 = pixf->buf[FF_ImgChanTypeBrightness].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest2++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                case 3: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                    }
                    break;
                }
                case 4: {
                    dest1 = pixf->buf[FF_ImgChanTypeRed].floats;
                    dest2 = pixf->buf[FF_ImgChanTypeGreen].floats;
                    dest3 = pixf->buf[FF_ImgChanTypeBlue].floats;
                    if (withAlpha) {
                        dest4 = pixf->buf[FF_ImgChanTypeMatte].floats;
                    }
                    while (src < stop) {
                        *(dest1++) = (Float32)*(src++);
                        *(dest2++) = (Float32)*(src++);
                        *(dest3++) = (Float32)*(src++);
                        if (withAlpha) {
                            *(dest4++) = (Float32)*(src++);
                        } else {
                            src++;
                        }
                    }
                    break;
                }
                default: {
                    UT_ErrFatal ("rawReadScan", "Illegal number of channels");
                    break;
                }
            }
            break;
        }
        default: {
            UT_ErrFatal ("rawReadScan", "Illegal pixel type");
            break;
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenWriteRaw
 *
 *      Usage:          Open a RAW pixel file for writing.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenWriteRaw
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f,
 *                               const char *opts)
 *
 *      Description:    RAW pixel file "name" is opened for writing.
 *
 *                      RAW files are always written in native byte order.
 *                      RAW files are written in FF_ImgScanOrderTypeBottomUp
 *                      scan order. The scan order may be changed by option
 *                      "-scanorder" using "BottomUp" or "TopDown" as option values.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenReadRaw
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenWriteRaw (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    Int32 numChans = 0, realChans;
    Int32 pixelSize = 0;
    FF_ImgFmtType chanFmt = FF_ImgFmtTypeNone;
    UT_MemState memstate;
    pixfile *pixf;
    rawImgFile *rawf;
    FILE *fp;
    FF_ImgScanOrderType optScanOrder;
    FF_ImgByteOrderType optByteOrder;
    UT_Bool topDown;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;
    UT_Bool swapBytes;
    UT_Bool useHeader;

    memstate = UT_MemRemember ();

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default scanOrder flag and check options of -scanorder key. */
    optScanOrder = FF_ImgScanOrderTypeBottomUp;
    FF_ImgGetOptScanOrder (opts, &optScanOrder);
    topDown = (optScanOrder == FF_ImgScanOrderTypeTopDown? UT_True: UT_False);

    /* Set default byteOrder flag and check options of -byteorder key. */
    optByteOrder = (UT_PortIsIntel ()? FF_ImgByteOrderTypeIntel: FF_ImgByteOrderTypeMotorola);
    FF_ImgGetOptByteOrder (opts, &optByteOrder);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);

    /* Set default header flag and check options of -useheader key. */
    useHeader = UT_True;
    FF_ImgGetOptUseHeader (opts, &useHeader);

    if (!(rawf = UT_MemTemp (rawImgFile))) {
        return UT_False;
    }

    if (!(fp = UT_FileOpen (name, "wb"))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    if (px_f->channel[FF_ImgChanTypeBrightness]) {
        chanFmt = px_f->channel[FF_ImgChanTypeBrightness];
        if (px_f->channel[FF_ImgChanTypeMatte]) {
            numChans = 2;
            rawf->chanMap = sDefMap2;
        } else {
            rawf->chanMap = sDefMap1;
            numChans = 1;
        }
    } else if (px_f->channel[FF_ImgChanTypeRed]) {
        chanFmt = px_f->channel[FF_ImgChanTypeRed];
        if (px_f->channel[FF_ImgChanTypeMatte]) {
            rawf->chanMap = sDefMap4;
            numChans = 4;
        } else {
            rawf->chanMap = sDefMap3;
            numChans = 3;
        }
    }

    realChans = numChans;
    if ((numChans == 2 || numChans == 4) && optWithAlpha == UT_False) {
        realChans--;
    }
    pixelSize = (chanFmt == FF_ImgFmtTypeUByte? 1: 4);
    if (!(rawf->rowbuf = UT_MemTempArray (px_f->desc.width * pixelSize * numChans, UInt8))) {
        UT_MemRestore (memstate);
        UT_FileClose (fp);
        return UT_False;
    }

    swapBytes = ((( UT_PortIsIntel () && (optByteOrder != FF_ImgByteOrderTypeIntel)) ||
                  (!UT_PortIsIntel () && (optByteOrder == FF_ImgByteOrderTypeIntel))) &&
                  pixelSize != 1);

    rawf->fp        = fp;
    rawf->numChans  = numChans;
    rawf->withAlpha = optWithAlpha;
    rawf->pixelSize = pixelSize;
    rawf->swapBytes = swapBytes;

    if (useHeader) {
        UT_Bool isIntel = (optByteOrder == FF_ImgByteOrderTypeIntel? UT_True: UT_False);
        switch (chanFmt) {
            case FF_ImgFmtTypeUByte: {
                if (!writeHeader (fp, px_f->desc.width, px_f->desc.height, realChans,
                                  isIntel, topDown, FF_ImgRawPixelTypeStrByte,
                                  optVerbose? name: NULL)) {
                    UT_MemRestore (memstate);
                    UT_FileClose (fp);
                    return UT_False;
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                if (!writeHeader (fp, px_f->desc.width, px_f->desc.height, realChans,
                                  isIntel, topDown, FF_ImgRawPixelTypeStrFloat,
                                  optVerbose? name: NULL)) {
                    UT_MemRestore (memstate);
                    UT_FileClose (fp);
                    return UT_False;
                }
                break;
            }
            default: {
                UT_ErrFatal ("FF_ImgOpenWriteRaw", "unknown pixel format");
            }
        }
    }

    /* Initialize the FF_ImgFileDesc. */
    ((pixfile *) px_f->pixprivate)->fmtprivate = rawf;
    pixf            = px_f->pixprivate;
    pixf->readscan  = NULL;
    pixf->writescan = rawWriteScan;
    pixf->close     = rawClose;
    px_f->scanorder = optScanOrder;

    UT_MemSave (memstate);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenReadRaw
 *
 *      Usage:          Open a raw pixel file for reading.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenReadRaw
 *                              (const char *name,
 *                              FF_ImgFileDesc *px_f,
 *                              const char *opts)
 *
 *      Description:    Raw pixel file "name" is opened for reading.
 *
 *                      Only FF_ImgChanTypeRed, FF_ImgChanTypeGreen,
 *                      FF_ImgChanTypeBlue and FF_ImgChanTypeMatte channel
 *                      types are supported.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenWriteRaw
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenReadRaw (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    FILE *fp;
    pixfile *pixf;
    rawImgFile *rawf;
    UT_MemState memstate;
    Int32 chn;
    FF_ImgFmtType chanType;
    Int32 width = -1, height = -1, numChans = -1;
    UT_Bool optWithAlpha;
    UT_Bool useHeader;
    UT_Bool optVerbose;
    UT_Bool swapBytes;
    Int32 pixelSize = 0;
    Int32 skipBytes = 0;
    FF_ImgScanOrderType scanOrder = FF_ImgScanOrderTypeTopDown;
    FF_ImgByteOrderType byteOrder = FF_ImgByteOrderTypeIntel;
    FF_ImgRawPixelType pixelType  = FF_ImgRawPixelTypeByte;

    memstate = UT_MemRemember ();
    pixf = px_f->pixprivate;

    if (!(fp = UT_FileOpen (name, "rb"))) {
        return UT_False;
    }

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default header flag and check options of -useheader key. */
    useHeader = UT_True;
    FF_ImgGetOptUseHeader (opts, &useHeader);

    if (useHeader) {
        if (!readHeader (fp, &width, &height, &numChans, &byteOrder, &scanOrder, &pixelType, optVerbose? name: NULL)) {
            UT_FileClose (fp);
            return UT_False;
        }
        /* Header fields like width, height or pixelType are already checked when reading the header.
         * No need to check further. */
    } else {
        /* Check values of necessary options: -width, -height, -numchan. */
        FF_ImgGetOptWidth    (opts, &width);
        FF_ImgGetOptHeight   (opts, &height);
        FF_ImgGetOptNumChans (opts, &numChans);
        /* Check values of optional options: -byteorder, -scanorder, -pixeltype. */
        FF_ImgGetOptByteOrder (opts, &byteOrder);
        FF_ImgGetOptScanOrder (opts, &scanOrder);
        FF_ImgGetOptPixelType (opts, &pixelType);
        FF_ImgGetOptSkipBytes (opts, &skipBytes);

        if (width <= 0 || height <= 0 || numChans <= 0) {
            UT_FileClose (fp);
            if (width <= 0) {
                UT_ErrSetNum (UT_ErrParamInvalid, "Image width is zero or negative");
            } else if (height <= 0) {
                UT_ErrSetNum (UT_ErrParamInvalid, "Image height is zero or negative");
            } else if (numChans <= 0) {
                UT_ErrSetNum (UT_ErrParamInvalid, "Number of channels is zero or negative");
            }
            return UT_False;
        }

        if (skipBytes > 0) {
            int c;
            while (((c = fgetc (fp)) != EOF) && skipBytes) {
                skipBytes--;
            }
        }
    }

    switch (pixelType) {
        case FF_ImgRawPixelTypeDouble: {
            pixelSize = 8;
            break;
        }
        case FF_ImgRawPixelTypeFloat: {
            pixelSize = 4;
            break;
        }
        case FF_ImgRawPixelTypeInt: {
            pixelSize = 4;
            break;
        }
        case FF_ImgRawPixelTypeShort: {
            pixelSize = 2;
            break;
        }
        case FF_ImgRawPixelTypeByte: {
            pixelSize = 1;
            break;
        }
    }

    swapBytes = ((( UT_PortIsIntel () && (byteOrder != FF_ImgByteOrderTypeIntel)) ||
                  (!UT_PortIsIntel () && (byteOrder == FF_ImgByteOrderTypeIntel))) &&
                  pixelSize != 1);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = (numChans == 2 || numChans == 4)? UT_True: UT_False;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);
    if (numChans == 1 || numChans == 3) {
        optWithAlpha = UT_False;
    }

    chanType = FF_ImgFmtTypeFloat;
    if (pixelType == FF_ImgRawPixelTypeByte) {
        chanType = FF_ImgFmtTypeUByte;
    }

    if (!(rawf = UT_MemTemp (rawImgFile)) ||
        !(rawf->rowbuf = UT_MemTempArray (width * pixelSize * numChans, UInt8))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    /* Fill image descriptor with default values.
     * These values are not contained in a RAW file. */
    px_f->desc.red.x   = 0.670f;
    px_f->desc.red.y   = 0.330f;
    px_f->desc.green.x = 0.210f;
    px_f->desc.green.y = 0.710f;
    px_f->desc.blue.x  = 0.140f;
    px_f->desc.blue.y  = 0.080f;
    px_f->desc.white.x = 0.313f;
    px_f->desc.white.y = 0.329f;
    px_f->desc.gamma   = 1.0;

    /* Fill image descriptor with values from RAW file. */
    px_f->desc.width  = width;
    px_f->desc.height = height;
    px_f->desc.aspect = (Float32)width / (Float32)height;

    px_f->scanorder = scanOrder;

    for (chn=0; chn<FF_NumImgChanTypes; chn++) {
        px_f->channel[chn] = FF_ImgFmtTypeNone;
    }

    switch (numChans) {
        case 1: {
            px_f->channel[FF_ImgChanTypeBrightness]  = chanType;
            break;
        }
        case 2: {
            px_f->channel[FF_ImgChanTypeBrightness]  = chanType;
            if (optWithAlpha) {
                px_f->channel[FF_ImgChanTypeMatte] = chanType;
            }
            break;
        }
        case 3: {
            px_f->channel[FF_ImgChanTypeRed]   = chanType;
            px_f->channel[FF_ImgChanTypeGreen] = chanType;
            px_f->channel[FF_ImgChanTypeBlue]  = chanType;
            break;
        }
        case 4: {
            px_f->channel[FF_ImgChanTypeRed]   = chanType;
            px_f->channel[FF_ImgChanTypeGreen] = chanType;
            px_f->channel[FF_ImgChanTypeBlue]  = chanType;
            if (optWithAlpha) {
                px_f->channel[FF_ImgChanTypeMatte] = chanType;
            }
            break;
        }
        default: {
            UT_ErrFatal ("FF_ImgOpenReadRaw", "Illegal number of channels");
            break;
        }
    }

    /* Fill in the "pixfile". */
    pixf->fmtprivate = rawf;
    pixf->readscan   = rawReadScan;
    pixf->close      = rawClose;

    /* Fill in the "rawImgFile". */
    rawf->fp = fp;
    rawf->fpPos = ftell (fp);
    rawf->numChans  = numChans;
    rawf->withAlpha = optWithAlpha;
    rawf->pixelSize = pixelSize;
    rawf->pixelType = pixelType;
    rawf->byteOrder = byteOrder;
    rawf->swapBytes = swapBytes;

    UT_MemSave (memstate);
    return UT_True;
}
