/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImageSgi.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The file format dependent layer of the image file
 *                      format library: Functions to read and write
 *                      files in Silicon Graphics' RGB image file format.
 *                      The following SGI image types are supported:
 *
 *                          Uncompressed RGB image
 *                          Run-length encoded RGB image
 *
 *                      Both types may have a transparency channel.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      FF_ImgOpenWriteSgi
 *                      FF_ImgOpenReadSgi
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Const.h"
#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Portable.h"

#include "FF_Image.h"
#include "FF_ImagePrivate.h"

#include "FF_ImageSgiLib.h"
#include "FF_ImageSgi.h"

#define SGI_INT_MAGIC 7531

/* Format dependent pixel file descriptor */

typedef struct {
    IMAGE   *image;     /* Image struct in SGI format */
    UT_Bool withAlpha;  /* Flag indicating whether to read/write alpha channel */
    UInt16  *chanBuf;   /* Buffer for one channel of a scanline. */
} SGIFILE;

static void printImgInfo (IMAGE *th, const char *fileName, const char *msg)
{
    printf ("%s %s\n", msg, fileName);
    printf ("\tSize in pixel     : %d x %d\n", th->xsize, th->ysize);
    printf ("\tNumber of channels: %d\n", (th->zsize));
    printf ("\tBytes per pixel   : %d\n", BPP(th->type));
    printf ("\tCompression       : %s\n", ISRLE(th->type)? "RLE": "NONE");
    fflush (stdout);
}

/* Close a pixel file previously opened with
   FF_ImgOpenWriteSgi or FF_ImgOpenReadSgi. */

static void closeSgi (FF_ImgFileDesc *px_f)
{
    SGIFILE *sgif;
    sgif = ((pixfile *)px_f->pixprivate)->fmtprivate;
    if (sgif) {
        if (sgif->image) {
            sgiClose (sgif->image);
        }
        if (sgif->chanBuf) {
            UT_MemFree (sgif->chanBuf);
        }
        UT_MemFree (sgif);
    }
    return;
}

/* Try to read information stored in the "name" field of the IMAGE header. */

static UT_Bool readIdData (const char *buf, FF_ImgFileDesc *px_f)
{
    Int16 magic;

    if ((magic = UT_PortByteToInt16 ((const UInt8 *)(buf + 0))) != SGI_INT_MAGIC) {
        return UT_False;
    }
    px_f->desc.gamma    = UT_PortByteToFloat32 ((const UInt8 *)(buf +  2));
    px_f->desc.aspect   = UT_PortByteToFloat32 ((const UInt8 *)(buf +  6));
    px_f->desc.red.x    = UT_PortByteToFloat32 ((const UInt8 *)(buf + 10));
    px_f->desc.red.y    = UT_PortByteToFloat32 ((const UInt8 *)(buf + 14));
    px_f->desc.green.x  = UT_PortByteToFloat32 ((const UInt8 *)(buf + 18));
    px_f->desc.green.y  = UT_PortByteToFloat32 ((const UInt8 *)(buf + 22));
    px_f->desc.blue.x   = UT_PortByteToFloat32 ((const UInt8 *)(buf + 26));
    px_f->desc.blue.y   = UT_PortByteToFloat32 ((const UInt8 *)(buf + 30));
    px_f->desc.white.x  = UT_PortByteToFloat32 ((const UInt8 *)(buf + 34));
    px_f->desc.white.y  = UT_PortByteToFloat32 ((const UInt8 *)(buf + 38));
    return UT_True;
}

/* Write the information into the name field of the IMAGE header. */

static void writeIdData (const char *buf, FF_ImgFileDesc *px_f)
{
    UT_PortInt16ToByte   (SGI_INT_MAGIC,        (UInt8 *)(buf +  0));
    UT_PortFloat32ToByte (px_f->desc.gamma,     (UInt8 *)(buf +  2));
    UT_PortFloat32ToByte (px_f->desc.aspect,    (UInt8 *)(buf +  6));
    UT_PortFloat32ToByte (px_f->desc.red.x,     (UInt8 *)(buf + 10));
    UT_PortFloat32ToByte (px_f->desc.red.y,     (UInt8 *)(buf + 14));
    UT_PortFloat32ToByte (px_f->desc.green.x,   (UInt8 *)(buf + 18));
    UT_PortFloat32ToByte (px_f->desc.green.y,   (UInt8 *)(buf + 22));
    UT_PortFloat32ToByte (px_f->desc.blue.x,    (UInt8 *)(buf + 26));
    UT_PortFloat32ToByte (px_f->desc.blue.y,    (UInt8 *)(buf + 30));
    UT_PortFloat32ToByte (px_f->desc.white.x,   (UInt8 *)(buf + 34));
    UT_PortFloat32ToByte (px_f->desc.white.y,   (UInt8 *)(buf + 38));
    return;
}

static UT_Bool readHeader (SGIFILE *sgif, FF_ImgFileDesc *px_f)
{
    Int32         chan;
    FF_ImgFmtType chanType;

    if (!readIdData (sgif->image->name, px_f)) {
        px_f->desc.red.x   = 0.670f;
        px_f->desc.red.y   = 0.330f;
        px_f->desc.green.x = 0.210f;
        px_f->desc.green.y = 0.710f;
        px_f->desc.blue.x  = 0.140f;
        px_f->desc.blue.y  = 0.080f;
        px_f->desc.white.x = 0.313f;
        px_f->desc.white.y = 0.329f;
        px_f->desc.aspect = (Float32)sgif->image->xsize /
                            (Float32)sgif->image->ysize;
        px_f->desc.gamma  = 1.0;
    }

    px_f->scanorder   = FF_ImgScanOrderTypeBottomUp;
    px_f->desc.width  = sgif->image->xsize;
    px_f->desc.height = sgif->image->ysize;

    for (chan=0; chan<FF_NumImgChanTypes; chan++) {
        px_f->channel[chan] = FF_ImgFmtTypeNone;
    }

    chanType  = BPP (sgif->image->type) == 1?
                     FF_ImgFmtTypeUByte: FF_ImgFmtTypeFloat;

    switch (sgif->image->zsize) {
        case 1: {
            px_f->channel[FF_ImgChanTypeBrightness] = chanType;
            break;
        }
        case 2: {
            px_f->channel[FF_ImgChanTypeBrightness] = chanType;
            if (sgif->withAlpha) {
                px_f->channel[FF_ImgChanTypeMatte] = chanType;
            }
            break;
        }
        case 3: {
            px_f->channel[FF_ImgChanTypeRed]   = chanType;
            px_f->channel[FF_ImgChanTypeGreen] = chanType;
            px_f->channel[FF_ImgChanTypeBlue]  = chanType;
            break;
        }
        case 4: {
            px_f->channel[FF_ImgChanTypeRed]   = chanType;
            px_f->channel[FF_ImgChanTypeGreen] = chanType;
            px_f->channel[FF_ImgChanTypeBlue]  = chanType;
            if (sgif->withAlpha) {
                px_f->channel[FF_ImgChanTypeMatte] = chanType;
            }
            break;
        }
        default: {
            UT_ErrFatal ("readHeader", "Illegal number of channels");
            return UT_False;
        }
    }
    return UT_True;
}

static UT_Bool writeChannel (const FF_ImgFileDesc *px_f, FF_ImgChanType chantype, 
                              Int32 sgiChan, Int32 y, Int32 n)
{
    UInt16  *dest;
    pixfile *pixf;
    SGIFILE *sgif;

    pixf = px_f->pixprivate;
    sgif = pixf->fmtprivate;

    dest = sgif->chanBuf;

    switch (px_f->channel[chantype]) {
        case FF_ImgFmtTypeNone: {
            break;
        }
        case FF_ImgFmtTypeUByte: {
            UInt8 *src = pixf->buf[chantype].ubytes;
            UInt8 *stop = src + n;
            while (src < stop) {
                *(dest++) = *(src++);
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            Float32 *src = pixf->buf[chantype].floats;
            Float32 *stop = src + n;
            while (src < stop) {
                *dest = (UInt16) UT_MIN (MaxUInt16, UT_MAX (0, (*src * MaxUInt16 + 0.5)));
                src++;
                dest++;
            }
            break;
        }
    }
    if (-1 == sgiPutrow (sgif->image, sgif->chanBuf, y, sgiChan)) {
        UT_ErrSetNum (UT_ErrFromOs (), "write");
        return UT_False;
    }
    return UT_True;
}

/* All channels of scan line number y for pixel file "file" are stored
   in the file. SGI files are written in FF_ImgScanOrderTypeBottomUp order. */

static UT_Bool sgiWriteScan (FF_ImgFileDesc *px_f, Int32 y)
{
    pixfile *pixf;
    SGIFILE *sgif;

    pixf = px_f->pixprivate;
    sgif = pixf->fmtprivate;

    /* Copy the data to their final destination. */
    if (!writeChannel (px_f, FF_ImgChanTypeRed,   0, y, px_f->desc.width) ||
        !writeChannel (px_f, FF_ImgChanTypeGreen, 1, y, px_f->desc.width) ||
        !writeChannel (px_f, FF_ImgChanTypeBlue,  2, y, px_f->desc.width)) {
        return UT_False;
    }

    if (sgif->image->zsize > 3 && sgif->withAlpha) {
        if (!writeChannel (px_f, FF_ImgChanTypeMatte, 3, y, px_f->desc.width)) {
            return UT_False;
        }
    }

    return UT_True;
}

static UT_Bool readChannel (const FF_ImgFileDesc *px_f, FF_ImgChanType chantype, 
                            Int32 sgiChan, Int32 y, Int32 n)
{
    UInt16  *src, *stop;
    pixfile *pixf;
    SGIFILE *sgif;

    pixf = px_f->pixprivate;
    sgif = pixf->fmtprivate;

    src  = sgif->chanBuf;
    stop = src + n;

    if (-1 == sgiGetrow (sgif->image, sgif->chanBuf, y, sgiChan)) {
        UT_ErrSetNum (UT_ErrFromOs (), "read");
        return UT_False;
    }

    switch (px_f->channel[chantype]) {
        case FF_ImgFmtTypeNone: {
            break;
        }
        case FF_ImgFmtTypeUByte: {
            UInt8 *dest = pixf->buf[chantype].ubytes;
            while (src < stop) {
                *(dest++) = *(src++);
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            Float32 *dest = pixf->buf[chantype].floats;
            while (src < stop) {
                *(dest++) = (Float32)(UInt32)*(src++) * (1.0 / MaxUInt16);
            }
            break;
        }
    }
    return UT_True;
}

/* The channels of scan line number "y" of the pixel file "px_f" are read.
   Note that the SGI file format only supports FF_ImgScanOrderTypeBottomUp 
   scanline order. */

static UT_Bool sgiReadScan (FF_ImgFileDesc *px_f, Int32 y)
{
    pixfile *pixf;
    SGIFILE *sgif;
    Int32   nchan;

    pixf = px_f->pixprivate;
    sgif = pixf->fmtprivate;

    nchan = sgif->image->zsize;

    /* Copy the data to their final destination. */
    switch (nchan) {
        case 1: {
            if (!readChannel (px_f, FF_ImgChanTypeBrightness, 0, y, px_f->desc.width)) {
                return UT_False;
            }
            break;
        }
        case 2: {
            if (!readChannel (px_f, FF_ImgChanTypeBrightness,  0, y, px_f->desc.width)) {
                return UT_False;
            }
            if (sgif->withAlpha) {
                if (!readChannel (px_f, FF_ImgChanTypeMatte, 1, y, px_f->desc.width)) {
                    return UT_False;
                }
            }
            break;
        }
        case 3: {
            if (!readChannel (px_f, FF_ImgChanTypeRed,   0, y, px_f->desc.width) ||
                !readChannel (px_f, FF_ImgChanTypeGreen, 1, y, px_f->desc.width) ||
                !readChannel (px_f, FF_ImgChanTypeBlue,  2, y, px_f->desc.width)) {
                return UT_False;
            }
            break;
        }
        case 4: {
            if (!readChannel (px_f, FF_ImgChanTypeRed,   0, y, px_f->desc.width) ||
                !readChannel (px_f, FF_ImgChanTypeGreen, 1, y, px_f->desc.width) ||
                !readChannel (px_f, FF_ImgChanTypeBlue,  2, y, px_f->desc.width)) {
                return UT_False;
            }
            if (sgif->withAlpha) {
                if (!readChannel (px_f, FF_ImgChanTypeMatte, 3, y, px_f->desc.width)) {
                    return UT_False;
                }
            }
            break;
        }
        default: {
            UT_ErrFatal ("sgiReadScan", "Illegal number of channels");
            return UT_False;
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenWriteSgi
 *
 *      Usage:          Open a SGI pixel file for writing.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenWriteSgi
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f,
 *                               const char *opts)
 *
 *      Description:    SGI pixel file "name" is opened for writing.
 *
 *                      Only FF_ImgChanTypeRed, FF_ImgChanTypeGreen,
 *                      FF_ImgChanTypeBlue and FF_ImgChanTypeMatte channel
 *                      types are supported.
 *                      The only data compression method supported is
 *                      run-length encoding.
 *                      SGI files are written in FF_ImgScanOrderTypeBottomUp order. 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenReadSgi
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenWriteSgi (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    pixfile *pixf;
    SGIFILE *sgif;
    UT_MemState memstate;
    FF_ImgComprType optCompression;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;
    UT_Bool matte;
    UInt32  bpp = 1;

    memstate = UT_MemRemember ();

    if (!(sgif = UT_MemTemp (SGIFILE)) ||
        !(sgif->chanBuf = UT_MemTempArray (px_f->desc.width, UInt16))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    pixf = px_f->pixprivate;
    pixf->fmtprivate = sgif;

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default compression value and check options of -compression key. */
    optCompression = FF_ImgComprTypeNone;
    FF_ImgGetOptCompression (opts, &optCompression);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);
    sgif->withAlpha = optWithAlpha;

    matte = UT_False;
    if (px_f->channel[FF_ImgChanTypeMatte] != FF_ImgFmtTypeNone && optWithAlpha) {
        matte = UT_True;
    }

    if ((px_f->channel[FF_ImgChanTypeRed]   == FF_ImgFmtTypeFloat) ||
        (px_f->channel[FF_ImgChanTypeGreen] == FF_ImgFmtTypeFloat) ||
        (px_f->channel[FF_ImgChanTypeBlue]  == FF_ImgFmtTypeFloat) || 
        (matte && px_f->channel[FF_ImgChanTypeMatte] == FF_ImgFmtTypeFloat)) {
        bpp = 2;
    }
    if (!(sgif->image = sgiOpen (name, "wb",
                                 optCompression == FF_ImgComprTypeNone? VERBATIM(bpp): RLE(bpp), 
                                 matte? 4: 3, 
                                 px_f->desc.width, px_f->desc.height, 
                                 matte? 4: 3))) {
        UT_ErrSetNum (UT_ErrFromOs(), str_open_write, name);
        return UT_False;
    }
    writeIdData (sgif->image->name, px_f);

    pixf->writescan = sgiWriteScan;
    pixf->close     = closeSgi;
    px_f->scanorder = FF_ImgScanOrderTypeBottomUp;

    if (optVerbose) {
         printImgInfo (sgif->image, name, "Saving image:");
    }

    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenReadSgi
 *
 *      Usage:          Open a SGI pixel file for reading.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenReadSgi
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f,
 *                               const char *opts)
 *
 *      Description:    SGI pixel file "name" is opened for reading.
 *
 *                      Only FF_ImgChanTypeRed, FF_ImgChanTypeGreen,
 *                      FF_ImgChanTypeBlue and FF_ImgChanTypeMatte channel
 *                      types are supported.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenWriteSgi
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenReadSgi (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    pixfile *pixf;
    SGIFILE *sgif;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;
    UT_MemState memstate;

    memstate = UT_MemRemember ();

    pixf = px_f->pixprivate;

    if (!(sgif = UT_MemTemp (SGIFILE))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);
    sgif->withAlpha = optWithAlpha;

    /* Open the file for reading. */
    if (!(sgif->image = sgiOpen (name, "rb", 0, 0, 0, 0, 0))) {
        UT_MemRestore (memstate);
        UT_ErrSetNum (FF_ImgErrFormat, str_open_read, name);
        return UT_False;
    }

    if (!readHeader (sgif, px_f)) {
        UT_MemRestore (memstate);
        sgiClose (sgif->image);
        return UT_False;
    }

    if (!(sgif->chanBuf = UT_MemTempArray (px_f->desc.width, UInt16))) {
        UT_MemRestore (memstate);
        sgiClose (sgif->image);
        return UT_False;
    }

    /* Fill in the "pixfile". */
    pixf->fmtprivate = sgif;
    pixf->readscan   = sgiReadScan;
    pixf->close      = closeSgi;

    UT_MemSave (memstate);

    if (optVerbose) {
         printImgInfo (sgif->image, name, "Reading image:");
    }
    return UT_True;
}
