/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         FileFormats
 *      Filename:       FF_ImageTarga.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The file format dependent layer of the image file
 *                      format library: Functions to read and write
 *                      files in Truevision's Targa pixel file format.
 *                      The following Targa image types are supported:
 *
 *                          Type  2: Uncompressed RGB image
 *                          Type 10: Run-length encoded RGB image
 *
 *                      Both types may have an optional transparency channel.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      FF_ImgOpenWriteTarga
 *                      FF_ImgOpenReadTarga
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Error.h"
#include "UT_Portable.h"
#include "UT_Memory.h"
#include "UT_FileIO.h"

#include "FF_Image.h"
#include "FF_ImagePrivate.h"
#include "FF_ImageTarga.h"

#define str_colormap "Invalid colormap size (%d)"

/* Supported TARGA version numbers */

#define TGA_RGB_UNCOMP   2
#define TGA_RGB_COMP    10

#define TGA_MODE_SAME   0
#define TGA_MODE_DIFF   1

#define TGA_INT_MAGIC   7531
#define TGA_HDR_LEN     18         /* Size of struct TGAHEADER */
#define TGA_ID_DATA_LEN (2 + 10*4) /* Magic number + 10 Float32's */

#define MINRUN   3
#define MAXRUN 127

/* Macros for acessing header fields */
#define ENC_LEFT_RIGHT(imgdes) (((imgdes >> 4) & 0x1)? UT_False: UT_True)
#define ENC_TOP_DOWN(imgdes)   (((imgdes >> 5) & 0x1)? UT_True: UT_False)
#define NCHAN(pixsize)         ((pixsize == 24) ? 3: 4)
#define IS_COMPRESSED(imgtyp)  ((imgtyp == TGA_RGB_COMP)? UT_True: UT_False)

/* Format dependent pixel file descriptor */

typedef struct {
    FILE  *fp;                   /* File pointer of Targa file */
    Int32 lastscan,              /* Last scan line read or written */
          nchan,                 /* Number of channels (3 or 4) */
          scanrest,              /* Number of pixels belonging to next scanline */
          scanmode;              /* Current scan mode */
    UInt8 *rowbuf;               /* Scanline buffer for reading uncompressed pixel data */
    UInt8 *red,                  /* Scanline buffer for red component */
          *green,                /* Scanline buffer for green component */
          *blue,                 /* Scanline buffer for blue component */
          *matte;                /* Scanline buffer for matte component */
    FF_ImgComprType compression; /* Compression type for writing */
    UT_Bool withAlpha;           /* Use alpha channel for reading or writing */
} TGAFILE;

/* A Targa header */

typedef struct {
    UInt8 numid;
    UInt8 maptyp;
    UInt8 imgtyp;
    Int16 maporig;
    Int16 mapsize;
    UInt8 mapbits;
    Int16 xorig;
    Int16 yorig;
    Int16 xsize;
    Int16 ysize;
    UInt8 pixsize;
    UInt8 imgdes;
} TGAHEADER;

/* Macros for reading from the pixbuf */

#define FROM_PIXBUF                     \
    {   *(blue++)  = pixbuf[0];         \
        *(green++) = pixbuf[1];         \
        *(red++)   = pixbuf[2];         \
        if (tgaf->nchan == 4)           \
            *(matte++) = pixbuf[3];     \
    }

static UInt8
        pixbuf[4],      /* Buffer for one pixel */
        *rowbuf,        /* Scanline pointers for reading uncompressed pixel values */
        *red,           /* Scanline pointers for stepping through */
        *green,
        *blue,
        *matte;

static UT_Bool          
        leftToRight = UT_True;   /* Are scanlines oriented from left to right or vice versa */

static void printImgInfo (TGAHEADER *th, const char *fileName, const char *msg)
{
    printf ("%s %s\n", msg, fileName);
    printf ("\tSize in pixel      : %d x %d\n", th->xsize, th->ysize);
    printf ("\tNumber of channels : %d\n", NCHAN(th->pixsize));
    printf ("\tCompression        : %s\n", IS_COMPRESSED(th->imgtyp)? "RLE": "NONE");
    printf ("\tVertical encoding  : %s\n", ENC_TOP_DOWN(th->imgdes)? "TopDown": "BottomUp");
    printf ("\tHorizontal encoding: %s\n", ENC_LEFT_RIGHT(th->imgdes)? "LeftRight": "RightLeft");
    fflush (stdout);
}

/* Close a pixel file previously opened with
   FF_ImgOpenWriteTarga or FF_ImgOpenReadTarga. */

static void tgaClose (FF_ImgFileDesc *px_f)
{
    TGAFILE *tgaf;
    tgaf = ((pixfile *)px_f->pixprivate)->fmtprivate;
    if (tgaf) {
        UT_FileClose (tgaf->fp);
        UT_MemFree (tgaf->rowbuf);
        UT_MemFree (tgaf->red);
        UT_MemFree (tgaf->green);
        UT_MemFree (tgaf->blue);
        UT_MemFree (tgaf->matte);
        UT_MemFree (tgaf);
    }
    return;
}

static UT_Bool readError (FILE *fp)
{
    if (feof (fp)) {
        UT_ErrSetNum (UT_ErrUnexpEof, "fread");
    } else {
        UT_ErrSetNum (UT_ErrFromOs (), "fread");
    }
    return UT_False;
}

static UT_Bool writeError (void)
{
    UT_ErrSetNum (UT_ErrFromOs (), "fwrite");
    return UT_False;
}

/* Read 1 byte, representing an unsigned integer number. */

static UT_Bool readUInt8 (FILE *fp, UInt8 *b)
{
    if (1 != fread (b, sizeof(*b), 1, fp)) {
        return readError (fp);
    }
    return UT_True;
}

/* Read 2 bytes, representing a short integer in the form <LowByte, HighByte>, 
   from a file and convert them into the current machine's format. */

static UT_Bool readInt16 (FILE *fp, Int16 *s)
{
    UInt8 buf[2];
    if (1 != fread (buf, sizeof(buf), 1, fp)) {
        return readError (fp);
    }
    *s = (buf[0] & 0xFF) | (buf[1] << 8);
    return UT_True;
}

/* Read 4 bytes, representing a floating point number in a machine-independent
   format, from a file and convert them into the current machine's format. */

static UT_Bool readFloat32 (FILE *fp, Float32 *f)
{
    UInt8 buf[4];
    if (1 != fread (buf, sizeof(buf), 1, fp)) {
        return readError (fp);
    }
    *f = UT_PortByteToFloat32 (buf);
    return UT_True;
}

/* Write a byte, representing an unsigned integer to a file. */

static UT_Bool writeUInt8 (FILE *fp, UInt8 b)
{
    UInt8 buf[1];
    buf[0] = b;
    if (1 != fwrite (buf, sizeof(buf), 1, fp)) {
        return writeError ();
    }
    return UT_True;
}

/* Write a byte, representing a signed integer to a file. */

static UT_Bool writeInt8 (FILE *fp, Int8 b)
{
    Int8 buf[1];
    buf[0] = b;
    if (1 != fwrite (buf, sizeof(buf), 1, fp)) {
        return writeError ();
    }
    return UT_True;
}

/* Convert a short integer number into the format <LowByte, HighByte> (an array
   of 2 bytes) and write the array to a file. */

static UT_Bool writeInt16 (FILE *fp, Int16 s)
{
    Int8 buf[2];
    buf[0] = s;
    buf[1] = s >> 8;
    if (1 != fwrite (buf, sizeof(buf), 1, fp)) {
        return writeError ();
    }
    return UT_True;
}

/* Convert a floating point number into a machine-independent format (an array
   of 4 bytes) and write the array to a file. */

static UT_Bool writeFloat32 (FILE *fp, Float32 f)
{
    UInt8 buf[4];
    UT_PortFloat32ToByte (f, buf);
    if (1 != fwrite (buf, sizeof(buf), 1, fp)) {
        return writeError ();
    }
    return UT_True;
}

/* Read the id data, which is located between file header and pixel data.
   We use this - up to 256 byte long - space for internal storage of gamma
   factor, aspect ratio and white point.
   If the information read is not useful, we skip the id_data. */

static UT_Bool readIdData (FILE *fp, FF_ImgFileDesc *px_f)
{
    Int16 magic;

    if (!readInt16 (fp, &magic) ||
        magic != TGA_INT_MAGIC) {
        return UT_False;
    }
    if (!readFloat32 (fp, &px_f->desc.gamma) ||
        !readFloat32 (fp, &px_f->desc.aspect) ||
        !readFloat32 (fp, &px_f->desc.red.x) ||
        !readFloat32 (fp, &px_f->desc.red.y) ||
        !readFloat32 (fp, &px_f->desc.green.x) ||
        !readFloat32 (fp, &px_f->desc.green.y) ||
        !readFloat32 (fp, &px_f->desc.blue.x) ||
        !readFloat32 (fp, &px_f->desc.blue.y) ||
        !readFloat32 (fp, &px_f->desc.white.x) ||
        !readFloat32 (fp, &px_f->desc.white.y) ||
        px_f->desc.gamma < 0.0  ||
        px_f->desc.aspect < 0.0 ||
        px_f->desc.red.x < 0.0 || px_f->desc.red.x > 1.0 ||
        px_f->desc.red.y < 0.0 || px_f->desc.red.y > 1.0 ||
        px_f->desc.green.x < 0.0 || px_f->desc.green.x > 1.0 ||
        px_f->desc.green.y < 0.0 || px_f->desc.green.y > 1.0 ||
        px_f->desc.blue.x < 0.0 || px_f->desc.blue.x > 1.0 ||
        px_f->desc.blue.y < 0.0 || px_f->desc.blue.y > 1.0 ||
        px_f->desc.white.x < 0.0 || px_f->desc.white.x > 1.0 ||
        px_f->desc.white.y < 0.0 || px_f->desc.white.y > 1.0) {
        return UT_False;
    }
    return UT_True;
}

/* Write the id data */

static UT_Bool writeIdData (FILE *fp, FF_ImgFileDesc *px_f)
{
    if (!writeInt16   (fp, TGA_INT_MAGIC) ||
        !writeFloat32 (fp, px_f->desc.gamma) ||
        !writeFloat32 (fp, px_f->desc.aspect) ||
        !writeFloat32 (fp, px_f->desc.red.x) ||
        !writeFloat32 (fp, px_f->desc.red.y) ||
        !writeFloat32 (fp, px_f->desc.green.x) ||
        !writeFloat32 (fp, px_f->desc.green.y) ||
        !writeFloat32 (fp, px_f->desc.blue.x) ||
        !writeFloat32 (fp, px_f->desc.blue.y) ||
        !writeFloat32 (fp, px_f->desc.white.x) ||
        !writeFloat32 (fp, px_f->desc.white.y)) {
        return UT_False;
    }
    return UT_True;
}

/* Write the Targa pixel file header with additional
   infomation into the id_data field. */

static UT_Bool writeHeader (FF_ImgFileDesc *px_f, TGAHEADER *th,
                            FF_ImgComprType compression, UT_Bool withAlpha)
{
    TGAFILE *tgaf;

    tgaf = ((pixfile *)px_f->pixprivate)->fmtprivate;
    th->numid = TGA_ID_DATA_LEN; 
    th->maptyp = 0;
    th->imgtyp = (compression == FF_ImgComprTypeNone? TGA_RGB_UNCOMP: TGA_RGB_COMP);
    th->maporig = 0;
    th->mapsize = 0;
    th->mapbits = 0;
    th->xorig = 0;
    th->yorig = 0;
    th->xsize = px_f->desc.width;
    th->ysize = px_f->desc.height;
    th->pixsize = px_f->channel[FF_ImgChanTypeMatte] && withAlpha? 32: 24;
    th->imgdes = 0;
    if (px_f->scanorder == FF_ImgScanOrderTypeTopDown) {
        th->imgdes = (1 << 5);
    }

    if (!writeUInt8 (tgaf->fp, th->numid) ||
        !writeUInt8 (tgaf->fp, th->maptyp) ||
        !writeUInt8 (tgaf->fp, th->imgtyp) ||
        !writeInt16 (tgaf->fp, th->maporig) ||
        !writeInt16 (tgaf->fp, th->mapsize) ||
        !writeUInt8 (tgaf->fp, th->mapbits) ||
        !writeInt16 (tgaf->fp, th->xorig) ||
        !writeInt16 (tgaf->fp, th->yorig) ||
        !writeInt16 (tgaf->fp, th->xsize) ||
        !writeInt16 (tgaf->fp, th->ysize) ||
        !writeUInt8 (tgaf->fp, th->pixsize) ||
        !writeUInt8 (tgaf->fp, th->imgdes)) {
        return UT_False;
    }
    if (!writeIdData (tgaf->fp, px_f)) {
        return UT_False;
    }
    return UT_True;
}

static UT_Bool readHeader (FILE *fp, const char *name, FF_ImgFileDesc *px_f, TGAHEADER *th, FF_ImgComprType *compression)
{
    Int32 chan;
    UT_Bool havePoInfo = UT_False;

    if (!readUInt8 (fp, &th->numid) ||
        !readUInt8 (fp, &th->maptyp) ||
        !readUInt8 (fp, &th->imgtyp) ||
        !readInt16 (fp, &th->maporig) ||
        !readInt16 (fp, &th->mapsize) ||
        !readUInt8 (fp, &th->mapbits) ||
        !readInt16 (fp, &th->xorig) ||
        !readInt16 (fp, &th->yorig) ||
        !readInt16 (fp, &th->xsize) ||
        !readInt16 (fp, &th->ysize) ||
        !readUInt8 (fp, &th->pixsize) ||
        !readUInt8 (fp, &th->imgdes)) {
        return UT_False;
    }

    /* Try to find out if this file can possibly be a Targa pixel file. */
    if (!((th->imgtyp == TGA_RGB_UNCOMP || th->imgtyp == TGA_RGB_COMP) &&
          (th->pixsize == 24 || th->pixsize == 32))) {
        UT_ErrSetNum (FF_ImgErrFormat, str_open_read, name);
        return UT_False;
    }

    /* If there is additional information in the id_data field, try to read it.
       If it could not be interpreted correctly, skip that data. */
    if (th->numid > 0) {
        if (th->numid != TGA_ID_DATA_LEN || !readIdData (fp, px_f)) {  
            rewind (fp);
            fseek (fp, TGA_HDR_LEN + th->numid, 0);
            havePoInfo = UT_False;
        } else {
            havePoInfo = UT_True;
        }
    }

    /* If no extended poSoft information is available,
       fill the image descriptor with default values. */
    if (!havePoInfo) {
        px_f->desc.red.x   = 0.670f;
        px_f->desc.red.y   = 0.330f;
        px_f->desc.green.x = 0.210f;
        px_f->desc.green.y = 0.710f;
        px_f->desc.blue.x  = 0.140f;
        px_f->desc.blue.y  = 0.080f;
        px_f->desc.white.x = 0.313f;
        px_f->desc.white.y = 0.329f;
        px_f->desc.aspect = (Float32)th->xsize / (Float32)th->ysize;      
        px_f->desc.gamma  = 1.0;
    }

    if (th->xsize < 1 || th->ysize < 1) {
        UT_ErrSetNum (FF_ImgErrSyntax, str_header);
        return UT_False;
    }

    px_f->scanorder = ENC_TOP_DOWN(th->imgdes)?  FF_ImgScanOrderTypeTopDown: FF_ImgScanOrderTypeBottomUp;
    leftToRight = ENC_LEFT_RIGHT(th->imgdes);
    px_f->desc.width  = th->xsize;
    px_f->desc.height = th->ysize;
    for (chan=0; chan<FF_NumImgChanTypes; chan++) {
        switch (chan) {
            case FF_ImgChanTypeRed:
            case FF_ImgChanTypeGreen:
            case FF_ImgChanTypeBlue: {
                px_f->channel[chan] = FF_ImgFmtTypeUByte;
                break;
            }
            case FF_ImgChanTypeMatte: {
                if (th->pixsize == 32) {
                    px_f->channel[chan] = FF_ImgFmtTypeUByte;
                    break;
                }
            }
            default: {
                px_f->channel[chan] = FF_ImgFmtTypeNone;
                break;
            }
        }
    }
    /* Skip colormap data, if present. */
    if (th->mapsize > 0) {
        Int32   mapbytes;
        UInt8 dummy;
        switch (th->mapbits) {
            case 15:
            case 16: {
                mapbytes = 2 * th->mapsize;
                break;
            }
            case 24: {
                mapbytes = 3 * th->mapsize;
                break;
            }
            case 32: {
                mapbytes = 4 * th->mapsize;
                break;
            }
            default: {
                UT_ErrSetNum (FF_ImgErrSyntax, str_colormap, th->mapbits);
                return UT_False;
            }
        }
        while (mapbytes--) {
            if (!readUInt8 (fp, &dummy)) {
                return UT_False;
            }
        }
    }
    *compression = (th->imgtyp == TGA_RGB_UNCOMP? FF_ImgComprTypeNone: FF_ImgComprTypeRle);
    return UT_True;
}

static UT_Bool writePixel (FILE *fp, UInt8 b, UInt8 g, UInt8 r,
                           UInt8 m, Int32 nchan)
{
    UInt8 buf[4];
    buf[0] = b;
    buf[1] = g;
    buf[2] = r;
    buf[3] = m;
    if (nchan != fwrite (buf, sizeof (buf[0]), nchan, fp)) {
        return writeError ();
    }
    return UT_True;
}

/* Write a scanline of channel "type". */

static void writeChannel (const FF_ImgFileDesc *px_f, const pixfile *pixf, 
                          FF_ImgChanType chantype, UInt8 *dest, UInt8 *stop)
{
    switch (px_f->channel[chantype]) {
        case FF_ImgFmtTypeUByte: {
            UInt8 *src = pixf->buf[chantype].ubytes;
            while (dest < stop) {
                *(dest++) = *(src)++;
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            Float32 *src = pixf->buf[chantype].floats;
            while (dest < stop) {
                *(dest++) = (UInt32)(*(src++) * 255.0);
            }
            break;
        }
        default: {
            while (dest < stop) {
                *(dest++) = 0;
            }
        }
    }
    return;
}

/* All channels of scan line number y for pixel file "file" are stored in the file.
   Targa files are written by default in FF_ImgScanOrderTypeBottomUp order. */

static UT_Bool tgaWriteScan (FF_ImgFileDesc *px_f, Int32 y)
{
    pixfile *pixf;
    TGAFILE *tgaf;
    UInt8 *dest;
    UInt8 *redStop, *greenStop, *blueStop, *matteStop;
    UInt8 *redEnd,  *greenEnd,  *blueEnd,  *matteEnd;
    Int32 chansToWrite;

    pixf = px_f->pixprivate;
    tgaf = pixf->fmtprivate;

    /* Collect the data for the red, green, blue and matte channels 
       and store them in the intermediate buffers. */
    dest = tgaf->red;
    redStop = dest + px_f->desc.width;
    writeChannel (px_f, pixf, FF_ImgChanTypeRed, dest, redStop);

    dest = tgaf->green;
    greenStop = dest + px_f->desc.width;
    writeChannel (px_f, pixf, FF_ImgChanTypeGreen, dest, greenStop);

    dest = tgaf->blue;
    blueStop = dest + px_f->desc.width;
    writeChannel (px_f, pixf, FF_ImgChanTypeBlue, dest, blueStop);

    if (tgaf->nchan == 4 && tgaf->withAlpha) {
        dest = tgaf->matte;
        matteStop = dest + px_f->desc.width;
        writeChannel (px_f, pixf, FF_ImgChanTypeMatte, dest, matteStop);
    }

    red   = tgaf->red;
    green = tgaf->green;
    blue  = tgaf->blue;
    matte = tgaf->matte;
    redStop   = red   + px_f->desc.width;
    greenStop = green + px_f->desc.width;
    blueStop  = blue  + px_f->desc.width;
    matteStop = matte + px_f->desc.width;

    chansToWrite = tgaf->nchan;
    if (tgaf->nchan == 4 && ! tgaf->withAlpha) {
        chansToWrite = 3;
    }

    /* Write the scanline data to the file. */
    if (tgaf->compression == FF_ImgComprTypeNone) {
        while (red < redStop) {
            if (!writePixel (tgaf->fp, *blue, *green, *red, *matte, chansToWrite)) {
                return UT_False;
            }
            blue++;
            green++;
            red++;
            matte++;
        }
    } else {
        /* Run-length Compression */
        redEnd   = red + 1;
        greenEnd = green + 1;
        blueEnd  = blue + 1;
        matteEnd = matte + 1;
        while (red < redStop) {
            while (redEnd < redStop &&
                   *red   == *redEnd &&
                   *green == *greenEnd &&
                   *blue  == *blueEnd &&
                   redEnd - red - 1 < MAXRUN) {
                if (tgaf->nchan == 4 && tgaf->withAlpha) {
                    if (*matte != *matteEnd) {
                        break;
                    }
                }
                redEnd++;
                greenEnd++;
                blueEnd++;
                matteEnd++;
            }
            if (redEnd - red >= MINRUN) {   /* Found a run of compressable data */
                if (!writeInt8 (tgaf->fp, (Int8)(((redEnd - red)-1)|0x80)) ||
                    !writePixel (tgaf->fp, *blue, *green, *red, *matte, chansToWrite)) {
                    return UT_False;
                }
                red = redEnd;
                green = greenEnd;
                blue = blueEnd;
                matte = matteEnd;
            } else {
                /* Found a run of uncompressable data */
                while (redEnd < redStop &&
                       ((redEnd + 1 >= redStop ||
                        *redEnd != *(redEnd + 1)) ||
                        (redEnd + 2 >= redStop ||
                        *(redEnd + 1) != *(redEnd + 2))) &&
                       ((greenEnd + 1 >= greenStop ||
                        *greenEnd != *(greenEnd + 1)) ||
                        (greenEnd + 2 >= greenStop ||
                        *(greenEnd + 1) != *(greenEnd + 2))) &&
                       ((blueEnd + 1 >= blueStop ||
                        *blueEnd != *(blueEnd + 1)) ||
                        (blueEnd + 2 >= blueStop ||
                        *(blueEnd + 1) != *(blueEnd + 2))) &&
                        redEnd - red < MAXRUN) {
                    if (tgaf->nchan == 4 && tgaf->withAlpha) {
                        if (! ((matteEnd + 1 >= matteStop ||
                               *matteEnd != *(matteEnd + 1)) ||
                               (matteEnd + 2 >= matteStop ||
                               *(matteEnd + 1) != *(matteEnd + 2)))) {
                            break;
                        }
                    }
                    redEnd++;
                    greenEnd++;
                    blueEnd++;
                    matteEnd++;
                }
                if (!writeInt8 (tgaf->fp, (Int8)((redEnd - red) - 1))) {
                    return UT_False;
                }
                while (red < redEnd) {
                    if (!writePixel (tgaf->fp, *blue, *green, *red, *matte, chansToWrite)) {
                        return UT_False;
                    }
                    red++;
                    green++;
                    blue++;
                    matte++;
                }
            }
            redEnd++;
            greenEnd++;
            blueEnd++;
            matteEnd++;
        }
    }
    tgaf->lastscan = y;
    return UT_True;
}

static void readChannel (const FF_ImgFileDesc *px_f, const pixfile *pixf,
                         FF_ImgChanType chantype, const UInt8 *src, Int32 n)
{
    const UInt8 *stop = src + n;
    if (leftToRight) {
        switch (px_f->channel[chantype]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                UInt8 *dest = pixf->buf[chantype].ubytes;
                while (src < stop) {
                    *(dest++) = *(src++);
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                Float32 *dest = pixf->buf[chantype].floats;
                while (src < stop) {
                    *(dest++) = (Float32)(UInt32)*(src++) * (1.0 / 255.0);
                }
                break;
            }
        }
    } else {
        switch (px_f->channel[chantype]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                UInt8 *dest = pixf->buf[chantype].ubytes + n - 1;
                while (src < stop) {
                    *(dest--) = *(src++);
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                Float32 *dest = pixf->buf[chantype].floats + n - 1;
                while (src < stop) {
                    *(dest--) = (Float32)(UInt32)*(src++) * (1.0 / 255.0);
                }
                break;
            }
        }
    }
    return;
}

/* Read one pixel value from file "fp". 
   A pixel is represented by 3 or 4 bytes in the order Blue/Green/Red/Matte. 
   The pixel value should be repeated in the scanline "n" times.
   Note that Targa allows pixel values to be compressed across scanline
   boundaries. */

static UT_Bool readPixels (UInt8 *stop, Int32 n, TGAFILE *tgaf)
{
    Int32 i;

    if (tgaf->nchan != fread (pixbuf, sizeof (pixbuf[0]), 
                              tgaf->nchan, tgaf->fp)) {
        return readError (tgaf->fp);
    }
    for (i=0; i<n; i++) {
        FROM_PIXBUF;

        if (red == stop) {
            /* Scanline is filled with pixel values.
               Determine the number of pixels to keep for next scanline. */
            tgaf->scanrest = n - i - 1;
            return UT_True;
        }
    }
    return UT_True;
}

/* The channels of scan line number "y" of the pixel file "px_f" are read.
   Note that the Targa file format supports both FF_ImgScanOrderTypeTopDown
   and FF_ImgScanOrderTypeBottomUp scanline order. */

static UT_Bool tgaReadScan (FF_ImgFileDesc *px_f, Int32 y)
{
    pixfile *pixf;
    TGAFILE *tgaf;
    UInt8   *stop;

    pixf = px_f->pixprivate;
    tgaf = pixf->fmtprivate;

    /* Read and decode the scan line data and store them in an
    intermediate buffer. */
    rowbuf = tgaf->rowbuf;
    red    = tgaf->red;
    green  = tgaf->green;
    blue   = tgaf->blue;
    matte  = tgaf->matte;
    stop = red + px_f->desc.width;

    if (tgaf->compression == FF_ImgComprTypeRle) {
        Int8 cbuf[1];
        Int32  pix, numpix;
        /* While there are pixels left from the previous scanline,
           either fill the current scanline with the pixel value
           still stored in "pixbuf" (TGA_MODE_SAME) or read in the
           appropriate number of pixel values (TGA_MODE_DIFF). */
        while (tgaf->scanrest) {
            if (tgaf->scanmode == TGA_MODE_DIFF) {   
                if (tgaf->nchan != fread (pixbuf, sizeof (pixbuf[0]), 
                                        tgaf->nchan, tgaf->fp)) {
                    return readError (tgaf->fp);
                }
            }
            FROM_PIXBUF;
            tgaf->scanrest--;
            /* If the image is small, the compression might go over several
               scanlines. */
            if (red == stop) {
                goto read_scanline;
            }
        }

        /* Read the byte telling us the compression mode and the compression
           count. Then read the pixel values till a scanline is filled. */
        do {
            if (1 != fread (cbuf, sizeof (cbuf), 1, tgaf->fp)) {
                return readError (tgaf->fp);
            }
            numpix = (cbuf[0] & 0x7F) + 1;

            if ((cbuf[0] & 0x80) != 0x80) {
                tgaf->scanmode = TGA_MODE_DIFF;
                for (pix=0; pix<numpix; pix++) {
                    if (!readPixels (stop, 1, tgaf)) {
                        return UT_False;
                    }
                    if (red == stop) {
                        tgaf->scanrest = numpix - pix - 1;
                        break;
                    }
                }
            } else {
                tgaf->scanmode = TGA_MODE_SAME;
                if (!readPixels (stop, numpix, tgaf)) {
                    return UT_False;
                }
            }
        } while (red < stop);
    } else {
        /* Read uncompressed pixel data. */
        Int32 i, bytesPerLine;

        bytesPerLine =  tgaf->nchan * px_f->desc.width;
        if (bytesPerLine != fread (rowbuf, 1, bytesPerLine, tgaf->fp)) {
            return readError (tgaf->fp);
        }
        for (i=0; i<px_f->desc.width; i++) {
            *(blue++)  = *(rowbuf++);
            *(green++) = *(rowbuf++);
            *(red++)   = *(rowbuf++);
            if (tgaf->nchan == 4) {
                *(matte++) = *(rowbuf++);
            }
        }
    }

read_scanline:
    /* Copy the data to their final destination. */
    readChannel (px_f, pixf, FF_ImgChanTypeRed,   tgaf->red,   px_f->desc.width);
    readChannel (px_f, pixf, FF_ImgChanTypeGreen, tgaf->green, px_f->desc.width);
    readChannel (px_f, pixf, FF_ImgChanTypeBlue,  tgaf->blue,  px_f->desc.width);

    if (tgaf->nchan == 4 && tgaf->withAlpha) {
        readChannel (px_f, pixf, FF_ImgChanTypeMatte, tgaf->matte, px_f->desc.width);
    }

    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenWriteTarga
 *
 *      Usage:          Open a Targa pixel file for writing.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenWriteTarga
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f,
 *                               const char *opts)
 *
 *      Description:    Targa pixel file "name" is opened for writing.
 *
 *                      Only FF_ImgChanTypeRed, FF_ImgChanTypeGreen,
 *                      FF_ImgChanTypeBlue and FF_ImgChanTypeMatte channel
 *                      types are supported.
 *                      Since the Targa file format requires, that all 
 *                      three/four supported channels are stored in the file,
 *                      unused channels in the image are filled with zeros.
 *                      Targa files are written in FF_ImgScanOrderTypeBottomUp
 *                      scan order. The scan order may be changed by option
 *                      "-scanorder" using "BottomUp" or "TopDown" as option values.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenReadTarga
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenWriteTarga (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    pixfile *pixf;
    TGAFILE *tgaf;
    TGAHEADER tgaHeader;
    UT_MemState memstate;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;
    FF_ImgScanOrderType optScanOrder;
    FF_ImgComprType optCompression;

    memstate = UT_MemRemember ();

    /* Allocate memory for a "TGAFILE". */
    if (!(tgaf =         UT_MemTemp (TGAFILE)) ||
        !(tgaf->rowbuf = UT_MemTempArray (px_f->desc.width*4, UInt8)) ||
        !(tgaf->red =    UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->green =  UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->blue =   UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->matte =  UT_MemTempArray (px_f->desc.width, UInt8))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    /* Open the file for writing. */
    if (!(tgaf->fp = UT_FileOpen (name, "wb"))) {
        UT_MemRestore (memstate);
        return UT_False;
    }
    pixf = px_f->pixprivate;
    pixf->fmtprivate = tgaf;

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default scan order and check options for -scanorder key. */
    optScanOrder = FF_ImgScanOrderTypeBottomUp;
    FF_ImgGetOptScanOrder (opts, &optScanOrder);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);

    /* Set default compression value and check options of -compression key. */
    optCompression = FF_ImgComprTypeNone;
    FF_ImgGetOptCompression (opts, &optCompression);

    px_f->scanorder = optScanOrder;

    /* Generate the file header and write it to the file. */
    if (!writeHeader (px_f, &tgaHeader, optCompression, optWithAlpha)) {
        return UT_False;
    }

    tgaf->lastscan    = -1;
    tgaf->nchan       = px_f->channel[FF_ImgChanTypeMatte]? 4: 3;
    tgaf->withAlpha   = optWithAlpha;
    tgaf->compression = optCompression;

    pixf->writescan = tgaWriteScan;
    pixf->close     = tgaClose;

    UT_MemSave (memstate);

    if (optVerbose) {
         printImgInfo (&tgaHeader, name, "Saving image:");
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           FF_ImgOpenReadTarga
 *
 *      Usage:          Open a Targa pixel file for reading.
 *
 *      Synopsis:       UT_Bool FF_ImgOpenReadTarga
 *                              (const char *name,
 *                               FF_ImgFileDesc *px_f,
 *                               const char *opts)
 *
 *      Description:    Targa pixel file "name" is opened for reading.
 *
 *                      Only FF_ImgChanTypeRed, FF_ImgChanTypeGreen,
 *                      FF_ImgChanTypeBlue and FF_ImgChanTypeMatte channel
 *                      types are supported.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       FF_ImgOpenWriteTarga
 *
 ***************************************************************************/

UT_Bool FF_ImgOpenReadTarga (const char *name, FF_ImgFileDesc *px_f, const char *opts)
{
    pixfile *pixf;
    TGAFILE *tgaf;
    TGAHEADER tgaHeader;
    FILE *fp;
    FF_ImgComprType optCompression;
    UT_Bool optWithAlpha;
    UT_Bool optVerbose;
    UT_MemState memstate;

    memstate = UT_MemRemember ();

    /* Set default verbose mode and check options for -verbose key. */
    optVerbose = UT_False;
    FF_ImgGetOptVerbose (opts, &optVerbose);

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (opts, &optWithAlpha);

    pixf = px_f->pixprivate;
    /* Open the file for reading. */
    if (!(fp = UT_FileOpen (name, "rb"))) {
        return UT_False;
    }

    if (!readHeader (fp, name, px_f, &tgaHeader, &optCompression)) {
        UT_FileClose (fp);
        return UT_False;
    }

    /* Generate the "TGAFILE". */
    if (!(tgaf =         UT_MemTemp (TGAFILE)) ||
        !(tgaf->rowbuf = UT_MemTempArray (px_f->desc.width*4, UInt8)) ||
        !(tgaf->red =    UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->green =  UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->blue =   UT_MemTempArray (px_f->desc.width, UInt8)) ||
        !(tgaf->matte =  UT_MemTempArray (px_f->desc.width, UInt8))) {
        UT_MemRestore (memstate);
        UT_FileClose (fp);
        return UT_False;
    }

    /* Fill the "pixfile". */
    pixf->fmtprivate = tgaf;
    pixf->readscan   = tgaReadScan;
    pixf->close      = tgaClose;

    /* Fill the "TGAFILE". */
    tgaf->lastscan = -1;
    tgaf->nchan = px_f->channel[FF_ImgChanTypeMatte]? 4: 3;
    tgaf->scanrest = 0;
    tgaf->fp = fp;
    tgaf->compression = optCompression;
    tgaf->withAlpha   = optWithAlpha;

    UT_MemSave (memstate);

    if (optVerbose) {
         printImgInfo (&tgaHeader, name, "Reading image:");
    }
    return UT_True;
}
