/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IPT_TclIfSingle.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The interface between the image
 *                      library and the Tcl interpreter.
 *                      This file contains all non-thread-aware functions.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"
#include "UT_Macros.h"

#include "UT_Error.h"
#include "UT_Memory.h"
#include "FF_Image.h"
#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#include <tcl.h>
#include "UTT_TclIf.h"
#include "IPT_TclIf.h"
#include "IPT_ImagePrivate.h"

/*>@u
        1. General information:
        =======================

        The following symbolic names and their numerical values 
        can be used as aequivalents:

        NONE    UT_False 0
        OFF     UT_False 0
        ON      UT_True  1

        A value of type "boolean" can be specified with one of these values.

        2. Channel type:
        ================

        There are currently 13 possible channel types.
        Their Tcl symbolic names as well as their 
        numerical values are listed below:

        BRIGHTNESS       0      Brightness for black and white images
        RED              1      Red for color images
        GREEN            2      Green for color images
        BLUE             3      Blue for color images
        MATTE            4      Transparency without color information
        REDMATTE         5      Transparency for red channel
        GREENMATTE       6      Transparency for green channel
        BLUEMATTE        7      Transparency for blue channel
        HNORMAL          8      Direction of horizontal surface normal
        VNORMAL          9      Direction of vertical surface normal
        DEPTH           10      Distance from the eye point
        TEMPERATURE     11      Temperature in Kelvin
        RADIANCE        12      Radiance in Watt/SquareRadian
        NUMCHAN         13      Number of different channel types

        3. Channel format:
        ==================

        The data stored in a channel can be in the following different formats:

        NONE             0      Channel is not used
        UBYTE            1      Channel holds 8-bit integer values (i.e. 0..255)
        FLOAT            2      Channel holds 32-bit floating point values 

        4. Image compression:
        =====================

        Channels can be compressed using one of the following methods:

        NONE            0       No compression.
        RLE             1       Lossless compression, e.g. run length encoding

        5. Drawing mode:
        ================

        The following different drawing modes are available:

        REPLACE         0       Replace old color with current color
        ADD             1       Add old color and current color
        SUB             2       Subtract old color from current color
        XOR             3       Exclusive-or old color and current color

<*/

/***************************************************************************
 *[@e
 *      Name:           HasPhotoSupport
 *
 *      Usage:          Check Tk photo support.
 *
 *      Description:    Check, if poImg has been compiled to support
 *                      Tk photos. This is the default mode.
 *
 *                      Support for Tk photos can be disabled at compile
 *                      time by setting configure flag:
 *                      --enable-tkphotos=off
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   If compiled with Tk photo support, return true.
 *                      Otherwise false.
 *
 *      See also:       MemCheck
 *
 ***************************************************************************/

UT_Bool IPT_HasPhotoSupport (TCLPARAMLIST)
{
    Int32 hasPhotoSupport = UT_False;
#ifdef USE_TK_PHOTOS
    hasPhotoSupport = UT_True;
#endif
    UTT_SetFixInt32Result (hasPhotoSupport);

    UTT_ParamHelp ("poImageState HasPhotoSupport bool:return");

    return UT_True;
}

UT_Bool IPT_GetChannelFormat (TCLPARAMLIST)
{
    FF_ImgFmtType channelFormat;
    char channelFormatName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetChannelFormat channelFormatName channelFormat:return");

    UTT_GetFixString (1, channelFormatName, MAXPATH, UT_False);

    if (!IP_GetChannelFormat (channelFormatName, &channelFormat)) {
        UTT_SetFixInt32Result (-1);
    } else {
        UTT_SetFixInt32Result (channelFormat);
    }
    return UT_True;
}

UT_Bool IPT_GetChannelType (TCLPARAMLIST)
{
    FF_ImgChanType channelType;
    char channelTypeName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetChannelType channelTypeName channelType:return");

    UTT_GetFixString (1, channelTypeName, MAXPATH, UT_False);

    if (!IP_GetChannelType (channelTypeName, &channelType)) {
        UTT_SetFixInt32Result (-1);
    } else {
        UTT_SetFixInt32Result (channelType);
    }
    return UT_True;
}

UT_Bool IPT_GetDrawModeType (TCLPARAMLIST)
{
    IP_DrawModeType drawModeType;
    char drawModeTypeName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetDrawModeType drawModeTypeName drawModeType:return");

    UTT_GetFixString (1, drawModeTypeName, MAXPATH, UT_False);

    if (!IP_GetDrawModeType (drawModeTypeName, &drawModeType)) {
        UTT_SetFixInt32Result (-1);
    } else {
        UTT_SetFixInt32Result (drawModeType);
    }
    return UT_True;
}

UT_Bool IPT_GetFillModeType (TCLPARAMLIST)
{
    IP_FillModeType fillModeType;
    char fillModeTypeName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetFillModeType fillModeTypeName fillModeType:return");

    UTT_GetFixString (1, fillModeTypeName, MAXPATH, UT_False);

    if (!IP_GetFillModeType (fillModeTypeName, &fillModeType)) {
        UTT_SetFixInt32Result (-1);
    } else {
        UTT_SetFixInt32Result (fillModeType);
    }
    return UT_True;
}

UT_Bool IPT_GetUnlaceModeType (TCLPARAMLIST)
{
    IP_UnlaceModeType unlaceModeType;
    char unlaceModeTypeName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetUnlaceModeType unlaceModeTypeName unlaceModeType:return");

    UTT_GetFixString (1, unlaceModeTypeName, MAXPATH, UT_False);

    if (!IP_GetUnlaceModeType (unlaceModeTypeName, &unlaceModeType)) {
        UTT_SetFixInt32Result (-1);
    } else {
        UTT_SetFixInt32Result (unlaceModeType);
    }
    return UT_True;
}

UT_Bool IPT_GetChannelFormatName (TCLPARAMLIST)
{
    Int32 channelFormat;
    const char *channelFormatName;

    UTT_ParamHelp ("poImageState GetChannelFormatName channelFormat channelFormatName:return");

    UTT_GetFixInt32 (1, channelFormat);

    channelFormatName = IP_GetChannelFormatName ((FF_ImgFmtType)channelFormat);

    UTT_SetFixStringResult ((char *)channelFormatName);
    return UT_True;
}

UT_Bool IPT_GetChannelTypeName (TCLPARAMLIST)
{
    Int32 channelType;
    const char *channelTypeName;

    UTT_ParamHelp ("poImageState GetChannelTypeName channelType channelTypeName:return");

    UTT_GetFixInt32 (1, channelType);

    channelTypeName = IP_GetChannelTypeName ((FF_ImgChanType)channelType);

    UTT_SetFixStringResult ((char *)channelTypeName);
    return UT_True;
}

UT_Bool IPT_GetDrawModeTypeName (TCLPARAMLIST)
{
    Int32 drawModeType;
    const char *drawModeTypeName;

    UTT_ParamHelp ("poImageState GetDrawModeTypeName drawModeType drawModeTypeName:return");

    UTT_GetFixInt32 (1, drawModeType);

    drawModeTypeName = IP_GetDrawModeTypeName ((IP_DrawModeType)drawModeType);

    UTT_SetFixStringResult ((char *)drawModeTypeName);
    return UT_True;
}

UT_Bool IPT_GetFillModeTypeName (TCLPARAMLIST)
{
    Int32 fillModeType;
    const char *fillModeTypeName;

    UTT_ParamHelp ("poImageState GetFillModeTypeName fillModeType fillModeTypeName:return");

    UTT_GetFixInt32 (1, fillModeType);

    fillModeTypeName = IP_GetFillModeTypeName ((IP_FillModeType)fillModeType);

    UTT_SetFixStringResult ((char *)fillModeTypeName);
    return UT_True;
}

UT_Bool IPT_GetUnlaceModeTypeName (TCLPARAMLIST)
{
    Int32 unlaceModeType;
    const char *unlaceModeTypeName;

    UTT_ParamHelp ("poImageState GetUnlaceModeTypeName unlaceModeType unlaceModeTypeName:return");

    UTT_GetFixInt32 (1, unlaceModeType);

    unlaceModeTypeName = IP_GetUnlaceModeTypeName ((IP_UnlaceModeType)unlaceModeType);

    UTT_SetFixStringResult ((char *)unlaceModeTypeName);
    return UT_True;
}

UT_Bool IPT_SetFormat (TCLPARAMLIST)
{
    Int32 i, n, *formatList;
    Int32 defFmt[FF_NumImgChanTypes];
    FF_ImgFmtType tmp[FF_NumImgChanTypes];
    void *memstate;

    UTT_ParamHelp ("poImageState SetFormat ?formatList?");

    memstate = UT_MemRemember ();
    for (i=0; i<FF_NumImgChanTypes; i++) {
        defFmt[i] = (Int32)IPT_StateFormatDefault[i];
    }
    UTT_GetOptInt32List (1, formatList, n, FF_NumImgChanTypes, defFmt);
    if (n == 0) {
        formatList = defFmt;
    }
    for (i=0; i<FF_NumImgChanTypes; i++) {
        tmp[i] = formatList[i];
    }
    UT_MemRestore (memstate);

    return IP_SetFormat (tmp);
}

UT_Bool IPT_GetFormat (TCLPARAMLIST)
{
    Int32 i, formatList[FF_NumImgChanTypes];
    FF_ImgFmtType tmp[FF_NumImgChanTypes];

    UTT_ParamHelp ("poImageState GetFormat formatList:out");

    IP_GetFormat (tmp);
    for (i=0; i<FF_NumImgChanTypes; i++) {
        formatList[i] = tmp[i];
    }

    UTT_SetFixInt32List (1, formatList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_SetDrawMask (TCLPARAMLIST)
{
    Int32   i, n, *drawMaskList;
    Int32   defMask[FF_NumImgChanTypes];
    UT_Bool tmp[FF_NumImgChanTypes];
    void *memstate;

    UTT_ParamHelp ("poImageState SetDrawMask ?drawMaskList?");

    memstate = UT_MemRemember ();
    for (i=0; i<FF_NumImgChanTypes; i++) {
        defMask[i] = (Int32)IPT_StateDrawMaskDefault[i];
    }
    UTT_GetOptInt32List (1, drawMaskList, n, FF_NumImgChanTypes, defMask);
    if (n == 0) {
        drawMaskList = defMask;
    }
    for (i=0; i<FF_NumImgChanTypes; i++) {
        tmp[i] = drawMaskList[i];
    }
    UT_MemRestore (memstate);

    return IP_SetDrawMask (tmp);
}

UT_Bool IPT_GetDrawMask (TCLPARAMLIST)
{
    Int32   i, drawMaskList[FF_NumImgChanTypes];
    UT_Bool tmp[FF_NumImgChanTypes];

    UTT_ParamHelp ("poImageState GetDrawMask drawMaskList:out");

    IP_GetDrawMask (tmp);

    for (i=0; i<FF_NumImgChanTypes; i++) {
        drawMaskList[i] = tmp[i];
    }

    UTT_SetFixInt32List (1, drawMaskList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_SetDrawColor (TCLPARAMLIST)
{
    Int32   i, n;
    Float32 *drawColorList;
    Float32 defColor[FF_NumImgChanTypes];
    Float32 tmp[FF_NumImgChanTypes];
    void *memstate;

    UTT_ParamHelp ("poImageState SetDrawColor ?drawColorList?");

    memstate = UT_MemRemember ();
    for (i=0; i<FF_NumImgChanTypes; i++) {
        defColor[i] = IPT_StateDrawColorDefault[i];
    }
    UTT_GetOptFloat32List (1, drawColorList, n, FF_NumImgChanTypes, defColor);
    if (n == 0) {
        drawColorList = defColor;
    }
    for (i=0; i<FF_NumImgChanTypes; i++) {
        tmp[i] = drawColorList[i];
    }
    UT_MemRestore (memstate);

    return IP_SetDrawColor (tmp);
}

UT_Bool IPT_GetDrawColor (TCLPARAMLIST)
{
    Float32 drawColorList[FF_NumImgChanTypes];

    UTT_ParamHelp ("poImageState GetDrawColor drawColorList:out");

    IP_GetDrawColor (drawColorList);

    UTT_SetFixFloat32List (1, drawColorList, FF_NumImgChanTypes);

    return UT_True;
}

UT_Bool IPT_SetDrawMode (TCLPARAMLIST)
{
    Int32 i, n, *drawModeList;
    Int32 defMode[FF_NumImgChanTypes];
    IP_DrawModeType tmp[FF_NumImgChanTypes];
    void *memstate;

    UTT_ParamHelp ("poImageState SetDrawMode ?drawModeList?");

    memstate = UT_MemRemember ();
    for (i=0; i<FF_NumImgChanTypes; i++) {
        defMode[i] = (Int32)IPT_StateDrawModeDefault[i];
    }
    UTT_GetOptInt32List (1, drawModeList, n, FF_NumImgChanTypes, defMode);

    if (n == 0) {
        drawModeList = defMode;
    }
    for (i=0; i<FF_NumImgChanTypes; i++) {
        tmp[i] = drawModeList[i];
    }
    UT_MemRestore (memstate);

    return IP_SetDrawMode (tmp);
}

UT_Bool IPT_GetDrawMode (TCLPARAMLIST)
{
    Int32 i, drawModeList[FF_NumImgChanTypes];
    IP_DrawModeType tmp[FF_NumImgChanTypes];

    UTT_ParamHelp ("poImageState GetDrawMode drawModeList:out");

    IP_GetDrawMode (tmp);

    for (i=0; i<FF_NumImgChanTypes; i++) {
        drawModeList[i] = tmp[i];
    }

    UTT_SetFixInt32List (1, drawModeList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_SetColorCorrect (TCLPARAMLIST)
{
    Float64 gamma;
    Float64 rx, ry, gx, gy, bx, by, wx, wy;
    CIE_hue red, green, blue, white;

    UTT_ParamHelp ("poImageState SetColorCorrect ?gamma? ?rx? ?ry? ?gx? ?gy? ?bx? ?by? ?wx? ?wy?");

    UTT_GetOptFloat64 (1, gamma, 1.0);
    UTT_GetOptFloat64 (2, rx,    0.670);
    UTT_GetOptFloat64 (3, ry,    0.330);
    UTT_GetOptFloat64 (4, gx,    0.210);
    UTT_GetOptFloat64 (5, gy,    0.710);
    UTT_GetOptFloat64 (6, bx,    0.140);
    UTT_GetOptFloat64 (7, by,    0.080);
    UTT_GetOptFloat64 (8, wx,    0.313);
    UTT_GetOptFloat64 (9, wy,    0.329);

    red.x   = rx; red.y   = ry;
    green.x = gx; green.y = gy;
    blue.x  = bx; blue.y  = by;
    white.x = wx; white.y = wy;

    return IP_SetColorCorrect (gamma, &red, &green, &blue, &white);
}

UT_Bool IPT_GetColorCorrect (TCLPARAMLIST)
{
    Float32 gamma;
    CIE_hue red, green, blue, white;

    UTT_ParamHelp ("poImageState GetColorCorrect gamma:out ?rx:out? ?ry:out? ?gx:out? ?gy:out? ?bx:out? ?by:out? ?wx:out? ?wy:out?");

    IP_GetColorCorrect (&gamma, &red, &green, &blue, &white);

    UTT_SetFixFloat64 (1, gamma);
    UTT_SetOptFloat64 (2, red.x);
    UTT_SetOptFloat64 (3, red.y);
    UTT_SetOptFloat64 (4, green.x);
    UTT_SetOptFloat64 (5, green.y);
    UTT_SetOptFloat64 (6, blue.x);
    UTT_SetOptFloat64 (7, blue.y);
    UTT_SetOptFloat64 (8, white.x);
    UTT_SetOptFloat64 (9, white.y);
    return UT_True;
}

UT_Bool IPT_PushState (TCLPARAMLIST)
{
    UTT_ParamHelp ("poImageState PushState stackIndex:return");

    UTT_SetFixInt32Result (IP_PushState ());
    return UT_True;
}

UT_Bool IPT_PopState (TCLPARAMLIST)
{
    UTT_ParamHelp ("poImageState PopState stackIndex:return");

    UTT_SetFixInt32Result (IP_PopState ());
    return UT_True;
}

UT_Bool IPT_SetNumThreads (TCLPARAMLIST)
{
    Int32 numThreads;

    UTT_ParamHelp ("poImageState SetNumThreads numThreads");

    UTT_GetFixInt32 (1, numThreads);

    return IP_SetNumThreads (numThreads);
}

UT_Bool IPT_GetNumThreads (TCLPARAMLIST)
{
    UTT_SetFixInt32Result (IP_GetNumThreads ());

    UTT_ParamHelp ("poImageState GetNumThreads numThreads:return");

    return UT_True;
}


/***************************************************************************
 *[@e
 *      Name:           MemCheck
 *
 *      Usage:          Check memory usage.
 *
 *      Description:    Check, if all memory allocated by the poImg library
 *                      has been freed.
 *
 *                      More detailed information regarding allocated memory
 *                      blocks can enabled at compile time by setting configure flag:
 *                      --enable-memdebug=on
 *                      The information is printed onto stdout.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   If all memory has been freed, return true.
 *                      Otherwise false.
 *
 *      See also:       HasPhotoSupport
 *
 ***************************************************************************/

UT_Bool IPT_MemCheck (TCLPARAMLIST)
{
    UTT_ParamHelp ("poImageState MemCheck bool:return");

    UTT_SetFixInt32Result (UT_MemCheck ());

    return UT_True;
}

UT_Bool IPT_NewImage (OBJPARAMLIST)
{
    IP_ImageId img;
    Int32 w, h;
    Float64 asp;

    UTT_ParamHelp ("poImage NewImage width height ?aspect? poImg:return");

    UTT_GetFixInt32   (1, w);
    UTT_GetFixInt32   (2, h);
    UTT_GetOptFloat64 (3, asp, (Float64)w / (Float64)h);

    if (!(img = IP_NewImage (w, h, asp, NULL))) {
        return UT_False;
    }

    *clientData = img;
    return UT_True;
}

UT_Bool IPT_HasChannel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    UT_Bool hasChannel;

    UTT_ParamHelp ("$img HasChannel channel bool:return");

    IPT_GetFixChannelEnum (1, channel);

    hasChannel = IP_HasChannel (img, channel);
    UTT_SetFixInt32Result (hasChannel);

    return UT_True;
}

UT_Bool IPT_GetImageInfo (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgHdr desc;
    Int32 width, height;

    UTT_ParamHelp ("$img GetImageInfo width:out height:out ?aspect:out? ?gamma:out? ?rx:out? ?ry:out? ?gx:out? ?gy:out? ?bx:out? ?by:out? ?wx:out? ?wy:out?");

    IP_GetImageInfo (img, &desc);
    /* Only needed to confirm to ParamHelp names. */
    width  = desc.width;
    height = desc.height;

    UTT_SetFixInt32    (1, width);
    UTT_SetFixInt32    (2, height);
    UTT_SetOptFloat64  (3, desc.aspect);
    UTT_SetOptFloat64  (4, desc.gamma);
    UTT_SetOptFloat64  (5, desc.red.x);
    UTT_SetOptFloat64  (6, desc.red.y);
    UTT_SetOptFloat64  (7, desc.green.x);
    UTT_SetOptFloat64  (8, desc.green.y);
    UTT_SetOptFloat64  (9, desc.blue.x);
    UTT_SetOptFloat64 (10, desc.blue.y);
    UTT_SetOptFloat64 (11, desc.white.x);
    UTT_SetOptFloat64 (12, desc.white.y);

    return UT_True;
}

UT_Bool IPT_GetImageFormat (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 i, formatList[FF_NumImgChanTypes];
    FF_ImgFmtType tmp[FF_NumImgChanTypes];

    UTT_ParamHelp ("$img GetImageFormat formatList:out");

    IP_GetImageFormat (img, tmp);

    for (i=0; i<FF_NumImgChanTypes; i++) {
        formatList[i] = tmp[i];
    }

    UTT_SetFixInt32List (1, formatList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_WriteImage (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    char fileName[MAXPATH + 1], format[Tmpsize + 1], options[MAXPATH + 1];

    UTT_ParamHelp ("$img WriteImage fileName format ?options?");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);
    UTT_GetFixString (2, format,   Tmpsize, UT_False);
    UTT_GetOptString (3, options,  MAXPATH, UT_False, "");

    return IP_WriteImage (img, fileName, format, options);
}

UT_Bool IPT_ReadImage (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    char fileName[MAXPATH + 1], options[MAXPATH + 1];

    UTT_ParamHelp ("$img ReadImage fileName ?options?");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);
    UTT_GetOptString (2, options,  MAXPATH, UT_False, "");

    return IP_ReadImage (img, fileName, options);
}

UT_Bool IPT_GetFileInfo (TCLPARAMLIST)
{
    char      fileName[MAXPATH + 1];
    FF_ImgHdr desc;
    Int32     width, height;
    Float64   aspect, gamma, rx, ry, gx, gy, bx, by, wx, wy;

    UTT_ParamHelp ("poImageState GetFileInfo fileName width:out height:out ?aspect:out? ?gamma:out? ?rx:out? ?ry:out? ?gx:out? ?gy:out? ?bx:out? ?by:out? ?wx:out? ?wy:out?");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);

    if (!IP_GetFileInfo (fileName, &desc)) {
        return UT_False;
    }

    /* Copying needed for correct error message output. */
    width  = desc.width;
    height = desc.height;
    aspect = desc.aspect;
    gamma  = desc.gamma;
    rx     = desc.red.x;
    ry     = desc.red.y;
    gx     = desc.green.x;
    gy     = desc.green.y;
    bx     = desc.blue.x;
    by     = desc.blue.y;
    wx     = desc.white.x;
    wy     = desc.white.y;

    UTT_SetFixInt32   ( 2, width);
    UTT_SetFixInt32   ( 3, height);
    UTT_SetOptFloat64 ( 4, aspect);
    UTT_SetOptFloat64 ( 5, gamma);
    UTT_SetOptFloat64 ( 6, rx);
    UTT_SetOptFloat64 ( 7, ry);
    UTT_SetOptFloat64 ( 8, gx);
    UTT_SetOptFloat64 ( 9, gy);
    UTT_SetOptFloat64 (10, bx);
    UTT_SetOptFloat64 (11, by);
    UTT_SetOptFloat64 (12, wx);
    UTT_SetOptFloat64 (13, wy);

    return UT_True;
}

UT_Bool IPT_GetFileFormat (TCLPARAMLIST)
{
    Int32 i, formatList[FF_NumImgChanTypes];
    FF_ImgFmtType tmp[FF_NumImgChanTypes];
    char fileName[MAXPATH + 1];

    UTT_ParamHelp ("poImageState GetFileFormat fileName formatList:out");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);

    if (!IP_GetFileFormat (fileName, tmp)) {
        return UT_False;
    }
    for (i=0; i<FF_NumImgChanTypes; i++) {
        formatList[i] = tmp[i];
    }

    UTT_SetFixInt32List (2, formatList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_WriteSimple (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    char fileName[MAXPATH + 1], format[Tmpsize+1], options[MAXPATH+1];

    UTT_ParamHelp ("$img WriteSimple fileName format ?options?");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);
    UTT_GetFixString (2, format,   Tmpsize, UT_False);
    UTT_GetOptString (3, options,  MAXPATH, UT_False, "");

    return IP_WriteSimple (img, fileName, format, options);
}

UT_Bool IPT_NewImageFromFile (OBJPARAMLIST)
{
    IP_ImageId img;
    char fileName[MAXPATH + 1], options[MAXPATH + 1];

    UTT_ParamHelp ("poImage NewImageFromFile fileName ?options? poImg:return");

    UTT_GetFixString (1, fileName, MAXPATH, UT_False);
    UTT_GetOptString (2, options,  MAXPATH, UT_False, "");

    if (!(img = IP_NewImageFromFile (fileName, options))) {
        return UT_False;
    }

    *clientData = img;
    return UT_True;
}

UT_Bool IPT_AddChannel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    FF_ImgFmtType  formatType;

    UTT_ParamHelp ("$img AddChannel channel formatType");

    IPT_GetFixChannelEnum (1, channel);
    IPT_GetFixFormatEnum  (2, formatType);
    return IP_AddChannel (img, channel, formatType);
}

UT_Bool IPT_DeleteChannel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;

    UTT_ParamHelp ("$img DeleteChannel channel");

    IPT_GetFixChannelEnum (1, channel);

    return IP_DeleteChannel (img, channel);
}

UT_Bool IPT_AsPhoto (ELEMPARAMLIST)
{
    IP_ImageId srcImg = (IP_ImageId) clientData;
    Int32 i, n, *chan = NULL;
    Int32 defMap3[] = { FF_ImgChanTypeRed, FF_ImgChanTypeGreen, FF_ImgChanTypeBlue, FF_ImgChanTypeNone };
    Int32 defMap4[] = { FF_ImgChanTypeRed, FF_ImgChanTypeGreen, FF_ImgChanTypeBlue, FF_ImgChanTypeMatte };
    FF_ImgFmtType imgFmt[FF_NumImgChanTypes];
    FF_ImgChanType chanMap[4];
    char photoName[MAXPATH + 1];
    Float64 gamma, minValue, maxValue;
    void *memstate;

    UTT_ParamHelp ("$srcImg AsPhoto photoName ?chanMapList? ?gamma=1.0? ?minValue=-1.0? ?maxValue=-1.0?");

    memstate = UT_MemRemember ();
    UTT_GetFixString (1, photoName, MAXPATH, UT_False);

    /* If no channel mapping is specified, use default mapping depending
       on whether the image has a transpareny channel.*/
    IP_GetImageFormat (srcImg, imgFmt);
    if (imgFmt[FF_ImgChanTypeMatte] != FF_ImgFmtTypeNone) {
        UTT_GetOptInt32List (2, chan, n, 1, defMap4);
    } else {
        UTT_GetOptInt32List (2, chan, n, 1, defMap3);
    }

    if (n == 0) {
        /* An empty list was specified as channel mapping. Use default mapping. */
        if (imgFmt[FF_ImgChanTypeMatte] != FF_ImgFmtTypeNone) {
            chan = defMap4;
            n = 4;
        } else {
            chan = defMap3;
            n = 3;
        }
    }
    for (i=0; i<n; i++) {
        chanMap[i] = (FF_ImgChanType) chan[i];
    }
    for (i=n; i<4; i++) {
        chanMap[i] = FF_ImgChanTypeNone;
    }

    UTT_GetOptFloat64 (3, gamma,     1.0);
    UTT_GetOptFloat64 (4, minValue, -1.0);
    UTT_GetOptFloat64 (5, maxValue, -1.0);

    UT_MemRestore (memstate);

    return IP_AsPhoto (interp, srcImg, photoName, chanMap, gamma, minValue, maxValue);
}

UT_Bool IPT_NewImageFromPhoto (OBJPARAMLIST)
{
    char photoName[MAXPATH + 1];
    IP_ImageId img;

    UTT_ParamHelp ("poImage NewImageFromPhoto photoName poImg:return");

    UTT_GetFixString (1, photoName, MAXPATH, UT_False);

    img = IP_NewImageFromPhoto (interp, photoName);

    if (!img) {
        return UT_False;
    }
    *clientData = img;
    return UT_True;
}

UT_Bool IPT_AsByteArray (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x, y;
    Int32 chan;
    Int32 numChans, numBytes;
    Int32 foundByteChans, foundFloatChans;
    FF_ImgChanType chanList[FF_NumImgChanTypes];
    FF_ImgDesc *pimg;
    Tcl_Obj *byteArray;
    unsigned char *byteArrayPtr;
    Tcl_Size byteArrayLen;

    UTT_ParamHelp ("$img AsByteArray numChans:out pixelType:out");

    pimg = &img->pimg;

    numChans = 0;
    foundByteChans  = 0;
    foundFloatChans = 0;
    for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
        if (pimg->channel[chan] != FF_ImgFmtTypeNone && IP_StateDrawMask[chan]) {
            if (pimg->channel[chan] == FF_ImgFmtTypeUByte) {
                foundByteChans++;
            } else if (pimg->channel[chan] == FF_ImgFmtTypeFloat) {
                foundFloatChans++;
            }
            chanList[numChans] = chan;
            numChans++;
        }
    }
    if (foundByteChans > 0 && foundFloatChans > 0) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Image has both byte and float channels."));
        return UT_False;
    }
    if (numChans < 1) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Image has no valid channels or draw masks are off."));
        return UT_False;
    }
    if (numChans > 4) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Image has more than 4 channels."));
        return UT_False;
    }
    if (foundByteChans > 0) {
        numBytes = pimg->desc.width * pimg->desc.height * numChans * sizeof (UInt8);
    } else {
        numBytes = pimg->desc.width * pimg->desc.height * numChans * sizeof (Float32);
    }

    byteArray = Tcl_NewByteArrayObj (NULL, numBytes);
    if (!byteArray) {
        Tcl_SetObjResult(interp,
            Tcl_ObjPrintf("Could not allocate ByteArray of size %d.", numBytes));
        return UT_False;
    }
    byteArrayPtr = Tcl_GetByteArrayFromObj(byteArray, &byteArrayLen);

    /* Always write ByteArray in TopDown order. If channels are in UByte format,
       img::raw can use a faster algorithms to convert into a photo image. */
    for (y = pimg->desc.height-1; y >= 0; y--) {
        for (x = 0; x < pimg->desc.width; x++) {
            for (chan = 0; chan < numChans; chan++) {
                Int32 curChan = chanList[chan];
                switch (pimg->channel[curChan]) {
                    case FF_ImgFmtTypeNone: {
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        *(byteArrayPtr++) = *FF_ImgUBytePixel (pimg, curChan, x, y);
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        *((Float32 *)byteArrayPtr) = *FF_ImgFloatPixel (pimg, curChan, x, y);
                        byteArrayPtr += sizeof (Float32);
                        break;
                    }
                }
            }
        }
    }
    UTT_SetFixInt32  (1, numChans);
    UTT_SetFixString (2, foundByteChans > 0? "byte": "float");

    UTT_SetFixTclObjResult (byteArray);
    return UT_True;
}

UT_Bool IPT_Interlace (ELEMPARAMLIST)
{
    IP_ImageId frameImg = (IP_ImageId) clientData;
    IP_ImageId fieldImg1, fieldImg2;
    UT_Bool skipLines;

    UTT_ParamHelp ("$frameImg Interlace fieldImg1 fieldImg2 ?skipLines=true?");

    IPT_GetFixImage (1, fieldImg1);
    IPT_GetFixImage (2, fieldImg2);
    UTT_GetOptBool  (3, skipLines, UT_True);

    return IP_Interlace (fieldImg1, fieldImg2, frameImg, skipLines);
}

UT_Bool IPT_Unlace (ELEMPARAMLIST)
{
    IP_ImageId fieldImg = (IP_ImageId) clientData;
    IP_ImageId frameImg;
    UT_Bool useOddRows;
    IP_UnlaceModeType unlaceMode;

    UTT_ParamHelp ("$fieldImg Unlace frameImg ?useOddRows=true? ?unlaceMode=EXTRACT?");

    IPT_GetFixImage          (1, frameImg);
    UTT_GetOptBool           (2, useOddRows, UT_True);
    IPT_GetOptUnlaceModeEnum (3, unlaceMode, IP_UnlaceModeExtract);

    return IP_Unlace (frameImg, fieldImg, useOddRows, unlaceMode);
}

UT_Bool IPT_ChangeChannelGamma (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    Float64 gamma, scale, offset;

    UTT_ParamHelp ("$img ChangeChannelGamma channel gamma ?scale=1.0? ?offset=0.0?");

    IPT_GetFixChannelEnum (1, channel);
    UTT_GetFixFloat64     (2, gamma);
    UTT_GetOptFloat64     (3, scale,  1.0);
    UTT_GetOptFloat64     (4, offset, 0.0);

    return IP_ChangeChannelGamma (img, channel, gamma, scale, offset);
}

UT_Bool IPT_RemapChannelValues (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    Int32 lutSize;
    UT_Bool retVal;
    Float32 *lutList;

    UTT_ParamHelp ("$img RemapChannelValues channel lutList");

    IPT_GetFixChannelEnum (1, channel);
    UTT_GetFixFloat32List (2, lutList, lutSize, 2);

    retVal = IP_RemapChannelValues (img, channel, lutSize, lutList);

    UT_MemFree (lutList);
    return retVal;
}

UT_Bool IPT_ComputeChannelDerivates (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    FF_ImgChanType srcChannel, horizontalChannel, verticalChannel;
    UT_Bool wrap;

    UTT_ParamHelp ("$destImg ComputeChannelDerivates srcImg srcChannel horizontalChannel verticalChannel ?wrap=true?");

    IPT_GetFixImage       (1, srcImg);
    IPT_GetFixChannelEnum (2, srcChannel);
    IPT_GetFixChannelEnum (3, horizontalChannel);
    IPT_GetFixChannelEnum (4, verticalChannel);
    UTT_GetOptBool        (5, wrap, UT_True);

    return IP_ComputeChannelDerivates (srcImg, srcChannel, destImg, horizontalChannel, verticalChannel, wrap);
}

UT_Bool IPT_FlipDiagonal (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img FlipDiagonal");

    IP_FlipDiagonal (img);

    return UT_True;
}

UT_Bool IPT_FlipHorizontal (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img FlipHorizontal");

    IP_FlipHorizontal (img);

    return UT_True;
}

UT_Bool IPT_FlipVertical (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img FlipVertical");

    IP_FlipVertical (img);

    return UT_True;
}

UT_Bool IPT_Rotate90 (ELEMPARAMLIST)
{
    IP_ImageId srcImg = (IP_ImageId) clientData;
    IP_ImageId destImg = NULL;
    FF_ImgHdr desc;
    FF_ImgFmtType fmt[FF_NumImgChanTypes];
    Int32 angle;
    Int32 w, h;
    Float32 aspect;

    UTT_ParamHelp ("$srcImg Rotate90 angle destImg:return");

    UTT_GetFixInt32 (1, angle);

    if (IP_PushState () < 0) {
        return UT_False;
    }
    IP_GetImageInfo   (srcImg, &desc);
    IP_GetImageFormat (srcImg, fmt);
    IP_SetFormat (fmt);

    if (angle == 180) {
        w = desc.width;
        h = desc.height;
        aspect = desc.aspect;
    } else {
        w = desc.height;
        h = desc.width;
        aspect = 1.0 / desc.aspect;
    }

    if (!(destImg = IP_NewImage (w, h, aspect, NULL))) {
        IP_PopState ();
        return UT_False;
    }
    IP_Blank (destImg);
    IP_PopState ();

    if (! IP_Rotate90 (srcImg, destImg, angle)) {
        IP_DeleteImage (destImg);
        return UT_False;
    }
    return IPT_CreateTclCmd (interp, destImg);
}

UT_Bool IPT_ScaleRect (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    Int32 xs1, ys1, xs2, ys2, xd1, yd1, xd2, yd2;
    UT_Bool usePyramidFilter, gammaCorrect;

    UTT_ParamHelp ("$destImg ScaleRect srcImg xs1 ys1 xs2 ys2 xd1 yd1 xd2 yd2 ?usePyramidFilter=false? ?gammaCorrect=false?");

    IPT_GetFixImage ( 1, srcImg);
    UTT_GetFixInt32 ( 2, xs1);
    UTT_GetFixInt32 ( 3, ys1);
    UTT_GetFixInt32 ( 4, xs2);
    UTT_GetFixInt32 ( 5, ys2);
    UTT_GetFixInt32 ( 6, xd1);
    UTT_GetFixInt32 ( 7, yd1);
    UTT_GetFixInt32 ( 8, xd2);
    UTT_GetFixInt32 ( 9, yd2);
    UTT_GetOptBool  (10, usePyramidFilter, UT_False);
    UTT_GetOptBool  (11, gammaCorrect, UT_False);

    return IP_ScaleRect
           (srcImg, destImg,
           xs1, ys1, xs2, ys2,
           xd1, yd1, xd2, yd2,
           usePyramidFilter, gammaCorrect);
}

UT_Bool IPT_CreateChannelNoise (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    Int32 seed, period, coherency;
    Float64 slice;

    UTT_ParamHelp ("$img CreateChannelNoise channel seed period coherency slice");

    IPT_GetFixChannelEnum (1, channel);
    UTT_GetFixInt32       (2, seed);
    UTT_GetFixInt32       (3, period);
    UTT_GetFixInt32       (4, coherency);
    UTT_GetFixFloat64     (5, slice);

    return IP_CreateChannelNoise (img, channel, seed, period, coherency, slice);
}

UT_Bool IPT_JuliaSet (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 i, n, numIterations;
    UInt8 rmap[256], gmap[256], bmap[256];
    Int32 *redColorList, *greenColorList, *blueColorList;
    Float64 radd, iadd, rcen, icen, rlen;

    UTT_ParamHelp ("$img JuliaSet radd iadd rcen icen rlen numIterations redColorList greenColorList blueColorList");

    UTT_GetFixFloat64   (1, radd);
    UTT_GetFixFloat64   (2, iadd);
    UTT_GetFixFloat64   (3, rcen);
    UTT_GetFixFloat64   (4, icen);
    UTT_GetFixFloat64   (5, rlen);
    UTT_GetFixInt32     (6, numIterations);
    UTT_GetFixInt32List (7, redColorList,   n, numIterations);
    UTT_GetFixInt32List (8, greenColorList, n, numIterations);
    UTT_GetFixInt32List (9, blueColorList,  n, numIterations);

    for (i=0; i<UT_MIN (n, 256); i++) {
        rmap[i] = (UInt8) redColorList[i];
        gmap[i] = (UInt8) greenColorList[i];
        bmap[i] = (UInt8) blueColorList[i];
    }
    UT_MemFree (redColorList);
    UT_MemFree (greenColorList);
    UT_MemFree (blueColorList);

    return IP_JuliaSet (img, radd, iadd, rcen, icen, rlen, numIterations, rmap, gmap, bmap);
}

UT_Bool IPT_Mandelbrot (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32  i, n, numIterations;
    UInt8  rmap[256], gmap[256], bmap[256];
    Int32  *redColorList, *greenColorList, *blueColorList;
    Float64 rcen, icen, rlen;

    UTT_ParamHelp ("$img Mandelbrot rcen icen rlen numIterations redColorList greenColorList blueColorList");

    UTT_GetFixFloat64   (1, rcen);
    UTT_GetFixFloat64   (2, icen);
    UTT_GetFixFloat64   (3, rlen);
    UTT_GetFixInt32     (4, numIterations);
    UTT_GetFixInt32List (5, redColorList,   n, numIterations);
    UTT_GetFixInt32List (6, greenColorList, n, numIterations);
    UTT_GetFixInt32List (7, blueColorList,  n, numIterations);

    for (i=0; i<UT_MIN (n, 256); i++) {
        rmap[i] = (UInt8) redColorList[i];
        gmap[i] = (UInt8) greenColorList[i];
        bmap[i] = (UInt8) blueColorList[i];
    }
    UT_MemFree (redColorList);
    UT_MemFree (greenColorList);
    UT_MemFree (blueColorList);

    return IP_Mandelbrot (img, rcen, icen, rlen, numIterations, rmap, gmap, bmap);
}

UT_Bool IPT_Median (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img Median");

    return IP_Median (img);
}

UT_Bool IPT_GetChannelHistogram (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    Int32 histogramList[256];

    UTT_ParamHelp ("$img GetChannelHistogram channel histogramList:out");

    IPT_GetFixChannelEnum (1, channel);

    if (!IP_GetChannelHistogram (img, channel, histogramList)) {
        return UT_False;
    }

    UTT_SetFixInt32List (2, histogramList, 256);
    return UT_True;
}

UT_Bool IPT_GetChannelStats (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64 meanValue, stdDev;
    FF_ImgChanType channel;
    Int32   x1, y1, x2, y2;
    UInt32  numPixels;
    UT_Bool useMatte;

    UTT_ParamHelp ("$img GetChannelStats channel x1 y1 x2 y2 meanValue:out ?stdDev:out? ?numPixels:out? ?useMatte=false?");

    IPT_GetFixChannelEnum (1, channel);
    UTT_GetFixInt32       (2, x1);
    UTT_GetFixInt32       (3, y1);
    UTT_GetFixInt32       (4, x2);
    UTT_GetFixInt32       (5, y2);
    UTT_GetOptBool        (9, useMatte, UT_False);

    if (!IP_GetChannelStats (img, channel, x1, y1, x2, y2, &meanValue, &stdDev, &numPixels, useMatte)) {
        return UT_False;
    }

    UTT_SetFixFloat64 (6, meanValue);
    UTT_SetOptFloat64 (7, stdDev);
    UTT_SetOptInt32   (8, numPixels);
    return UT_True;
}

UT_Bool IPT_GetChannelRange (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    FF_ImgChanType channel;
    Int32   x1, y1, x2, y2;
    Float32 minValue, maxValue;

    UTT_ParamHelp ("$img GetChannelRange channel x1 y1 x2 y2 minValue:out maxValue:out");

    IPT_GetFixChannelEnum (1, channel);
    UTT_GetFixInt32       (2, x1);
    UTT_GetFixInt32       (3, y1);
    UTT_GetFixInt32       (4, x2);
    UTT_GetFixInt32       (5, y2);

    if (!IP_GetChannelRange (img, channel, x1, y1, x2, y2, &minValue, &maxValue)) {
        return UT_False;
    }

    UTT_SetFixFloat64 (6, minValue);
    UTT_SetFixFloat64 (7, maxValue);
    return UT_True;
}

UT_Bool IPT_DiffChannel (ELEMPARAMLIST)
{
    IP_ImageId srcImg1 = (IP_ImageId) clientData;
    IP_ImageId srcImg2 = NULL;
    FF_ImgChanType channel;
    Int32 lessList1[256], lessList2[256];

    UTT_ParamHelp ("$srcImg1 DiffChannel srcImg2 channel lessList1:out lessList2:out");

    IPT_GetFixImage       (1, srcImg2);
    IPT_GetFixChannelEnum (2, channel);

    if (!IP_DiffChannel (srcImg1, srcImg2, channel, lessList1, lessList2)) {
        return UT_False;
    }

    UTT_SetFixInt32List (3, lessList1, 256);
    UTT_SetFixInt32List (4, lessList2, 256);
    return UT_True;
}

UT_Bool IPT_CorrelateChannel (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    FF_ImgChanType channel;
    Int32   xs1, ys1, xs2, ys2, xd1, yd1;
    Float32 correlationFactor;

    UTT_ParamHelp ("$destImg CorrelateChannel srcImg channel xs1 ys1 xs2 ys2 xd1 yd1 correlationFactor:return");

    IPT_GetFixImage       (1, srcImg);
    IPT_GetFixChannelEnum (2, channel);
    UTT_GetFixInt32       (3, xs1);
    UTT_GetFixInt32       (4, ys1);
    UTT_GetFixInt32       (5, xs2);
    UTT_GetFixInt32       (6, ys2);
    UTT_GetFixInt32       (7, xd1);
    UTT_GetFixInt32       (8, yd1);

    if (!IP_CorrelateChannel (destImg, srcImg, channel, xs1, ys1, xs2, ys2, xd1, yd1, &correlationFactor)) {
        return UT_False;
    }

    UTT_SetFixFloat64Result (correlationFactor);
    return UT_True;
}

UT_Bool IPT_GetSample (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64 x, y;
    Float32 colorList[FF_NumImgChanTypes] = { 0.0 };

    UTT_ParamHelp ("$img GetSample x y colorList:out");

    UTT_GetFixFloat64 (1, x);
    UTT_GetFixFloat64 (2, y);

    if (!IP_GetSample (img, x, y, colorList)) {
        return UT_False;
    }

    UTT_SetFixFloat32List (3, colorList, FF_NumImgChanTypes);
    return UT_True;
}

UT_Bool IPT_GetPixel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x, y;
    Float32 colorList[FF_NumImgChanTypes] = { 0.0 };

    UTT_ParamHelp ("$img GetPixel x y colorList:out");

    UTT_GetFixInt32 (1, x);
    UTT_GetFixInt32 (2, y);

    if (!IP_GetPixel (img, x, y, colorList)) {
        return UT_False;
    }

    UTT_SetFixFloat32List (3, colorList, FF_NumImgChanTypes);
    return UT_True;
}


UT_Bool IPT_MedianSequence (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg1, srcImg2, srcImg3;

    UTT_ParamHelp ("$destImg MedianSequence srcImg1 srcImg2 srcImg3");

    IPT_GetFixImage (1, srcImg1);
    IPT_GetFixImage (2, srcImg2);
    IPT_GetFixImage (3, srcImg3);

    return IP_MedianSequence (srcImg1, srcImg2, srcImg3, destImg);
}

UT_Bool IPT_Dilatation (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 count;

    UTT_ParamHelp ("$img Dilatation ?count=1?");

    UTT_GetOptInt32 (1, count, 1);

    return IP_Dilatation (img, count);
}

UT_Bool IPT_Erosion (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 count;

    UTT_ParamHelp ("$img Erosion ?count=1?");

    UTT_GetOptInt32 (1, count, 1);

    return IP_Erosion (img, count);
}

UT_Bool IPT_CutOff (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64    thresholdMin, thresholdMax;

    UTT_ParamHelp ("$img CutOff thresholdMin ?thresholdMax=1.0?");

    UTT_GetFixFloat64 (1, thresholdMin);
    UTT_GetOptFloat64 (2, thresholdMax, 1.0);
    if (thresholdMin > thresholdMax) {
        UT_SWAP (thresholdMin, thresholdMax, Float64);
    }

    return IP_CutOff (img, thresholdMin, thresholdMax);
}

UT_Bool IPT_Threshold (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    Float64 threshold;
    UT_Bool overwrite;

    UTT_ParamHelp ("$destImg Threshold srcImg threshold ?overwrite=false?");

    IPT_GetFixImage   (1, srcImg);
    UTT_GetFixFloat64 (2, threshold);
    UTT_GetOptBool    (3, overwrite, UT_False);

    return IP_Threshold (srcImg, img, threshold, overwrite);
}

UT_Bool IPT_MarkNonZeroPixels (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg  = NULL;
    IP_ImageId newImg  = NULL;
    FF_ImgHdr  desc;
    FF_ImgFmtType fmt[FF_NumImgChanTypes];
    Float32 threshold;
    Int32 numMarkedPixels = -1;

    UTT_ParamHelp ("$srcImg  MarkNonZeroPixels threshold numMarkedPixels:out destImg:return\n"
                   "$destImg MarkNonZeroPixels threshold numMarkedPixels:out srcImg");

    UTT_GetFixFloat32 (1, threshold);
    UTT_SetFixInt32   (2, numMarkedPixels);
    IPT_GetOptImage   (3, srcImg, NULL);

    if (srcImg != NULL) {
        /* If a srcImg was supplied, copy the image data from srcImg into destImg. */
        if (! IP_MarkNonZeroPixels (srcImg, destImg, threshold, &numMarkedPixels)) {
            return UT_False;
        }
        UTT_SetFixInt32 (2, numMarkedPixels);
        return UT_True;
    }

    if (IP_PushState () < 0) {
        return UT_False;
    }
    IP_GetImageInfo   (destImg, &desc);
    IP_GetImageFormat (destImg, fmt);
    IP_SetFormat (fmt);

    if (!(newImg = IP_NewImage (desc.width, desc.height, desc.aspect, NULL))) {
        IP_PopState ();
        return UT_False;
    }
    IP_Blank (newImg);
    IP_PopState ();

    if (! IP_MarkNonZeroPixels (destImg, newImg, threshold, &numMarkedPixels)) {
        IP_DeleteImage (newImg);
        return UT_False;
    }

    UTT_SetFixInt32 (2, numMarkedPixels);
    return IPT_CreateTclCmd (interp, newImg);
}

UT_Bool IPT_DifferenceImage (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    IP_ImageId destImg  = NULL;
    IP_ImageId newImg   = NULL;
    IP_ImageId srcImg1  = NULL;
    FF_ImgHdr  desc;
    FF_ImgFmtType fmt[FF_NumImgChanTypes];

    UTT_ParamHelp ("$srcImg2 DifferenceImage srcImg1 destImg:return\n"
                   "$destImg DifferenceImage srcImg1 srcImg2");

    IPT_GetFixImage (1, srcImg1);
    IPT_GetOptImage (2, destImg, NULL);

    if (destImg != NULL) {
        /* If a destImg was supplied, copy the image data from img into destImg. */
        return IP_DifferenceImage (img, srcImg1, destImg);
    }

    if (IP_PushState () < 0) {
        return UT_False;
    }
    IP_GetImageInfo   (img, &desc);
    IP_GetImageFormat (img, fmt);
    IP_SetFormat (fmt);

    if (!(newImg = IP_NewImage (desc.width, desc.height, desc.aspect, NULL))) {
        IP_PopState ();
        return UT_False;
    }
    IP_Blank (newImg);
    IP_PopState ();

    if (! IP_DifferenceImage (img, srcImg1, newImg)) {
        IP_DeleteImage (newImg);
        return UT_False;
    }
    return IPT_CreateTclCmd (interp, newImg);
}

UT_Bool IPT_DrawPixel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x, y;

    UTT_ParamHelp ("$img DrawPixel x y");

    UTT_GetFixInt32 (1, x);
    UTT_GetFixInt32 (2, y);

    IP_DrawPixel (img, x, y);

    return UT_True;
}

UT_Bool IPT_DrawLine (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x1, y1;
    Int32 x2, y2;
    Int32 stipple;

    UTT_ParamHelp ("$img DrawLine x1 y1 x2 y2 ?stipple=1?");

    UTT_GetFixInt32 (1, x1);
    UTT_GetFixInt32 (2, y1);
    UTT_GetFixInt32 (3, x2);
    UTT_GetFixInt32 (4, y2);
    UTT_GetOptInt32 (5, stipple, 1);

    IP_DrawLine (img, x1, y1, x2, y2, stipple);

    return UT_True;
}

UT_Bool IPT_DrawRect (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x1, y1;
    Int32 x2, y2;
    UT_Bool fill;

    UTT_ParamHelp ("$img DrawRect x1 y1 x2 y2 ?fill=true?");

    UTT_GetFixInt32 (1, x1);
    UTT_GetFixInt32 (2, y1);
    UTT_GetFixInt32 (3, x2);
    UTT_GetFixInt32 (4, y2);
    UTT_GetOptBool  (5, fill, UT_True);

    IP_DrawRect (img, x1, y1, x2, y2, fill);

    return UT_True;
}

UT_Bool IPT_DrawEllipse (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 cx, cy;
    Int32 radius1, radius2;
    UT_Bool fill;

    UTT_ParamHelp ("$img DrawEllipse cx cy radius1 radius2 ?fill=true?");

    UTT_GetFixInt32 (1, cx);
    UTT_GetFixInt32 (2, cy);
    UTT_GetFixInt32 (3, radius1);
    UTT_GetFixInt32 (4, radius2);
    UTT_GetOptBool  (5, fill, UT_True);

    IP_DrawEllipse (img, cx, cy, radius1, radius2, fill);

    return UT_True;
}

UT_Bool IPT_DrawCircle (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 cx, cy;
    Int32 radius;
    UT_Bool fill;

    UTT_ParamHelp ("$img DrawCircle cx cy radius ?fill=true?");

    UTT_GetFixInt32 (1, cx);
    UTT_GetFixInt32 (2, cy);
    UTT_GetFixInt32 (3, radius);
    UTT_GetOptBool  (4, fill, UT_True);

    IP_DrawCircle (img, cx, cy, radius, fill);

    return UT_True;
}

UT_Bool IPT_DrawBrushLine (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    IP_ImageId brushImg;
    Int32   x1, y1, x2, y2, strokes;
    Float64 intensity;

    UTT_ParamHelp ("$img DrawBrushLine brushImg x1 y1 x2 y2 ?strokes=1? ?intensity=1.0?");

    IPT_GetFixImage   (1, brushImg);
    UTT_GetFixInt32   (2, x1);
    UTT_GetFixInt32   (3, y1);
    UTT_GetFixInt32   (4, x2);
    UTT_GetFixInt32   (5, y2);
    UTT_GetOptInt32   (6, strokes, 1);
    UTT_GetOptFloat64 (7, intensity, 1.0);

    IP_DrawBrushLine (img, brushImg, x1, y1, x2, y2, strokes, intensity);

    return UT_True;
}

UT_Bool IPT_DrawBrush (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    IP_ImageId brushImg;
    Int32   x, y;
    Float64 intensity;

    UTT_ParamHelp ("$img DrawBrush brushImg x y ?intensity=1.0?");

    IPT_GetFixImage   (1, brushImg);
    UTT_GetFixInt32   (2, x);
    UTT_GetFixInt32   (3, y);
    UTT_GetOptFloat64 (4, intensity, 1.0);

    return IP_DrawBrush (img, brushImg, x, y, intensity);
}

UT_Bool IPT_MergeBrush (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg, brushImg;
    Int32 x, y;
    Float64 intensity;

    UTT_ParamHelp ("$destImg MergeBrush srcImg brushImg x y ?intensity=1.0?");

    IPT_GetFixImage   (1, srcImg);
    IPT_GetFixImage   (2, brushImg);
    UTT_GetFixInt32   (3, x);
    UTT_GetFixInt32   (4, y);
    UTT_GetOptFloat64 (5, intensity, 1.0);

    return IP_MergeBrush (srcImg, destImg, brushImg, x, y, intensity);
}

UT_Bool IPT_DrawText (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x, y;
    char string[Tmpsize + 1];

    UTT_ParamHelp ("$img DrawText x y string");

    UTT_GetFixInt32  (1, x);
    UTT_GetFixInt32  (2, y);
    UTT_GetFixString (3, string, Tmpsize, UT_False);

    IP_DrawText (img, x, y, string);

    return UT_True;
}

UT_Bool IPT_GetTextSize (TCLPARAMLIST)
{
    Int32 width, height;
    char string[Tmpsize + 1];

    UTT_ParamHelp ("poImageState GetTextSize string width:out height:out");

    UTT_GetFixString (1, string, Tmpsize, UT_False);

    IP_GetTextSize (string, &width, &height);

    UTT_SetFixInt32 (2, width);
    UTT_SetFixInt32 (3, height);
    return UT_True;
}

UT_Bool IPT_SetTextScale (TCLPARAMLIST)
{
    Int32 scale;

    UTT_ParamHelp ("poImageState SetTextScale scale");

    UTT_GetFixInt32 (1, scale);

    return IP_SetTextScale (scale);
}

UT_Bool IPT_GetTextScale (TCLPARAMLIST)
{
    UTT_ParamHelp ("poImageState GetTextScale textScale:return");

    UTT_SetFixInt32Result (IP_GetTextScale ());

    return UT_True;
}

UT_Bool IPT_BlankRect (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Int32 x1, y1;
    Int32 x2, y2;

    UTT_ParamHelp ("$img BlankRect x1 y1 x2 y2");

    UTT_GetFixInt32 (1, x1);
    UTT_GetFixInt32 (2, y1);
    UTT_GetFixInt32 (3, x2);
    UTT_GetFixInt32 (4, y2);

    IP_BlankRect (img, x1, y1, x2, y2);

    return UT_True;
}

UT_Bool IPT_Blank (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img Blank");

    IP_Blank (img);

    return UT_True;
}

UT_Bool IPT_CopyRect (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    Int32 xs1, ys1, xs2, ys2, xd1, yd1;

    UTT_ParamHelp ("$destImg CopyRect srcImg xs1 ys1 xs2 ys2 xd1 yd1");

    IPT_GetFixImage (1, srcImg);
    UTT_GetFixInt32 (2, xs1);
    UTT_GetFixInt32 (3, ys1);
    UTT_GetFixInt32 (4, xs2);
    UTT_GetFixInt32 (5, ys2);
    UTT_GetFixInt32 (6, xd1);
    UTT_GetFixInt32 (7, yd1);

    IP_CopyRect (srcImg, destImg, xs1, ys1, xs2, ys2, xd1, yd1);

    return UT_True;
}

UT_Bool IPT_WrapRect (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    Int32 xs1, ys1, xs2, ys2, sx, sy;

    UTT_ParamHelp ("$destImg WrapRect srcImg xs1 ys1 xs2 ys2 sx sy");

    IPT_GetFixImage (1, srcImg);
    UTT_GetFixInt32 (2, xs1);
    UTT_GetFixInt32 (3, ys1);
    UTT_GetFixInt32 (4, xs2);
    UTT_GetFixInt32 (5, ys2);
    UTT_GetFixInt32 (6, sx);
    UTT_GetFixInt32 (7, sy);

    IP_WrapRect (srcImg, destImg, xs1, ys1, xs2, ys2, sx, sy);

    return UT_True;
}

UT_Bool IPT_CopyChannel (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId srcImg;
    FF_ImgChanType srcChannel, destChannel;

    UTT_ParamHelp ("$destImg CopyChannel srcImg srcChannel destChannel");

    IPT_GetFixImage       (1, srcImg);
    IPT_GetFixChannelEnum (2, srcChannel);
    IPT_GetFixChannelEnum (3, destChannel);

    return IP_CopyChannel (srcImg, destImg, srcChannel, destChannel);
}

UT_Bool IPT_CopyImage (ELEMPARAMLIST)
{
    IP_ImageId img    = (IP_ImageId) clientData;
    IP_ImageId srcImg = NULL;
    IP_ImageId newImg = NULL;
    FF_ImgHdr desc;
    FF_ImgFmtType fmt[FF_NumImgChanTypes];

    UTT_ParamHelp ("$srcImg  CopyImage destImg:return\n"
                   "$destImg CopyImage srcImg");

    IPT_GetOptImage (1, srcImg, NULL);

    if (srcImg != NULL) {
        /* If a srcImg was supplied, copy the image data from srcImg into img. */
        return IP_CopyImage (srcImg, img);
    }

    /* No srcImg was supplied: img contains the image data.
     * The image data is copied into a new image and a new
     * image Tcl command is created. */
    if (IP_PushState () < 0) {
        return UT_False;
    }
    IP_GetImageInfo   (img, &desc);
    IP_GetImageFormat (img, fmt);
    IP_SetFormat (fmt);

    if (!(newImg = IP_NewImage (desc.width, desc.height, desc.aspect, NULL))) {
        IP_PopState ();
        return UT_False;
    }
    IP_Blank (newImg);
    IP_PopState ();

    if (!IP_CopyImage (img, newImg)) {
        IP_DeleteImage (newImg);
        return UT_False;
    }
    return IPT_CreateTclCmd (interp, newImg);
}

UT_Bool IPT_DrawAAPixel (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64 x, y;
    Float32 *colorList;
    Int32 n;

    UTT_ParamHelp ("$img DrawAAPixel x y colorList");

    UTT_GetFixFloat64     (1, x);
    UTT_GetFixFloat64     (2, y);
    UTT_GetFixFloat32List (3, colorList, n, FF_NumImgChanTypes);

    IP_DrawAAPixel (img, x, y, colorList);

    UT_MemFree (colorList);
    return UT_True;
}

UT_Bool IPT_DrawAALine (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64 x1, y1, x2, y2;
    Float32 *colorList1 = NULL, *colorList2 = NULL;
    Float32 colorList[FF_NumImgChanTypes];
    Float32 defColorList[FF_NumImgChanTypes];
    Int32 i, n;
    void *memstate;

    UTT_ParamHelp ("$img DrawAALine x1 y1 x2 y2 colorList1 ?colorList2=colorList1?");

    memstate = UT_MemRemember();

    UTT_GetFixFloat64     (1, x1);
    UTT_GetFixFloat64     (2, y1);
    UTT_GetFixFloat64     (3, x2);
    UTT_GetFixFloat64     (4, y2);
    UTT_GetFixFloat32List (5, colorList1, n, FF_NumImgChanTypes);
    /* Optional value must not be a pointer. So copy colorList1 into defColorList. */
    memcpy (defColorList, colorList1, FF_NumImgChanTypes * sizeof (Float32));
    UTT_GetOptFloat32List (6, colorList2, n, 1, defColorList);

    if (n == 0) {
        /* An empty list was specified as color list. Use default color list. */
        colorList2 = defColorList;
        n = FF_NumImgChanTypes;
    }
    for (i=0; i<n; i++) {
        colorList[i] = colorList2[i];
    }
    for (i=n; i<FF_NumImgChanTypes; i++) {
        colorList[i] = 0.0;
    }

    IP_DrawAALine (img, x1, y1, x2, y2, colorList1, colorList);

    UT_MemRestore (memstate);
    return UT_True;
}

UT_Bool IPT_DrawAAText (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    Float64 x, y;
    Float32 *colorList;
    Int32 n;
    char string[Tmpsize + 1];

    UTT_ParamHelp ("$img DrawAAText x y string colorList");

    UTT_GetFixFloat64     (1, x);
    UTT_GetFixFloat64     (2, y);
    UTT_GetFixString      (3, string, Tmpsize, UT_False);
    UTT_GetFixFloat32List (4, colorList, n, FF_NumImgChanTypes);

    IP_DrawAAText (img, x, y, string, colorList);

    UT_MemFree (colorList);
    return UT_True;
}
