/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IPT_TclIfThreded.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The interface between the image
 *                      library and the Tcl interpreter.
 *                      This file contains all thread-aware functions.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"

#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "FF_Image.h"
#include "IP_Image.h"

#include <tcl.h>
#include "UTT_TclIf.h"
#include "IPT_TclIf.h"
#include "IPT_ImagePrivate.h"

UT_Bool IPT_CompositeMatte (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId fgImg, bgImg, mtImg;
    double  fr, fg, fb;
    double  br, bg, bb;
    UT_Bool fastMode;

    UTT_ParamHelp ("$destImg CompositeMatte fgImg bgImg ?fr=1.0? ?fg=1.0? ?fb=1.0? ?br=1.0? ?bg=1.0? ?bb=1.0? ?fastMode=false? ?matteImg=None?");

    IPT_GetFixImage   ( 1, fgImg);
    IPT_GetFixImage   ( 2, bgImg);
    UTT_GetOptFloat64 ( 3, fr, 1.0);
    UTT_GetOptFloat64 ( 4, fg, 1.0);
    UTT_GetOptFloat64 ( 5, fb, 1.0);
    UTT_GetOptFloat64 ( 6, br, 1.0);
    UTT_GetOptFloat64 ( 7, bg, 1.0);
    UTT_GetOptFloat64 ( 8, bb, 1.0);
    UTT_GetOptBool    ( 9, fastMode, UT_False);
    IPT_GetOptImage   (10, mtImg, NULL);

    return IP_CompositeMatte (fgImg, bgImg, destImg,
                              fr, fg, fb, br, bg, bb,
                              fastMode, mtImg);
}

UT_Bool IPT_CompositeColorMatte (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId fgImg, bgImg, mtImg;
    double  fr, fg, fb;
    double  br, bg, bb;
    UT_Bool fastMode;

    UTT_ParamHelp ("$destImg CompositeColorMatte fgImg bgImg ?fr=1.0? ?fg=1.0? ?fb=1.0? ?br=1.0? ?bg=1.0? ?bb=1.0? ?fastMode=false? ?matteImg=None?");

    IPT_GetFixImage   ( 1, fgImg);
    IPT_GetFixImage   ( 2, bgImg);
    UTT_GetOptFloat64 ( 3, fr, 1.0);
    UTT_GetOptFloat64 ( 4, fg, 1.0);
    UTT_GetOptFloat64 ( 5, fb, 1.0);
    UTT_GetOptFloat64 ( 6, br, 1.0);
    UTT_GetOptFloat64 ( 7, bg, 1.0);
    UTT_GetOptFloat64 ( 8, bb, 1.0);
    UTT_GetOptBool    ( 9, fastMode, UT_False);
    IPT_GetOptImage   (10, mtImg, NULL);

    return IP_CompositeColorMatte (fgImg, bgImg, destImg,
                                   fr, fg, fb, br, bg, bb,
                                   fastMode, mtImg);
}

UT_Bool IPT_CompositeDepth (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId fgImg, bgImg;

    UTT_ParamHelp ("$destImg CompositeDepth fgImg bgImg");

    IPT_GetFixImage (1, fgImg);
    IPT_GetFixImage (2, bgImg);
    return IP_CompositeDepth (fgImg, bgImg, destImg);
}

UT_Bool IPT_CrossDissolve (ELEMPARAMLIST)
{
    IP_ImageId destImg = (IP_ImageId) clientData;
    IP_ImageId fgImg, bgImg;
    double mixFactor;

    UTT_ParamHelp ("$destImg CrossDissolve fgImg bgImg mixFactor");

    IPT_GetFixImage   (1, fgImg);
    IPT_GetFixImage   (2, bgImg);
    UTT_GetFixFloat64 (3, mixFactor);
    return IP_CrossDissolve (fgImg, bgImg, destImg, mixFactor);
}

UT_Bool IPT_Blur (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    double hori, vert;

    UTT_ParamHelp ("$img Blur hori ?vert=hori?");

    UTT_GetFixFloat64 (1, hori);
    UTT_GetOptFloat64 (2, vert, hori);
    return IP_Blur (img, hori, vert);
}

UT_Bool IPT_DepthBlur (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    double hori, vert, minDepth, maxDepth;

    UTT_ParamHelp ("$img DepthBlur hori ?vert=hori? ?minDepth=-1.0? ?maxDepth=-1.0?");

    UTT_GetFixFloat64 (1, hori);
    UTT_GetOptFloat64 (2, vert, hori);
    UTT_GetOptFloat64 (3, minDepth, -1.0);
    UTT_GetOptFloat64 (4, maxDepth, -1.0);
    return IP_DepthBlur (img, hori, vert, minDepth, maxDepth);
}

UT_Bool IPT_Flicker (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;

    UTT_ParamHelp ("$img Flicker");

    return IP_Flicker (img);
}

UT_Bool IPT_FindChannelPattern (ELEMPARAMLIST)
{
    IP_ImageId srcImg1 = (IP_ImageId) clientData;
    IP_ImageId srcImg2;
    FF_ImgChanType channel;
    Int32   xs1, ys1, xs2, ys2, xd1, yd1, xd2, yd2;
    Float32 xp, yp, rp;

    UTT_ParamHelp ("$srcImg1 FindChannelPattern srcImg2 channel xs1 ys1 xs2 ys2 xd1 yd1 xd2 yd2 xp:out yp:out rp:out");

    IPT_GetFixImage       ( 1, srcImg2);
    IPT_GetFixChannelEnum ( 2, channel);
    UTT_GetFixInt32       ( 3, xs1);
    UTT_GetFixInt32       ( 4, ys1);
    UTT_GetFixInt32       ( 5, xs2);
    UTT_GetFixInt32       ( 6, ys2);
    UTT_GetFixInt32       ( 7, xd1);
    UTT_GetFixInt32       ( 8, yd1);
    UTT_GetFixInt32       ( 9, xd2);
    UTT_GetFixInt32       (10, yd2);
    if (!IP_FindChannelPattern (srcImg1, srcImg2, channel, xs1, ys1, xs2, ys2,
                                xd1, yd1, xd2, yd2, &xp, &yp, &rp)) {
        return UT_False;
    }
    UTT_SetFixFloat64 (11, xp);
    UTT_SetFixFloat64 (12, yp);
    UTT_SetFixFloat64 (13, rp);
    return UT_True;
}

UT_Bool IPT_ShearVertical (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    double m, t;

    UTT_ParamHelp ("$img ShearVertical m t");

    UTT_GetFixFloat64 (1, m);
    UTT_GetFixFloat64 (2, t);
    return IP_ShearVertical (img, m, t);
}

UT_Bool IPT_ShearHorizontal (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    double m, t;

    UTT_ParamHelp ("$img ShearHorizontal m t");

    UTT_GetFixFloat64 (1, m);
    UTT_GetFixFloat64 (2, t);
    return IP_ShearHorizontal (img, m, t);
}

UT_Bool IPT_Stamp (ELEMPARAMLIST)
{
    IP_ImageId img = (IP_ImageId) clientData;
    IP_ImageId stampImg;
    Int32 x, y;
    double sr, sg, sb;

    UTT_ParamHelp ("$img Stamp stampImg x y ?sr=1.0? ?sg=1.0? ?sb=1.0?");

    IPT_GetFixImage   (1, stampImg);
    UTT_GetFixInt32   (2, x);
    UTT_GetFixInt32   (3, y);
    UTT_GetOptFloat64 (4, sr, 1.0);
    UTT_GetOptFloat64 (5, sg, 1.0);
    UTT_GetOptFloat64 (6, sb, 1.0);

    return IP_Stamp (img, stampImg, x, y, sr, sg, sb);
}
