/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IPT_TclIfWarp.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The interface between the image
 *                      library and the Tcl interpreter.
 *                      This file contains warp and blend related functions.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"

#include "UT_Error.h"
#include "UT_Memory.h"
#include "FF_Image.h"
#include "IP_Image.h"

#include <tcl.h>
#include "UTT_TclIf.h"
#include "IPT_TclIf.h"
#include "IPT_ImagePrivate.h"

#define str_warp_recursion "'warp' and 'blend' cannot be invoked recursively"

static UT_Bool sRecursive = UT_False;    /* Flag to detect recursion */

static char d1Funct[Tmpsize + 1],       /* First displacement function */
            d1Deriv[Tmpsize + 1],       /* Derivatives of "d1Funct" */
            d2Funct[Tmpsize + 1],       /* Second displacement function */
            d2Deriv[Tmpsize + 1],       /* Derivatives of "d2Funct" */
            mixFunct[Tmpsize + 1];      /* Pixel color mixing function */

static Tcl_Interp *myInterp;            /* Current Tcl interpreter */


/* Invoke Tcl function "funct", passing two Float32 parameters, "x" and "y"
   to "funct"; expect "funct" to return two Float32 numbers, "u" and "v". */

static UT_Bool CallFuncWith2ReturnValues
        (const char *funct, Float32 x, Float32 y,
         Float32 *u, Float32 *v)
{
    Tcl_Size listArgc;
    int result;
    Float32 utmp, vtmp;
    const char **listArgv = NULL;
    Tcl_DString command;
    char paramStr[100];

    Tcl_DStringInit (&command);
    Tcl_DStringAppend (&command, funct, -1);

    sprintf (paramStr, "%f", x);
    Tcl_DStringAppendElement (&command, paramStr);
    sprintf (paramStr, "%f", y);
    Tcl_DStringAppendElement (&command, paramStr);

    result = Tcl_Eval (myInterp, Tcl_DStringValue (&command));

    if (result != TCL_OK) {
        Tcl_AppendResult (myInterp, " (Could not evaluate \"", Tcl_DStringValue (&command), "\")", (char *)NULL);
        Tcl_DStringFree (&command);
        return UT_False;
    }

    Tcl_DStringFree (&command);
    if (Tcl_SplitList (myInterp, Tcl_GetStringResult (myInterp),
                       &listArgc, &listArgv) != TCL_OK) {
        return UT_False;
    }

    if (listArgc < 2) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("Could not evaluate command \"%s\" (Have %d parameters instead of 2)", funct, (int)listArgc));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[0], "%f", &utmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 0 is not a valid float.", listArgv[0]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[1], "%f", &vtmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 1 is not a valid float.", listArgv[1]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    *u = utmp;
    *v = vtmp;
    
    if (listArgv) {
        Tcl_Free ((char *) listArgv);
    }
    return UT_True;
}

/* Invoke Tcl function "funct", passing two Float32 parameters, "x" and "y"
   to "funct"; expect "funct" to return four Float32 numbers, "ux, "uy", "vx"
   and "vy". */

static UT_Bool CallFuncWith4ReturnValues
        (char *funct, Float32 x, Float32 y,
         Float32 *ux, Float32 *uy, Float32 *vx, Float32 *vy)
{
    Tcl_Size listArgc;
    int result;
    Float32 uxtmp, uytmp, vxtmp, vytmp;
    const char **listArgv = NULL;
    Tcl_DString command;
    char paramStr[100];

    Tcl_DStringInit (&command);
    Tcl_DStringAppend (&command, funct, -1);

    sprintf (paramStr, "%f", x);
    Tcl_DStringAppendElement (&command, paramStr);
    sprintf (paramStr, "%f", y);
    Tcl_DStringAppendElement (&command, paramStr);

    result = Tcl_Eval (myInterp, Tcl_DStringValue (&command));

    if (result != TCL_OK) {
        Tcl_AppendResult (myInterp, " (Could not evaluate \"", Tcl_DStringValue (&command), "\")", (char *)NULL);
        Tcl_DStringFree (&command);
        return UT_False;
    }

    Tcl_DStringFree (&command);
    if (Tcl_SplitList (myInterp, Tcl_GetStringResult (myInterp),
                       &listArgc, &listArgv) != TCL_OK) {
        return UT_False;
    }
    if (listArgc < 4) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("Could not evaluate command \"%s\" (Have %d parameters instead of 4)", funct, (int)listArgc));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[0], "%f", &uxtmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 0 is not a valid float.", listArgv[0]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[1], "%f", &uytmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 1 is not a valid float.", listArgv[1]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[2], "%f", &vxtmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 2 is not a valid float.", listArgv[2]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[3], "%f", &vytmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 3 is not a valid float.", listArgv[3]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    *ux = uxtmp;
    *uy = uytmp;
    *vx = vxtmp;
    *vy = vytmp;

    if (listArgv) {
        Tcl_Free ((char *) listArgv);
    }
    return UT_True;
}

/* Invoke Tcl function "funct", passing two Float32 parameters, "x" and "y"
   to "funct"; expect "funct" to return one Float32 number, "f". */

static UT_Bool CallFuncWith1ReturnValue
        (const char *funct, Float32 x, Float32 y, Float32 *f)
{
    Tcl_Size listArgc;
    int result;
    Float32 ftmp;
    const char **listArgv = NULL;
    Tcl_DString command;
    char paramStr[100];

    Tcl_DStringInit (&command);
    Tcl_DStringAppend (&command, funct, -1);

    sprintf (paramStr, "%f", x);
    Tcl_DStringAppendElement (&command, paramStr);
    sprintf (paramStr, "%f", y);
    Tcl_DStringAppendElement (&command, paramStr);

    result = Tcl_Eval (myInterp, Tcl_DStringValue (&command));

    if (result != TCL_OK) {
        Tcl_AppendResult (myInterp, " (Could not evaluate \"", Tcl_DStringValue (&command), "\")", (char *)NULL);
        Tcl_DStringFree (&command);
        return UT_False;
    }

    Tcl_DStringFree (&command);
    if (Tcl_SplitList (myInterp, Tcl_GetStringResult (myInterp),
                       &listArgc, &listArgv) != TCL_OK) {
        return UT_False;
    }

    if (listArgc < 1) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("Could not evaluate command \"%s\" (Have %d parameters instead of 1)", funct, (int)listArgc));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    if (1 != sscanf (listArgv[0], "%f", &ftmp)) {
        Tcl_SetObjResult (myInterp, Tcl_ObjPrintf ("List value \"%s\" at index 0 is not a valid float.", listArgv[0]));
        if (listArgv) {
            Tcl_Free ((char *) listArgv);
        }
        return UT_False;
    }
    *f = ftmp;
    
    if (listArgv) {
        Tcl_Free ((char *) listArgv);
    }
    return UT_True;
}

/* Invoke the Tcl versions of "d1Funct", "d1Deriv", etc. */

static UT_Bool CallFunc1 (Float32 x, Float32 y, Float32 *u, Float32 *v)
{
    return CallFuncWith2ReturnValues (d1Funct, x, y, u, v);
}

static UT_Bool CallFuncDeriv1 (Float32 x, Float32 y, Float32 *ux, Float32 *uy, Float32 *vx, Float32 *vy)
{
    return CallFuncWith4ReturnValues (d1Deriv, x, y, ux, uy, vx, vy);
}

static UT_Bool CallFunc2 (Float32 x, Float32 y, Float32 *u, Float32 *v)
{
    return CallFuncWith2ReturnValues (d2Funct, x, y, u, v);
}

static UT_Bool CallFuncDeriv2 (Float32 x, Float32 y, Float32 *ux, Float32 *uy, Float32 *vx, Float32 *vy)
{
    return CallFuncWith4ReturnValues (d2Deriv, x, y, ux, uy, vx, vy);
}

static UT_Bool CallMixFunc (Float32 x, Float32 y, Float32 *f)
{
    return CallFuncWith1ReturnValue (mixFunct, x, y, f);
}


UT_Bool IPT_WarpFunct (ELEMPARAMLIST)
{
    IP_ImageId      destImg = (IP_ImageId) clientData;
    IP_ImageId      srcImg;
    Int32           interpStep;
    IP_FillModeType fillMode;
    UT_Bool         success;

    UTT_ParamHelp ("$destImg WarpFunct srcImg d1Funct d1Deriv interpStep ?fillMode=FILL?");

    /* Check for recursive invocation of IPT_WarpFunct. */
    if (sRecursive) {
        UT_ErrSetNum (UT_ErrNotSupported, str_warp_recursion);
        return UT_False;
    }

    /* Get hold of the parameters passed by the Tcl interpreter. */
    IPT_GetFixImage        (1, srcImg);
    UTT_GetFixFunct        (2, d1Funct, Tmpsize);
    UTT_GetFixFunct        (3, d1Deriv, Tmpsize);
    UTT_GetFixInt32        (4, interpStep);
    IPT_GetOptFillModeEnum (5, fillMode, IP_FillModeFill);

    /* Run the image warping algorithm. */
    sRecursive = UT_True;
    myInterp = interp;
    success = IP_WarpFunct (srcImg, destImg, CallFunc1, CallFuncDeriv1, interpStep, fillMode);
    sRecursive = UT_False;
    return success;
}

UT_Bool IPT_BlendFunct (ELEMPARAMLIST)
{
    IP_ImageId      destImg = (IP_ImageId) clientData;
    IP_ImageId      srcImg1, srcImg2;
    Int32           interpStep;
    IP_FillModeType fillMode;
    UT_Bool         success;

    UTT_ParamHelp ("$destImg BlendFunct srcImg1 srcImg2 d1Funct d1Deriv d2Funct d2Deriv mixFunct interpStep ?fillMode=FILL?");

    /* Check for recursive invocation of IPT_BlendFunct. */
    if (sRecursive) {
        UT_ErrSetNum (UT_ErrNotSupported, str_warp_recursion);
        return UT_False;
    }

    /* Get hold of the parameters passed by the Tcl interpreter. */
    IPT_GetFixImage        (1, srcImg1);
    IPT_GetFixImage        (2, srcImg2);
    UTT_GetFixFunct        (3, d1Funct,  Tmpsize);
    UTT_GetFixFunct        (4, d1Deriv,  Tmpsize);
    UTT_GetFixFunct        (5, d2Funct,  Tmpsize);
    UTT_GetFixFunct        (6, d2Deriv,  Tmpsize);
    UTT_GetFixFunct        (7, mixFunct, Tmpsize);
    UTT_GetFixInt32        (8, interpStep);
    IPT_GetOptFillModeEnum (9, fillMode, IP_FillModeFill);

    /* Run the image warping algorithm. */
    sRecursive = UT_True;
    myInterp = interp;
    success = IP_BlendFunct (srcImg1, srcImg2, destImg, CallFunc1, CallFuncDeriv1,
                             CallFunc2, CallFuncDeriv2, CallMixFunc, interpStep, fillMode);
    sRecursive = UT_False;
    return success;
}

UT_Bool IPT_WarpKeypoint (ELEMPARAMLIST)
{
    IP_ImageId      destImg = (IP_ImageId) clientData;
    IP_ImageId      srcImg;
    UT_Bool         retVal;
    Int32           numKeys, interpStep,
                    nxsrc, nysrc, nxdest, nydest;
    double          smoothness;
    IP_FillModeType fillMode;
    Float32         *xsrcList, *ysrcList, *xdestList, *ydestList;

    UTT_ParamHelp ("$destImg WarpKeypoint srcImg numKeys xsrcList ysrcList xdestList ydestList interpStep ?smoothness=0.0? ?fillMode=FILL?");

    IPT_GetFixImage        (1, srcImg);
    UTT_GetFixInt32        (2, numKeys);
    UTT_GetFixFloat32List  (3, xsrcList,  nxsrc,  numKeys);
    UTT_GetFixFloat32List  (4, ysrcList,  nysrc,  numKeys);
    UTT_GetFixFloat32List  (5, xdestList, nxdest, numKeys);
    UTT_GetFixFloat32List  (6, ydestList, nydest, numKeys);
    UTT_GetFixInt32        (7, interpStep);
    UTT_GetOptFloat64      (8, smoothness, 0.0);
    IPT_GetOptFillModeEnum (9, fillMode, IP_FillModeFill);
    retVal = IP_WarpKeypoint
        (srcImg, destImg, numKeys, xsrcList, ysrcList, xdestList, ydestList, interpStep, smoothness, fillMode);
    UT_MemFree (xsrcList);
    UT_MemFree (ysrcList);
    UT_MemFree (xdestList);
    UT_MemFree (ydestList);
    return retVal;
}

UT_Bool IPT_BlendKeypoint (ELEMPARAMLIST)
{
    IP_ImageId      destImg = (IP_ImageId) clientData;
    IP_ImageId      srcImg1, srcImg2;
    UT_Bool         retVal;
    Int32           numKeys, interpStep,
                    nxsrc1, nysrc1, nxsrc2, nysrc2, nmix;
    double          smoothness;
    IP_FillModeType fillMode;
    Float32         *xsrc1List, *ysrc1List, *xsrc2List, *ysrc2List, *mixList;

    UTT_ParamHelp ("$destImg BlendKeypoint srcImg1 srcImg2 numKeys xsrc1List ysrc1List xsrc2List ysrc2List mixList interpStep ?smoothness=0.0? ?fillMode=FILL?");

    IPT_GetFixImage        ( 1, srcImg1);
    IPT_GetFixImage        ( 2, srcImg2);
    UTT_GetFixInt32        ( 3, numKeys);
    UTT_GetFixFloat32List  ( 4, xsrc1List, nxsrc1, numKeys);
    UTT_GetFixFloat32List  ( 5, ysrc1List, nysrc1, numKeys);
    UTT_GetFixFloat32List  ( 6, xsrc2List, nxsrc2, numKeys);
    UTT_GetFixFloat32List  ( 7, ysrc2List, nysrc2, numKeys);
    UTT_GetFixFloat32List  ( 8, mixList,   nmix,   numKeys);
    UTT_GetFixInt32        ( 9, interpStep);
    UTT_GetOptFloat64      (10, smoothness, 0.0);
    IPT_GetOptFillModeEnum (11, fillMode,   IP_FillModeFill);
    retVal = IP_BlendKeypoint
             (srcImg1, srcImg2, destImg, numKeys,
              xsrc1List, ysrc1List, xsrc2List, ysrc2List, mixList, interpStep, smoothness, fillMode);
    UT_MemFree (xsrc1List);
    UT_MemFree (ysrc1List);
    UT_MemFree (xsrc2List);
    UT_MemFree (ysrc2List);
    UT_MemFree (mixList);
    return retVal;
}
