/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IPT_TclLib.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The interface between the image
 *                      library and the Tcl interpreter.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IPT_InitTclImgLib
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"

#include "UT_Error.h"
#include "UT_Memory.h"
#include "FF_Image.h"
#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#include <tcl.h>
#include "UTT_TclIf.h"
#include "IPT_TclIf.h"
#include "IPT_ImagePrivate.h"

/* Default values of states. */
FF_ImgFmtType   IPT_StateFormatDefault[FF_NumImgChanTypes];
UT_Bool         IPT_StateDrawMaskDefault[FF_NumImgChanTypes];
Float32         IPT_StateDrawColorDefault[FF_NumImgChanTypes];
IP_DrawModeType IPT_StateDrawModeDefault[FF_NumImgChanTypes];

static UT_Bool sInitialized = UT_False;
static UInt32  sImgNum = 0;

static Tcl_HashTable sTclImgTable;

static UTT_TclConstInt32 sImgConsts[] = {
    /* Channel types */
    {FF_ImgChanTypeBrightness,  "BRIGHTNESS"},
    {FF_ImgChanTypeRed,         "RED"},
    {FF_ImgChanTypeGreen,       "GREEN"},
    {FF_ImgChanTypeBlue,        "BLUE"},
    {FF_ImgChanTypeMatte,       "MATTE"},
    {FF_ImgChanTypeRedMatte,    "REDMATTE"},
    {FF_ImgChanTypeGreenMatte,  "GREENMATTE"},
    {FF_ImgChanTypeBlueMatte,   "BLUEMATTE"},
    {FF_ImgChanTypeHoriSnr,     "HNORMAL"},
    {FF_ImgChanTypeVertSnr,     "VNORMAL"},
    {FF_ImgChanTypeDepth,       "DEPTH"},
    {FF_ImgChanTypeTemperature, "TEMPERATURE"},
    {FF_ImgChanTypeRadiance,    "RADIANCE"},
    {FF_NumImgChanTypes,        "NUMCHAN"},

    /* Format types */
    {FF_ImgFmtTypeNone,         "NONE"},
    {FF_ImgFmtTypeUByte,        "UBYTE"},
    {FF_ImgFmtTypeFloat,        "FLOAT"},

    /* Drawing modes */
    {IP_DrawModeReplace,        "REPLACE"},
    {IP_DrawModeAdd,            "ADD"},
    {IP_DrawModeSub,            "SUB"},
    {IP_DrawModeXor,            "XOR"},

    /* Fill modes for blending and warping */
    {IP_FillModeFill,           "FILL"},
    {IP_FillModeWrap,           "WRAP"},
    {IP_FillModeClip,           "CLIP"},

    {IP_UnlaceModeExtract,      "EXTRACT"},
    {IP_UnlaceModeReplicate,    "REPLICATE"},
    {IP_UnlaceModeInterpolate,  "INTERPOLATE"}
};

static UTT_TclElemFunc sImgStateFuncs[] = {
    {IPT_GetChannelFormat,       "GetChannelFormat"},
    {IPT_GetChannelFormatName,   "GetChannelFormatName"},
    {IPT_GetChannelType,         "GetChannelType"},
    {IPT_GetChannelTypeName,     "GetChannelTypeName"},
    {IPT_GetColorCorrect,        "GetColorCorrect"},
    {IPT_GetDrawColor,           "GetDrawColor"},
    {IPT_GetDrawMask,            "GetDrawMask"},
    {IPT_GetDrawMode,            "GetDrawMode"},
    {IPT_GetDrawModeType,        "GetDrawModeType"},
    {IPT_GetDrawModeTypeName,    "GetDrawModeTypeName"},
    {IPT_GetFileFormat,          "GetFileFormat"},
    {IPT_GetFileInfo,            "GetFileInfo"},
    {IPT_GetFillModeType,        "GetFillModeType"},
    {IPT_GetFillModeTypeName,    "GetFillModeTypeName"},
    {IPT_GetFormat,              "GetFormat"},
    {IPT_GetNumThreads,          "GetNumThreads"},
    {IPT_GetTextScale,           "GetTextScale"},
    {IPT_GetTextSize,            "GetTextSize"},
    {IPT_GetUnlaceModeType,      "GetUnlaceModeType"},
    {IPT_GetUnlaceModeTypeName,  "GetUnlaceModeTypeName"},
    {IPT_HasPhotoSupport,        "HasPhotoSupport"},
    {IPT_MemCheck,               "MemCheck"},
    {IPT_PopState,               "PopState"},
    {IPT_PushState,              "PushState"},
    {IPT_SetColorCorrect,        "SetColorCorrect"},
    {IPT_SetDrawColor,           "SetDrawColor"},
    {IPT_SetDrawMask,            "SetDrawMask"},
    {IPT_SetDrawMode,            "SetDrawMode"},
    {IPT_SetFormat,              "SetFormat"},
    {IPT_SetNumThreads,          "SetNumThreads"},
    {IPT_SetTextScale,           "SetTextScale"}
};

static UTT_TclObjFunc sImgFuncs[] = {
    {IPT_NewImage,              "NewImage"},
    {IPT_NewImageFromFile,      "NewImageFromFile"},
    {IPT_NewImageFromPhoto,     "NewImageFromPhoto"}
};

static UTT_TclElemFunc sImgElemFuncs[] = {
    {IPT_AddChannel,              "AddChannel"},
    {IPT_AsByteArray,             "AsByteArray"},
    {IPT_AsPhoto,                 "AsPhoto"},
    {IPT_Blank,                   "Blank"},
    {IPT_BlankRect,               "BlankRect"},
    {IPT_BlendFunct,              "BlendFunct"},
    {IPT_BlendKeypoint,           "BlendKeypoint"},
    {IPT_Blur,                    "Blur"},
    {IPT_ChangeChannelGamma,      "ChangeChannelGamma"},
    {IPT_CompositeColorMatte,     "CompositeColorMatte"},
    {IPT_CompositeDepth,          "CompositeDepth"},
    {IPT_CompositeMatte,          "CompositeMatte"},
    {IPT_ComputeChannelDerivates, "ComputeChannelDerivates"},
    {IPT_CopyChannel,             "CopyChannel"},
    {IPT_CopyImage,               "CopyImage"},
    {IPT_CopyRect,                "CopyRect"},
    {IPT_CorrelateChannel,        "CorrelateChannel"},
    {IPT_CreateChannelNoise,      "CreateChannelNoise"},
    {IPT_CrossDissolve,           "CrossDissolve"},
    {IPT_CutOff,                  "CutOff"},
    {IPT_DeleteChannel,           "DeleteChannel"},
    {IPT_DepthBlur,               "DepthBlur"},
    {IPT_DiffChannel,             "DiffChannel"},
    {IPT_DifferenceImage,         "DifferenceImage"},
    {IPT_Dilatation,              "Dilatation"},
    {IPT_DrawAALine,              "DrawAALine"},
    {IPT_DrawAAPixel,             "DrawAAPixel"},
    {IPT_DrawAAText,              "DrawAAText"},
    {IPT_DrawBrush,               "DrawBrush"},
    {IPT_DrawBrushLine,           "DrawBrushLine"},
    {IPT_DrawCircle,              "DrawCircle"},
    {IPT_DrawEllipse,             "DrawEllipse"},
    {IPT_DrawLine,                "DrawLine"},
    {IPT_DrawPixel,               "DrawPixel"},
    {IPT_DrawRect,                "DrawRect"},
    {IPT_DrawText,                "DrawText"},
    {IPT_Erosion,                 "Erosion"},
    {IPT_FindChannelPattern,      "FindChannelPattern"},
    {IPT_Flicker,                 "Flicker"},
    {IPT_FlipDiagonal,            "FlipDiagonal"},
    {IPT_FlipHorizontal,          "FlipHorizontal"},
    {IPT_FlipVertical,            "FlipVertical"},
    {IPT_GetChannelHistogram,     "GetChannelHistogram"},
    {IPT_GetChannelRange,         "GetChannelRange"},
    {IPT_GetChannelStats,         "GetChannelStats"},
    {IPT_GetImageFormat,          "GetImageFormat"},
    {IPT_GetImageInfo,            "GetImageInfo"},
    {IPT_GetPixel,                "GetPixel"},
    {IPT_GetSample,               "GetSample"},
    {IPT_HasChannel,              "HasChannel"},
    {IPT_Interlace,               "Interlace"},
    {IPT_JuliaSet,                "JuliaSet"},
    {IPT_Mandelbrot,              "Mandelbrot"},
    {IPT_MarkNonZeroPixels,       "MarkNonZeroPixels"},
    {IPT_Median,                  "Median"},
    {IPT_MedianSequence,          "MedianSequence"},
    {IPT_MergeBrush,              "MergeBrush"},
    {IPT_ReadImage,               "ReadImage"},
    {IPT_RemapChannelValues,      "RemapChannelValues"},
    {IPT_Rotate90,                "Rotate90"},
    {IPT_ScaleRect,               "ScaleRect"},
    {IPT_ShearHorizontal,         "ShearHorizontal"},
    {IPT_ShearVertical,           "ShearVertical"},
    {IPT_Stamp,                   "Stamp"},
    {IPT_Threshold,               "Threshold"},
    {IPT_Unlace,                  "Unlace"},
    {IPT_WarpFunct,               "WarpFunct"},
    {IPT_WarpKeypoint,            "WarpKeypoint"},
    {IPT_WrapRect,                "WrapRect"},
    {IPT_WriteImage,              "WriteImage"},
    {IPT_WriteSimple,             "WriteSimple"}
};

static UInt32 GetNextImgNum (void)
{
    sImgNum++;
    return sImgNum;
}

static void DeleteImage (ClientData clientData)
{
    IP_ImageId image = (IP_ImageId) clientData;
 
    IP_DeleteImage (image);
    return;
}

static int ElementCmd (TCLPARAMLIST)
{
    Int32 i;

    if (argc < 2) {
        Tcl_AppendResult (interp, "Available poImage options:\n", (char *)NULL); 
        for (i=0; i< sizeof (sImgElemFuncs) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, "  ", sImgElemFuncs[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }
    if (argc == 2 && argv[1][0] == '?') {
        for (i=0; i< sizeof (sImgElemFuncs) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, sImgElemFuncs[i].name, " ", (char *)NULL);
        }
        return TCL_OK;
    }

    for (i=0; i< sizeof (sImgElemFuncs) / sizeof (UTT_TclElemFunc); i++) {
        if (strcmp (argv[1], sImgElemFuncs[i].name) == 0) {
            if (!sImgElemFuncs[i].func (clientData, interp, argc, argv)) {
                if (strcmp (Tcl_GetStringResult (interp), "") == 0) {
                    Tcl_AppendResult (interp, UT_ErrStr, " (", UT_ErrAddStr, ")", (char *)NULL); 
                }
                return TCL_ERROR;
            } else {
                return TCL_OK;
            }
        }
    }

    Tcl_AppendResult (interp, "Bad poImage option \"", argv[1], "\". Must be:\n", (char *)NULL); 
    for (i=0; i< sizeof (sImgElemFuncs) / sizeof (UTT_TclElemFunc); i++) {
        Tcl_AppendResult (interp, "  ", sImgElemFuncs[i].name, "\n", (char *)NULL);
    }
    return TCL_ERROR;
}

static int ImageCmd (TCLPARAMLIST)
{
    Int32 i;
    UT_Bool found = UT_False;
    IP_ImageId image = NULL;

    if (argc < 2) {
        Tcl_AppendResult (interp, "Available poImage subcommands:\n", (char *)NULL);
        for (i=0; i< sizeof (sImgFuncs) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, "  ", sImgFuncs[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }
    if (argc == 2 && argv[1][0] == '?') {
        for (i=0; i< sizeof (sImgFuncs) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, sImgFuncs[i].name, " ", (char *)NULL);
        }
        return TCL_OK;
    }

    for (i=0; i< sizeof (sImgFuncs) / sizeof (UTT_TclObjFunc); i++) {
        if (strcmp (argv[1], sImgFuncs[i].name) == 0) {
            if (!sImgFuncs[i].func ((ClientData *)&image, interp, argc, argv)) {
                if (strcmp (Tcl_GetStringResult (interp), "") == 0) {
                    Tcl_AppendResult (interp, UT_ErrStr, " (", UT_ErrAddStr, ")", (char *)NULL);
                }
                return TCL_ERROR;
            } else {
                found = UT_True;
                break;
            }
        }
    }
    if (!found) {
        Tcl_AppendResult (interp, "Bad poImage subcommand \"", argv[1], "\". Must be:\n", (char *)NULL);
        for (i=0; i< sizeof (sImgFuncs) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, "  ", sImgFuncs[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }

    if (image) {
        if( ! IPT_CreateTclCmd (interp, image)) {
            return TCL_ERROR;
        }
    }
    return TCL_OK;
}

static int ImageModeCmd (TCLPARAMLIST)
{
    Int32 i;
    UT_Bool found = UT_False;

    if (argc < 2) {
        Tcl_AppendResult (interp, "Available poImageState options:\n", (char *)NULL); 
        for (i=0; i< sizeof (sImgStateFuncs) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, "  ", sImgStateFuncs[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }
    if (argc == 2 && argv[1][0] == '?') {
        for (i=0; i< sizeof (sImgStateFuncs) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, sImgStateFuncs[i].name, " ", (char *)NULL);
        }
        return TCL_OK;
    }

    for (i=0; i< sizeof (sImgStateFuncs) / sizeof (UTT_TclElemFunc); i++) {
        if (strcmp (argv[1], sImgStateFuncs[i].name) == 0) {
            if (!sImgStateFuncs[i].func ((ClientData)NULL, interp, argc, argv)) {
                if (strcmp (Tcl_GetStringResult (interp), "") == 0) {
                    Tcl_AppendResult (interp, UT_ErrStr, " (", UT_ErrAddStr, ")", (char *)NULL); 
                }
                return TCL_ERROR;
            } else {
                found = UT_True;
                break;
            }
        }
    }
    if (!found) {
        Tcl_AppendResult (interp, "Bad poImageState option \"", argv[1], "\". Must be:\n", (char *)NULL);
        for (i=0; i< sizeof (sImgStateFuncs) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, "  ", sImgStateFuncs[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }
    return TCL_OK;
}

static int SetInt32Global (Tcl_Interp *interp, char *varName, int value)
{
    char valStr[20];
    sprintf (valStr, "%d", value);
    if (!Tcl_SetVar (interp, varName, valStr, TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG)) {
        return TCL_ERROR;
    }
    return TCL_OK;
}

static int CreateConstants (Tcl_Interp *interp)
{
    Int32 i;

    for (i=0; i< sizeof (sImgConsts) / sizeof (UTT_TclConstInt32); i++) {
        if (SetInt32Global (interp, sImgConsts[i].name, sImgConsts[i].value) != TCL_OK) {
            return TCL_ERROR;
        }
    }
    return TCL_OK;
}

int IPT_GetImage (Tcl_Interp *interp, const char *varValue, IP_ImageId *img)
{
    Tcl_HashEntry *hashEntryPtr;
    IP_ImageId image;

    hashEntryPtr = Tcl_FindHashEntry (&sTclImgTable, varValue);
    if (!hashEntryPtr) {
        Tcl_AppendResult (interp, "No image named ", varValue, (char *)NULL);
        return TCL_ERROR;
    }
    image = (IP_ImageId)Tcl_GetHashValue (hashEntryPtr);
    if (!image) {
        return TCL_ERROR;
    }
    *img = image;
    return TCL_OK;
}

int IPT_GetFormatEnum (Tcl_Interp *interp, const char *varValue, FF_ImgFmtType *fmt)
{
    int retVal;
    int tmpVal;
    FF_ImgFmtType fmtVal;

    retVal = Tcl_GetInt (interp, varValue, &tmpVal);
    if (retVal != TCL_OK) {
        if (!IP_GetChannelFormat (varValue, &fmtVal)) {
            Tcl_AppendResult (interp, "Invalid format ", varValue, " specified", (char *)NULL);
            return TCL_ERROR;
        } else {
            Tcl_ResetResult (interp);
            tmpVal = fmtVal;
        }
    }
    *fmt = tmpVal;
    return TCL_OK;
}

int IPT_GetChannelEnum (Tcl_Interp *interp, const char *varValue, FF_ImgChanType *chan)
{
    int retVal;
    int tmpVal;
    FF_ImgChanType chanVal;

    retVal = Tcl_GetInt (interp, varValue, &tmpVal);
    if (retVal != TCL_OK) {
        if (!FF_ImgGetChannelType (varValue, &chanVal)) {
            Tcl_AppendResult (interp, "Invalid channel ", varValue, " specified", (char *)NULL);
            return TCL_ERROR;
        } else {
            Tcl_ResetResult (interp);
            tmpVal = chanVal;
        }
    }
    *chan = tmpVal;
    return TCL_OK;
}

int IPT_GetDrawModeEnum (Tcl_Interp *interp, const char *varValue, IP_DrawModeType *drawMode)
{
    int retVal;
    int tmpVal;
    IP_DrawModeType drawModeVal;

    retVal = Tcl_GetInt (interp, varValue, &tmpVal);
    if (retVal != TCL_OK) {
        if (!IP_GetDrawModeType (varValue, &drawModeVal)) {
            Tcl_AppendResult (interp, "Invalid draw mode ", varValue, " specified", (char *)NULL);
            return TCL_ERROR;
        } else {
            Tcl_ResetResult (interp);
            tmpVal = drawModeVal;
        }
    }
    *drawMode = tmpVal;
    return TCL_OK;
}

int IPT_GetFillModeEnum (Tcl_Interp *interp, const char *varValue, IP_FillModeType *fillMode)
{
    int retVal;
    int tmpVal;
    IP_FillModeType fillModeVal;

    retVal = Tcl_GetInt (interp, varValue, &tmpVal);
    if (retVal != TCL_OK) {
        if (!IP_GetFillModeType (varValue, &fillModeVal)) {
            Tcl_AppendResult (interp, "Invalid fill mode ", varValue, " specified", (char *)NULL);
            return TCL_ERROR;
        } else {
            Tcl_ResetResult (interp);
            tmpVal = fillModeVal;
        }
    }
    *fillMode = tmpVal;
    return TCL_OK;
}

int IPT_GetUnlaceModeEnum (Tcl_Interp *interp, const char *varValue, IP_UnlaceModeType *unlaceMode)
{
    int retVal;
    int tmpVal;
    IP_UnlaceModeType unlaceModeVal;

    retVal = Tcl_GetInt (interp, varValue, &tmpVal);
    if (retVal != TCL_OK) {
        if (!IP_GetUnlaceModeType (varValue, &unlaceModeVal)) {
            Tcl_AppendResult (interp, "Invalid unlace mode ", varValue, " specified", (char *)NULL);
            return TCL_ERROR;
        } else {
            Tcl_ResetResult (interp);
            tmpVal = unlaceModeVal;
        }
    }
    *unlaceMode = tmpVal;
    return TCL_OK;
}

UT_Bool IPT_CreateTclCmd (Tcl_Interp *interp, IP_ImageId img)
{
    char imgName[20];
    Tcl_HashEntry *hashEntryPtr;
    int newEntryFlag;

    sprintf (imgName, "poImage%d", GetNextImgNum());
    hashEntryPtr = Tcl_CreateHashEntry (&sTclImgTable, imgName, &newEntryFlag);
    if (newEntryFlag == 0) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Hash entry for %s already exists", imgName));
        return UT_False;
    }
    if (hashEntryPtr == NULL) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Could not create hash table entry for %s", imgName));
        return UT_False;
    }
    Tcl_SetHashValue (hashEntryPtr, img);
    Tcl_CreateCommand (interp, imgName, (Tcl_CmdProc *)ElementCmd,
                       (ClientData) img, (Tcl_CmdDeleteProc *) DeleteImage);
    Tcl_SetResult (interp, imgName, TCL_VOLATILE);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IPT_InitTclImgLib
 *
 *      Usage:          Initialize the Tcl image library.
 *
 *      Synopsis:       int IPT_InitTclImgLib (Tcl_Interp *interp)
 *
 *      Description:    Make the image data types and functions
 *                      available to the Tcl interpreter.
 *
 *      Return value:   TCL_OK if successful, else TCL_ERROR.
 *
 *      See also:       IPT_CreateTclCmd
 *
 ***************************************************************************/

int IPT_InitTclImgLib (Tcl_Interp *interp)
{
    if (sInitialized) {
        return TCL_OK;
    }

    memcpy (IPT_StateFormatDefault,    IP_StateFormat,         sizeof (IP_StateFormat));
    memcpy (IPT_StateDrawMaskDefault,  IP_StateDrawMask,       sizeof (IP_StateDrawMask));
    memcpy (IPT_StateDrawColorDefault, IP_StateDrawColorFloat, sizeof (IP_StateDrawColorFloat));
    memcpy (IPT_StateDrawModeDefault,  IP_StateDrawMode,       sizeof (IP_StateDrawMode));

    Tcl_InitHashTable (&sTclImgTable, TCL_STRING_KEYS);

    Tcl_CreateCommand (interp, "poImage", (Tcl_CmdProc *) ImageCmd,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand (interp, "poImageState", (Tcl_CmdProc *) ImageModeCmd,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    CreateConstants (interp);
    sInitialized = UT_True;
    return TCL_OK;
}
