/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_TkPhoto.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions for transfering poSoft images into Tk photo
 *                      images and vice versa.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_AsPhoto
 *                      IP_NewImageFromPhoto
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <tcl.h>
#ifdef USE_TK_PHOTOS
#include <tk.h>
#endif

#include "UT_Compat.h"

#include "UT_Error.h"
#include "UT_Memory.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#define str_no_photo "poImg was compiled without Tk photo support"

#ifdef USE_TK_PHOTOS

static UT_Bool gtableFloat (Float32 gamma, Int32 n, Float32 table[])
{
    Int32 i;

    for (i = 0; i < n - 1; ++i) {
        table[i] = pow ((Float32) i / (Float32) (n - 2), 1.0 / gamma);
    }
    table[n - 1] = 1.0;
    return UT_True;
}

static UT_Bool img_photo (Tcl_Interp *interp, IP_ImageId img, const char *photoName,
                          FF_ImgChanType chanMap[4],
                          Float32 gamma, Float32 minVal, Float32 maxVal)
{
    Int32       x, y, chan, numChans;
    FF_ImgDesc  *pimg;
    Tk_PhotoHandle photo;
    Tk_PhotoImageBlock photoCont;
    unsigned char *pPtr, *imgBuf;
    UT_MemState memstate;

    #if defined (DEBUG_LOCAL)
        printf ("Using channel mapping: %d %d %d %d\n",
                chanMap[0], chanMap[1], chanMap[2], chanMap[3]);
    #endif

    pimg = &img->pimg;

    /* Check, if any of the channel map values exceeds the number of channel types. */
    for (chan = 0; chan < 4; chan++) {
        if (chanMap[chan] >= FF_NumImgChanTypes) {
            Tcl_SetObjResult(interp,
                Tcl_ObjPrintf("Invalid channel mapping (%d, %d, %d, %d). Value at index %d too large.",
                chanMap[0], chanMap[1], chanMap[2], chanMap[3], chan));
            return UT_False;
        }
    }
    /* At least the first three channel map values must be valid and existent in the source image. */
    if (chanMap[0] == FF_ImgChanTypeNone ||
        chanMap[1] == FF_ImgChanTypeNone || 
        chanMap[2] == FF_ImgChanTypeNone ||
        pimg->channel[chanMap[0]] == FF_ImgFmtTypeNone ||
        pimg->channel[chanMap[1]] == FF_ImgFmtTypeNone || 
        pimg->channel[chanMap[2]] == FF_ImgFmtTypeNone ) {
        Tcl_SetObjResult(interp,
            Tcl_ObjPrintf("Invalid channel mapping (%d, %d, %d, %d). Indices 0, 1 and 2 must be greater or equal to zero.",
            chanMap[0], chanMap[1], chanMap[2], chanMap[3]));
        return UT_False;
    }
    if (chanMap[3] != FF_ImgChanTypeNone && pimg->channel[chanMap[3]] == FF_ImgFmtTypeNone) {
        Tcl_SetObjResult(interp,
            Tcl_ObjPrintf("Invalid channel mapping (%d, %d, %d, %d). Source image does not have channel type %d.",
            chanMap[0], chanMap[1], chanMap[2], chanMap[3], chanMap[3]));
        return UT_False;
    }
    numChans = (chanMap[3] != FF_ImgChanTypeNone ? 4: 3);

    if (!(photo = Tk_FindPhoto (interp, (char *)photoName))) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Photo image \"%s\" does not exist.", photoName));
        return UT_False;
    }

    if (gamma < 0.0) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf(str_notless, str_gamma, 0.0));
        return UT_False;
    }

    if (Tk_PhotoSetSize (interp, photo, pimg->desc.width, pimg->desc.height) != TCL_OK) {
        return UT_False;
    }

    memstate = UT_MemRemember ();
    /* Transfer a complete image at a time with PhotoPutBlock. */
    if (!(imgBuf = UT_MemTempArray (pimg->desc.width * pimg->desc.height * numChans, unsigned char))) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    photoCont.pixelSize = numChans;
    photoCont.pitch = pimg->desc.width * numChans;
    photoCont.width = pimg->desc.width;
    photoCont.height = pimg->desc.height;
    photoCont.offset[0] = 0;
    photoCont.offset[1] = 1;
    photoCont.offset[2] = 2;
    photoCont.offset[3] = (numChans == 3? 0: 3);

    photoCont.pixelPtr = imgBuf;

    if (pimg->channel[chanMap[0]] == FF_ImgFmtTypeFloat &&
        pimg->channel[chanMap[1]] == FF_ImgFmtTypeFloat &&
        pimg->channel[chanMap[2]] == FF_ImgFmtTypeFloat ) {
        /* All channels are in FLOAT format. */
        Float32 *fSrc0 = NULL, *fSrc1 = NULL, *fSrc2 = NULL, *fSrc3 = NULL;
        Float32 *fDst;
        Float32 *fBuf;
        Float32 gtable[GTABSIZE];
        Float32 minValTmp;
        Float32 maxValTmp;
        Float32 m[4] = { 0.0 };
        Float32 t[4] = { 0.0 };

        for (chan = 0; chan < 4; chan++) {
            if (numChans == 3 && chan == 3) {
                break;
            }
            if (minVal < 0.0 && maxVal < 0.0 && minVal == maxVal) {
                if (!IP_GetChannelRange (img, chanMap[chan], -1, -1, -1, -1, &minValTmp, &maxValTmp)) {
                    UT_MemRestore (memstate);
                    return UT_False;
                }
                m[chan] = (1.0 - 0.0) / (maxValTmp - minValTmp);
                t[chan] = 0.0 - m[chan] * minValTmp;
            } else {
                m[chan] = (1.0 - 0.0) / (maxVal - minVal);
                t[chan] = 0.0 - m[chan] * minVal;
            }
        }

        if (!(fBuf = UT_MemTempArray (pimg->desc.width * numChans, Float32))) {
            UT_MemRestore (memstate);
            return UT_False;
        }
        gtableFloat (gamma, GTABSIZE, gtable);

        pPtr = imgBuf;
        if (numChans == 3) {
            for (y = pimg->desc.height-1; y >= 0; y--) {
                fSrc0 = FF_ImgFloatRow (pimg, chanMap[0], y);
                fSrc1 = FF_ImgFloatRow (pimg, chanMap[1], y);
                fSrc2 = FF_ImgFloatRow (pimg, chanMap[2], y);
                fDst  = fBuf;
                for (x = 0; x < pimg->desc.width; x++) {
                    *(fDst++) = *(fSrc0++) * m[0] + t[0];
                    *(fDst++) = *(fSrc1++) * m[1] + t[1];
                    *(fDst++) = *(fSrc2++) * m[2] + t[2];
                }
                IP_GammaFloat2UByte (numChans * pimg->desc.width, fBuf, gtable, pPtr);
                pPtr += numChans * pimg->desc.width;
            }
        } else {
            for (y = pimg->desc.height-1; y >= 0; y--) {
                fSrc0 = FF_ImgFloatRow (pimg, chanMap[0], y);
                fSrc1 = FF_ImgFloatRow (pimg, chanMap[1], y);
                fSrc2 = FF_ImgFloatRow (pimg, chanMap[2], y);
                fSrc3 = FF_ImgFloatRow (pimg, chanMap[3], y);
                fDst  = fBuf;
                for (x = 0; x < pimg->desc.width; x++) {
                    *(fDst++) = *(fSrc0++) * m[0] + t[0];
                    *(fDst++) = *(fSrc1++) * m[1] + t[1];
                    *(fDst++) = *(fSrc2++) * m[2] + t[2];
                    *(fDst++) = *(fSrc3++) * m[3] + t[3];
                }
                IP_GammaFloat2UByte (numChans * pimg->desc.width, fBuf, gtable, pPtr);
                pPtr += numChans * pimg->desc.width;
            }
        }
    } else if (pimg->channel[chanMap[0]] == FF_ImgFmtTypeUByte &&
               pimg->channel[chanMap[1]] == FF_ImgFmtTypeUByte &&
               pimg->channel[chanMap[2]] == FF_ImgFmtTypeUByte) {
        /* All channels are in UBYTE format. */
        UInt8 * redSrc, *greenSrc, *blueSrc, *matteSrc;
        pPtr = imgBuf;
        if (numChans == 3) {
            for (y = pimg->desc.height-1; y >= 0; y--) {
                redSrc   = FF_ImgUByteRow (pimg, chanMap[0], y);
                greenSrc = FF_ImgUByteRow (pimg, chanMap[1], y);
                blueSrc  = FF_ImgUByteRow (pimg, chanMap[2], y);
                for (x = 0; x < pimg->desc.width; x++) {
                    *(pPtr++) = *(redSrc++);
                    *(pPtr++) = *(greenSrc++);
                    *(pPtr++) = *(blueSrc++);
                }
            }
        } else {
            for (y = pimg->desc.height-1; y >= 0; y--) {
                redSrc   = FF_ImgUByteRow (pimg, chanMap[0], y);
                greenSrc = FF_ImgUByteRow (pimg, chanMap[1], y);
                blueSrc  = FF_ImgUByteRow (pimg, chanMap[2], y);
                matteSrc = FF_ImgUByteRow (pimg, chanMap[3], y);
                for (x = 0; x < pimg->desc.width; x++) {
                    *(pPtr++) = *(redSrc++);
                    *(pPtr++) = *(greenSrc++);
                    *(pPtr++) = *(blueSrc++);
                    *(pPtr++) = *(matteSrc++);
                }
            }
        }
    } else {
        Tcl_SetObjResult(interp, 
            Tcl_ObjPrintf("Invalid channel types (%d, %d, %d, %d).",
            chanMap[0], chanMap[1], chanMap[2], chanMap[3]));
        UT_MemRestore (memstate);
        return UT_False;
    }

    if (Tk_PhotoPutBlock (interp, photo, &photoCont, 0, 0,
                          pimg->desc.width, pimg->desc.height, 
                          numChans == 3? TK_PHOTO_COMPOSITE_SET: TK_PHOTO_COMPOSITE_OVERLAY) != TCL_OK) {
        UT_MemRestore (memstate);
        return UT_False;
    }

    UT_MemRestore (memstate);
    return UT_True;
}

static IP_ImageId photo_img (Tcl_Interp *interp, const char *photoName)
{
    Int32              x, y, w, h, numChans;
    UInt8              *redDest, *greenDest, *blueDest, *matteDest;
    FF_ImgDesc         *pimg;
    IP_ImageId         img;
    FF_ImgFmtType      pixfmt[FF_NumImgChanTypes] = { FF_ImgFmtTypeNone };
    FF_ImgFmtType      savepixfmt[FF_NumImgChanTypes];
    Tk_PhotoHandle     photo;
    Tk_PhotoImageBlock photoCont;
    unsigned char      *pPtr;

    if (!(photo = Tk_FindPhoto (interp, (char *)photoName))) {
        Tcl_SetObjResult(interp, Tcl_ObjPrintf("Photo image \"%s\" does not exist.", photoName));
        return NULL;
    }
    Tk_PhotoGetSize (photo, &w, &h);
    Tk_PhotoGetImage (photo, &photoCont);
    numChans = photoCont.pixelSize;
    w = photoCont.width;
    h = photoCont.height;
    pPtr = photoCont.pixelPtr;

    /* Only red, green, blue and alpha channels are currently supported. */
    IP_GetFormat (savepixfmt);
    pixfmt[(int)FF_ImgChanTypeRed]   = FF_ImgFmtTypeUByte;
    pixfmt[(int)FF_ImgChanTypeGreen] = FF_ImgFmtTypeUByte;
    pixfmt[(int)FF_ImgChanTypeBlue]  = FF_ImgFmtTypeUByte;
    if ( numChans > 3 ) {
        pixfmt[(int)FF_ImgChanTypeMatte] = FF_ImgFmtTypeUByte;
    }
    IP_SetFormat (pixfmt);

    if (!(img = IP_NewImage (w, h, (float)w / (float)h, NULL))) {
        IP_SetFormat (savepixfmt);
        return NULL;
    }

    IP_SetFormat (savepixfmt);
    pimg = &img->pimg;

    if (numChans == 3) {
        for (y = h-1; y >= 0; y--) {
            redDest   = FF_ImgUByteRow (pimg, FF_ImgChanTypeRed,   y);
            greenDest = FF_ImgUByteRow (pimg, FF_ImgChanTypeGreen, y);
            blueDest  = FF_ImgUByteRow (pimg, FF_ImgChanTypeBlue,  y);
            for (x = 0; x < w; x++) {
                *redDest++   = *pPtr++;
                *greenDest++ = *pPtr++;
                *blueDest++  = *pPtr++;
            }
        }
    } else {
        for (y = h-1; y >= 0; y--) {
            redDest   = FF_ImgUByteRow (pimg, FF_ImgChanTypeRed,   y);
            greenDest = FF_ImgUByteRow (pimg, FF_ImgChanTypeGreen, y);
            blueDest  = FF_ImgUByteRow (pimg, FF_ImgChanTypeBlue,  y);
            matteDest = FF_ImgUByteRow (pimg, FF_ImgChanTypeMatte, y);
            for (x = 0; x < w; x++) {
                *redDest++   = *pPtr++;
                *greenDest++ = *pPtr++;
                *blueDest++  = *pPtr++;
                *matteDest++ = *pPtr++;
            }
        }
    }
    return img;
}

#endif /* USE_TK_PHOTOS */

/***************************************************************************
 *[@e
 *      Name:           IP_AsPhoto
 *
 *      Usage:          Copy an image into a Tk photo image.
 *
 *      Synopsis:       UT_Bool IP_AsPhoto(
 *                              Tcl_Interp *interp,
 *                              IP_ImageId srcImg,
 *                              const char *photoName,
 *                              FF_ImgChanType chanMapList[4],
 *                              Float32 gamma,
 *                              Float32 minValue, Float32 maxValue)
 *
 *      Description:    Copy image "srcImg" into Tk photo image "photoName".
 *
 *                      Notes:
 *                      - To have more advanced floating point mapping,
 *                        ex. "Automatic Gain Control", use poImgUtil::AsPhoto.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_NewImageFromPhoto
 *                      poImgUtil::AsPhoto
 *
 ***************************************************************************/

UT_Bool IP_AsPhoto (Tcl_Interp *interp, IP_ImageId srcImg, const char *photoName,
                    FF_ImgChanType chanMapList[4],
                    Float32 gamma, Float32 minValue, Float32 maxValue)
{
    UT_Bool success = UT_False;

#ifdef USE_TK_PHOTOS
    success = img_photo (interp, srcImg, photoName, chanMapList, gamma, minValue, maxValue);
#else
    UT_ErrSetNum (UT_ErrNotSupported, str_no_photo);
#endif
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_NewImageFromPhoto
 *
 *      Usage:          Create an image from a Tk photo.
 *
 *      Synopsis:       IP_ImageId IP_NewImageFromPhoto(
 *                                 Tcl_Interp *interp,
 *                                 const char *photoName)
 *
 *      Description:    The Tk photo "photoName" is copied into 
 *                      a new image.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: RGB
 *                      Float format: No
 *
 *      Return value:   An identifier for the new image if successful,
 *                      else NULL.
 *
 *      See also:       IP_AsPhoto
 *
 ***************************************************************************/

IP_ImageId IP_NewImageFromPhoto (Tcl_Interp *interp, const char *photoName)
{
    IP_ImageId img = NULL;

#ifdef USE_TK_PHOTOS
    img = photo_img (interp, photoName);
#else
    UT_ErrSetNum (UT_ErrNotSupported, str_no_photo);
#endif
    return img;
}
