/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Comp.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Image compositing functions.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_CompositeMatte
 *                      IP_CompositeColorMatte
 *                      IP_CompositeDepth
 *                      IP_CrossDissolve
 *
 **************************************************************************/

#include <stdio.h>

#include "UT_Compat.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Parallel.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

/* Determine if channel "chn" of image "img" can be used for a fast
   compositing operation, i.e. if the channel is either in FF_ImgFmtTypeNone
   or FF_ImgFmtTypeUByte format.  */

static UT_Bool isfast (IP_ImageId img, Int32 chn)
{
    return
        img->pimg.channel[chn] == FF_ImgFmtTypeNone ||
        img->pimg.channel[chn] == FF_ImgFmtTypeUByte;
}

/* Composite two images using a black and white matte: Decide whether to use
   the fast or the slow compositing method, and invoke the right algorithm. */

static UT_Bool comp_bwmatte
        (IP_ImageId fgimg,
         IP_ImageId bgimg,
         IP_ImageId cmimg,
         Float32 fr, Float32 fg, Float32 fb,
         Float32 br, Float32 bg, Float32 bb,
         IP_ImageId mtimg,
         UT_Bool fast)
{
    /* Make sure that we have a transparency channel. */
    if (mtimg) {
        if (!mtimg->pimg.channel[FF_ImgChanTypeMatte]) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_mt_bwtrans);
            return UT_False;
        }
    } else {
        if (!fgimg->pimg.channel[FF_ImgChanTypeMatte]) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_fg_bwtrans);
            return UT_False;
        }
    }

    /* Invoke the compositing algorithm. */
    if (mtimg) {
        fast &= isfast (mtimg, FF_ImgChanTypeMatte);
    } else {
        fast &= isfast (fgimg, FF_ImgChanTypeMatte);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeBrightness] &&
        cmimg->pimg.channel[FF_ImgChanTypeBrightness]) {
        fast &= isfast (fgimg, FF_ImgChanTypeBrightness);
        fast &= isfast (bgimg, FF_ImgChanTypeBrightness);
        fast &= isfast (cmimg, FF_ImgChanTypeBrightness);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeRed] &&
        cmimg->pimg.channel[FF_ImgChanTypeRed]) {
        fast &= isfast (fgimg, FF_ImgChanTypeRed);
        fast &= isfast (bgimg, FF_ImgChanTypeRed);
        fast &= isfast (cmimg, FF_ImgChanTypeRed);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeGreen] &&
        cmimg->pimg.channel[FF_ImgChanTypeGreen]) {
        fast &= isfast (fgimg, FF_ImgChanTypeGreen);
        fast &= isfast (bgimg, FF_ImgChanTypeGreen);
        fast &= isfast (cmimg, FF_ImgChanTypeGreen);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeBlue] &&
        cmimg->pimg.channel[FF_ImgChanTypeBlue]) {
        fast &= isfast (fgimg, FF_ImgChanTypeBlue);
        fast &= isfast (bgimg, FF_ImgChanTypeBlue);
        fast &= isfast (cmimg, FF_ImgChanTypeBlue);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeMatte] &&
        cmimg->pimg.channel[FF_ImgChanTypeMatte]) {
        fast &= isfast (fgimg, FF_ImgChanTypeMatte);
        fast &= isfast (bgimg, FF_ImgChanTypeMatte);
        fast &= isfast (cmimg, FF_ImgChanTypeMatte);
    }

    return fast?
        IP_CompFastMatte (fgimg, bgimg, cmimg,
                          fr, fg, fb, br, bg, bb, mtimg):
        IP_CompSlowMatte (fgimg, bgimg, cmimg,
                          fr, fg, fb, br, bg, bb, mtimg);
}

/* Composite two images using a colored matte: Decide whether to use the
   fast or the slow compositing method, and invoke the right algorithm. */

static UT_Bool comp_clrmatte
        (IP_ImageId fgimg,
         IP_ImageId bgimg,
         IP_ImageId cmimg,
         Float32 fr, Float32 fg, Float32 fb,
         Float32 br, Float32 bg, Float32 bb,
         IP_ImageId mtimg,
         UT_Bool fast)
{
    /* Make sure that we have red, green and blue transparency channels. */
    if (mtimg) {
        if (!mtimg->pimg.channel[FF_ImgChanTypeRedMatte] ||
            !mtimg->pimg.channel[FF_ImgChanTypeRedMatte] ||
            !mtimg->pimg.channel[FF_ImgChanTypeRedMatte]) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_mt_clrtrans);
            return UT_False;
        }
    } else {
        if (!fgimg->pimg.channel[FF_ImgChanTypeRedMatte] ||
            !fgimg->pimg.channel[FF_ImgChanTypeRedMatte] ||
            !fgimg->pimg.channel[FF_ImgChanTypeRedMatte]) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_fg_clrtrans);
            return UT_False;
        }
    }

    /* Invoke the compositing algorithm. */
    if (mtimg) {
        fast &= isfast (mtimg, FF_ImgChanTypeRedMatte);
        fast &= isfast (mtimg, FF_ImgChanTypeGreenMatte);
        fast &= isfast (mtimg, FF_ImgChanTypeBlueMatte);
    } else {
        fast &= isfast (fgimg, FF_ImgChanTypeRedMatte);
        fast &= isfast (fgimg, FF_ImgChanTypeGreenMatte);
        fast &= isfast (fgimg, FF_ImgChanTypeBlueMatte);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeRed] &&
        cmimg->pimg.channel[FF_ImgChanTypeRed]) {
        fast &= isfast (fgimg, FF_ImgChanTypeRed);
        fast &= isfast (bgimg, FF_ImgChanTypeRed);
        fast &= isfast (cmimg, FF_ImgChanTypeRed);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeGreen] &&
        cmimg->pimg.channel[FF_ImgChanTypeGreen]) {
        fast &= isfast (fgimg, FF_ImgChanTypeGreen);
        fast &= isfast (bgimg, FF_ImgChanTypeGreen);
        fast &= isfast (cmimg, FF_ImgChanTypeGreen);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeBlue] &&
        cmimg->pimg.channel[FF_ImgChanTypeBlue]) {
        fast &= isfast (fgimg, FF_ImgChanTypeBlue);
        fast &= isfast (bgimg, FF_ImgChanTypeBlue);
        fast &= isfast (cmimg, FF_ImgChanTypeBlue);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeRedMatte] &&
        cmimg->pimg.channel[FF_ImgChanTypeRedMatte]) {
        fast &= isfast (fgimg, FF_ImgChanTypeRedMatte);
        fast &= isfast (bgimg, FF_ImgChanTypeRedMatte);
        fast &= isfast (cmimg, FF_ImgChanTypeRedMatte);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeGreenMatte] &&
        cmimg->pimg.channel[FF_ImgChanTypeGreenMatte]) {
        fast &= isfast (fgimg, FF_ImgChanTypeGreenMatte);
        fast &= isfast (bgimg, FF_ImgChanTypeGreenMatte);
        fast &= isfast (cmimg, FF_ImgChanTypeGreenMatte);
    }
    if (IP_StateDrawMask[FF_ImgChanTypeBlueMatte] &&
        cmimg->pimg.channel[FF_ImgChanTypeBlueMatte]) {
        fast &= isfast (fgimg, FF_ImgChanTypeBlueMatte);
        fast &= isfast (bgimg, FF_ImgChanTypeBlueMatte);
        fast &= isfast (cmimg, FF_ImgChanTypeBlueMatte);
    }

    return fast?
        IP_CompFastColorMatte (fgimg, bgimg, cmimg,
                                fr, fg, fb, br, bg, bb, mtimg):
        IP_CompSlowColorMatte (fgimg, bgimg, cmimg,
                                fr, fg, fb, br, bg, bb, mtimg);
}

/* Composite two images using depth mattes: Decide whether to use the
   fast or the slow compositing method, and invoke the right algorithm. */

static UT_Bool comp_zmatte (IP_ImageId fgimg, IP_ImageId bgimg, IP_ImageId cmimg)
{
    Int32 chn;
    UT_Bool fast;

    /* Make sure that we have depth channels in both input images. */
    if (!fgimg->pimg.channel[FF_ImgChanTypeDepth]) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_fg_depth);
        return UT_False;
    }
    if (!bgimg->pimg.channel[FF_ImgChanTypeDepth]) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_bg_depth);
        return UT_False;
    }

    /* Invoke the compositing algorithm. */
    fast = UT_True;
    for (chn = FF_ImgChanTypeBrightness; chn <= FF_ImgChanTypeVertSnr; ++chn) {
        if (IP_StateDrawMask[chn] && cmimg->pimg.channel[chn]) {
            fast &= fgimg->pimg.channel[chn] == bgimg->pimg.channel[chn];
            fast &= bgimg->pimg.channel[chn] == cmimg->pimg.channel[chn];
        }
    }
    fast &= fgimg->pimg.channel[FF_ImgChanTypeDepth] == FF_ImgFmtTypeFloat;
    fast &= bgimg->pimg.channel[FF_ImgChanTypeDepth] == FF_ImgFmtTypeFloat;

    return fast?
        IP_CompFastDepth (fgimg, bgimg, cmimg):
        IP_CompSlowDepth (fgimg, bgimg, cmimg);
}


/* Cross-dissove between two images: Decide whether to use the fast
   or the slow compositing method, and invoke the right algorithm. */

static UT_Bool crossimage
        (IP_ImageId fgimg, IP_ImageId bgimg, IP_ImageId cmimg, Float32 mix)
{
    Int32 chn;
    UT_Bool fast;

    fast = UT_True;
    for (chn = FF_ImgChanTypeBrightness; chn <= FF_ImgChanTypeBlueMatte; ++chn) {
        if (IP_StateDrawMask[chn] && cmimg->pimg.channel[chn]) {
            fast &= isfast (fgimg, chn);
            fast &= isfast (bgimg, chn);
            fast &= isfast (cmimg, chn);
        }
    }
    fast &= mix >= 0.0;
    fast &= mix <= 1.0;

    return fast?
        IP_CrossDissolveFast (fgimg, bgimg, cmimg, mix):
        IP_CrossDissolveSlow (fgimg, bgimg, cmimg, mix);
}

/***************************************************************************
 *[@e
 *      Name:           IP_CompositeMatte
 *
 *      Usage:          Composite two images using a grayscale matte.
 *
 *      Synopsis:       UT_Bool IP_CompositeMatte(
 *                              IP_ImageId fgImg,
 *                              IP_ImageId bgImg,
 *                              IP_ImageId destImg,
 *                              Float32 fr, Float32 fg, Float32 fb,
 *                              Float32 br, Float32 bg, Float32 bb,
 *                              UT_Bool fastMode,
 *                              IP_ImageId matteImg)
 *
 *      Description:    Image "fgImg" is composited in front of image
 *                      "bgImg". The result is stored in "destImg".
 *                      If "matteImg" is NULL, the matte for compositing is
 *                      taken from image "fgImg". If "matteImg" is not NULL,
 *                      image "matteImg" supplies the matte.
 *                      
 *                      1. The red, green and blue channels of all pixels
 *                         from "fgImg" are scaled by "fr", "fg" and "fb".
 *                         The red, green and blue channels of all pixels
 *                         from "bgImg" are scaled by "br", "bg" and "bb".
 *
 *                      2. The brightness, red, green and blue channels
 *                         for every pixel in "destImg" are combined from the 
 *                         data in "fgImg" and "bgImg":
 *                          
 *                         if "matteImg" is NULL:
 *
 *                             cm.bright =
 *                                  fg.bright +
 *                                  bg.bright * (1-fg.matte);
 *                             cm.red =
 *                                  fg.red +
 *                                  bg.red * (1-fg.matte);
 *                             cm.green =
 *                                  fg.green +
 *                                  bg.green * (1-fg.matte);
 *                             cm.blue =
 *                                  fg.blue +
 *                                  bg.blue * (1-fg.matte);
 *                          else
 *
 *                             cm.bright =
 *                                  fg.bright * mt.matte +
 *                                  bg.bright * (1-mt.matte);
 *                             cm.red =
 *                                  fg.red * mt.matte +
 *                                  bg.red * (1-mt.matte);
 *                             cm.green =
 *                                  fg.green * mt.matte +
 *                                  bg.green * (1-mt.matte);
 *                             cm.blue =
 *                                  fg.blue * mt.matte +
 *                                  bg.blue * (1-mt.matte);
 *
 *                      3. The FF_ImgChanTypeMatte channel for every pixel in
 *                         "destImg" is computed:
 *
 *                         if "matteImg" is NULL:
 *
 *                             cm.matte = 1 - (1-fg.matte) * (1-bg.matte);
 *
 *                         else
 *
 *                             cm.matte = 1 - (1-mt.matte) * (1-bg.matte);
 *
 *                      Notes:
 *
 *                      - If necessary, "fgImg" or "bgImg" can be
 *                        identical to "destImg", so that IP_CompositeMatte
 *                        overwrites one of the input images with the result.
 *
 *                      - IP_CompositeMatte does not write to the
 *                        FF_ImgChanTypeRedMatte, FF_ImgChanTypeGreenMatte,
 *                        FF_ImgChanTypeBlueMatte, FF_ImgChanTypeHoriSnr,
 *                        FF_ImgChanTypeVertSnr and FF_ImgChanTypeDepth channels
 *                        in "destImg", even if the drawing mask for these 
 *                        channels is set.
 *
 *                      - For compositing, "fgImg", "bgImg" and
 *                        "destImg" are aligned at the lower left corner.
 *                        If "destImg" is wider or higher than "fgImg" or
 *                        "bgImg", IP_CompositeMatte does not change areas
 *                        in "destImg" which are not covered by "fgImg" or
 *                        "bgImg".
 *
 *                      - If the "fastMode" flag is set to UT_True, and all pixel
 *                        data involved in the compositing are in 
 *                        FF_ImgFmtTypeUByte format, IP_CompositeMatte will
 *                        switch to a faster, but less general compositing 
 *                        algorithm.
 *                        The results are usually the same as with the slower
 *                        algorithm; only if one of "fr", "fg", "fb", "br",
 *                        "bg" or "bb" is greater than 1.0 or less than 0.0,
 *                        color clipping may be different.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    Yes
 *                      UByte format: RGB, Brightness
 *                      Float format: RGB, Brightness
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CompositeColorMatte
 *                      IP_CompositeDepth
 *                      IP_CrossDissolve
 *                      IP_SetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_CompositeMatte
        (IP_ImageId fgImg,
         IP_ImageId bgImg,
         IP_ImageId destImg,
         Float32 fr, Float32 fg, Float32 fb,
         Float32 br, Float32 bg, Float32 bb,
         UT_Bool fastMode,
         IP_ImageId matteImg)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = comp_bwmatte
            (fgImg, bgImg, destImg,
             fr, fg, fb,
             br, bg, bb,
             matteImg, fastMode);
    UT_MemRestore (memstate);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_CompositeColorMatte
 *
 *      Usage:          Composite two images using a color matte.
 *
 *      Synopsis:       UT_Bool IP_CompositeColorMatte(
 *                              IP_ImageId fgImg,
 *                              IP_ImageId bgImg,
 *                              IP_ImageId destImg,
 *                              Float32 fr, Float32 fg, Float32 fb,
 *                              Float32 br, Float32 bg, Float32 bb,
 *                              UT_Bool fastMode,
 *                              IP_ImageId matteImg)
 *
 *      Description:    Image "fgImg" is composited in front of image
 *                      "bgImg". The result is stored in "destImg".
 *                      If "matteImg" is NULL, the matte for compositing is
 *                      taken from image "fgImg". If "matteImg" is not NULL,
 *                      image "matteImg" supplies the matte.
 *                      
 *                      1. The red, green and blue channels of all pixels
 *                         from "fgImg" are scaled by "fr", "fg" and "fb".
 *                         The red, green and blue channels of all pixels
 *                         from "bgImg" are scaled by "br", "bg" and "bb".
 *
 *                      2. The red, green and blue channels for every pixel
 *                         in "destImg" are combined from the data in "fgImg"
 *                         and "bgImg":
 *                          
 *                         if "matteImg" is not NUL:
 *
 *                             cm.red = fg.red * mt.red_matte +
 *                                  bg.red * (1-mt.red_matte);
 *
 *                             cm.green = fg.green * mt.green_matte +
 *                                  bg.green * (1-mt.green_matte);
 *
 *                             cm.blue = fg.blue * mt.blue_matte +
 *                                  bg.blue * (1-mt.blue_matte);
 *                         
 *                         else
 *
 *                             cm.red = fg.red +
 *                                  bg.red * (1-fg.red_matte);
 *
 *                             cm.green = fg.green +
 *                                  bg.green * (1-fg.green_matte);
 *
 *                             cm.blue = fg.blue +
 *                                  bg.blue * (1-fg.blue_matte);
 *
 *                      3. The FF_ImgChanTypeRedMatte, FF_ImgChanTypeGreenMatte
 *                         and FF_ImgChanTypeBlueMatte channels for every pixel
 *                         in "destImg" are combined from the data in "fgImg"
 *                         and "bgImg":
 *
 *                         if "matteImg" is not NULL:
 *
 *                             cm.red_matte = 1 -
 *                                  (1-mt.red_matte) * (1-bg.red_matte);
 *
 *                             cm.green_matte = 1 -
 *                                  (1-mt.green_matte) * (1-bg.green_matte);
 *
 *                             cm.blue_matte = 1 -
 *                                  (1-mt.blue_matte) * (1-bg.blue_matte);
 *                                  
 *                         else
 *
 *                             cm.red_matte = 1 -
 *                                  (1-fg.red_matte) * (1-bg.red_matte);
 *
 *                             cm.green_matte = 1 -
 *                                  (1-fg.green_matte) * (1-bg.green_matte);
 *
 *                             cm.blue_matte = 1 -
 *                                  (1-fg.blue_matte) * (1-bg.blue_matte);
 *                                  
 *                      Notes:
 *
 *                      - If necessary, "fgImg" or "bgImg" can be
 *                        identical to "destImg", so that IP_CompositeColorMatte
 *                        overwrites one of the input images with the result.
 *
 *                      - IP_CompositeColorMatte does not write to the
 *                        FF_ImgChanTypeBrightness, FF_ImgChanTypeMatte, 
 *                        FF_ImgChanTypeHoriSnr, FF_ImgChanTypeVertSnr and 
 *                        FF_ImgChanTypeDepth channels in "destImg", even if the 
 *                        drawing mask for these channels is set.
 *
 *                      - For compositing, "fgImg", "bgImg" and
 *                        "destImg" are aligned at the lower left corner.
 *                        If "destImg" is wider or higher than "fgImg" or
 *                        "bgImg", IP_CompositeColorMatte does not change areas
 *                        in "destImg" which are not covered by "fgImg" or
 *                        "bgImg".
 *
 *                      - If the "fastMode" flag is set to UT_True, and all pixel
 *                        data involved in the compositing are in
 *                        FF_ImgFmtTypeUByte format, IP_CompositeColorMatte will
 *                        switch to a faster, but less general compositing
 *                        algorithm.
 *                        The results are usually the same as with the slower
 *                        algorithm; only if one of "fr", "fg", "fb", "br",
 *                        "bg" or "bb" is greater than 1.0 or less than 0.0,
 *                        color clipping may be different.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    Yes
 *                      UByte format: RGB
 *                      Float format: RGB
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CompositeMatte
 *                      IP_CompositeDepth
 *                      IP_CrossDissolve
 *                      IP_SetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_CompositeColorMatte
        (IP_ImageId fgImg,
         IP_ImageId bgImg,
         IP_ImageId destImg,
         Float32 fr, Float32 fg, Float32 fb,
         Float32 br, Float32 bg, Float32 bb,
         UT_Bool fastMode,
         IP_ImageId matteImg)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = comp_clrmatte
            (fgImg, bgImg, destImg,
             fr, fg, fb,
             br, bg, bb,
             matteImg, fastMode);
    UT_MemRestore (memstate);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_CompositeDepth
 *
 *      Usage:          Composite two images using a depth matte.
 *
 *      Synopsis:       UT_Bool IP_CompositeDepth(
 *                              IP_ImageId fgImg,
 *                              IP_ImageId bgImg,
 *                              IP_ImageId destImg)
 *
 *      Description:    Image "fgImg" and "bgImg" are combined into a new
 *                      image "destImg". Both, "fgImg" and "bgImg" must
 *                      include a FF_ImgChanTypeDepth channel, indicating
 *                      the distance of the pixels from the viewer.
 *
 *                      The data from "fgImg" and "bgImg" are combined as
 *                      follows:
 *
 *                          if fgImg.pixel is closer than bgImg.pixel
 *
 *                              destImg.pixel = fgImg.pixel;
 *                          
 *                          else
 *                              
 *                              destImg.pixel = bgImg.pixel;
 *
 *                      Notes:
 *
 *                      - If necessary, "fgImg" or "bgImg" can be identical to
 *                        "destImg", so that IP_CompositeDepth overwrites one of
 *                        the input images with the result.
 *
 *                      - For compositing, "fgImg", "bgImg" and "destImg" are
 *                        aligned at the lower left corner. If "destImg" is wider
 *                        or higher than "fgImg" or "bgImg", IP_CompositeDepth
 *                        does not change areas in "destImg" which are not
 *                        covered by "fgImg" or "bgImg".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    Yes
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CompositeMatte
 *                      IP_CompositeColorMatte
 *                      IP_CrossDissolve
 *                      IP_SetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_CompositeDepth (IP_ImageId fgImg, IP_ImageId bgImg, IP_ImageId destImg)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = comp_zmatte (fgImg, bgImg, destImg);
    UT_MemRestore (memstate);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_CrossDissolve
 *
 *      Usage:          Cross-dissolve between two images.
 *
 *      Synopsis:       UT_Bool IP_CrossDissolve(
 *                              IP_ImageId fgImg,
 *                              IP_ImageId bgImg,
 *                              IP_ImageId destImg,
 *                              Float32 mixFactor)
 *
 *      Description:    Cross-dissolve between image "fgImg"
 *                      and "bgImg". The result is stored in "destImg":
 *                      
 *                      The pixel data for "destImg" are combined from
 *                      the data in "fgImg" and "bgImg" ("mixFactor" should be
 *                      in the range from 0.0 to 1.0):
 *                          
 *                             cm.pixel = fg.pixel * mixFactor + bg.pixel * (1-mixFactor)
 *
 *                      Notes:
 *
 *                      - If necessary, "fgImg" or "bgImg" can be
 *                        identical to "destImg", so that IP_CrossDissolve
 *                        overwrites one of the input images with the
 *                        result.
 *
 *                      - IP_CrossDissolve does not write to the 
 *                        FF_ImgChanTypeHoriSnr, FF_ImgChanTypeVertSnr,
 *                        FF_ImgChanTypeDepth, FF_ImgChanTypeTemperature
 *                        and FF_ImgChanTypeRadiance channels in "destImg",
 *                        even if the drawing mask for these channels is set.
 *
 *                      - For cross-dissolving, "fgImg", "bgImg" and
 *                        "destImg" are aligned at the lower left corner.
 *                        If "destImg" is wider or higher than "fgImg" or
 *                        "bgImg", IP_CrossDissolve does not change areas
 *                        in "destImg" which are not covered by "fgImg" or
 *                        "bgImg".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    Yes
 *                      UByte format: RGB, Brightness
 *                      Float format: RGB, Brightness
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CompositeMatte
 *                      IP_CompositeColorMatte
 *                      IP_CompositeDepth
 *                      IP_SetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_CrossDissolve
        (IP_ImageId fgImg, IP_ImageId bgImg, IP_ImageId destImg, Float32 mixFactor)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = crossimage (fgImg, bgImg, destImg, mixFactor);
    UT_MemRestore (memstate);
    return success;
}
