/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Filter.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Various image filtering functions.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_ChangeChannelGamma
 *                      IP_RemapChannelValues
 *                      IP_ComputeChannelDerivates
 *                      IP_Median
 *                      IP_MedianSequence
 *                      IP_Dilatation
 *                      IP_Erosion
 *                      IP_Threshold
 *                      IP_CutOff
 *                      IP_MarkNonZeroPixels
 *                      IP_DifferenceImage
 *                      IP_DiffChannel
 *
 **************************************************************************/

#include <stdio.h>

#include "UT_Compat.h"
#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Memory.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#define str_dilatationcount "Dilatation count"
#define str_erosioncount    "Erosion count"

/* Gamma-filter and scale one channel of an image. For an explanation
   of the parameters, see the comments on function IP_ChangeChannelGamma. */

static UT_Bool change_gamma
        (IP_ImageId img, FF_ImgChanType chan,
         Float32 gamma, Float32 scl, Float32 off)
{
    UInt8   ubgtab[256], *ubdata, *ubstop;
    Float32 fgtab[GTABSIZE], *fdata;
    Int32   y, i, tmp;

    switch (img->pimg.channel[chan]) {
        case FF_ImgFmtTypeUByte: {
            if (!IP_UByteGammaTable (1.0 / gamma, ubgtab)) {
                return UT_False;
            }
            for (i = 0; i < 256; ++i) {
                tmp = (Int32) (ubgtab[i] * scl + off * 255.0);
                ubgtab[i] = UT_MAX (0, UT_MIN (tmp, 255));
            }
            for (y = 0; y < img->pimg.desc.height; ++y) {
                ubdata = FF_ImgUByteRow (&img->pimg, chan, y);
                ubstop = ubdata + img->pimg.desc.width;
                while (ubdata < ubstop) {
                    *ubdata = ubgtab[*ubdata];
                    ++ubdata;
                }
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            if (!IP_FloatGammaTable (1.0 / gamma, fgtab)) {
                return UT_False;
            }
            for (i = 0; i < GTABSIZE; ++i) {
                fgtab[i] = fgtab[i] * scl + off;
            }
            for (y = 0; y < img->pimg.desc.height; ++y) {
                fdata = FF_ImgFloatRow (&img->pimg, chan, y);
                IP_GammaFloat2Float
                    (img->pimg.desc.width, fdata, fgtab, fdata);
            }
            break;
        }
        default: {
            UT_ErrFatal ("change_gamma", "unknown pixel data format");
        }
    }
    return UT_True;
}

/* Remap pixel values using a lookup table. For an explanation of the
   parameters, see the comments on function IP_RemapChannelValues. */

static void remap_values
        (IP_ImageId img, FF_ImgChanType chan,
         Int32 lutsize, const Float32 lut[])
{
    UInt8   *ubdata, *ubstop, ublut[256];
    Float32 *fdata, *fstop, lutscale, ftmp, s, t;
    Int32   lutclip, i, y, a, b, c;

    switch (img->pimg.channel[chan]) {
        case FF_ImgFmtTypeUByte: {
            lutscale = (lutsize - 1) / 255.0;
            lutclip = lutsize - 2;
            for (i = 0; i < 256; ++i) {
                ftmp = i * lutscale;
                a = (Int32)ftmp;
                a = UT_MAX (0, UT_MIN (a, lutclip));
                b = a + 1;
                s = ftmp - a;
                t = b - ftmp;
                c = (Int32) (255.0 * (t * lut[a] + s * lut[b]));
                ublut[i] = UT_MAX (0, UT_MIN (c, 255));
            }
            for (y = 0; y < img->pimg.desc.height; ++y) {
                ubdata = FF_ImgUByteRow (&img->pimg, chan, y);
                ubstop = ubdata + img->pimg.desc.width;
                while (ubdata < ubstop) {
                    *ubdata = ublut[*ubdata];
                    ++ubdata;
                }
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            lutscale = (Float32) (lutsize - 1);
            lutclip = lutsize - 2;
            for (y = 0; y < img->pimg.desc.height; ++y) {
                fdata = FF_ImgFloatRow (&img->pimg, chan, y);
                fstop = fdata + img->pimg.desc.width;
                while (fdata < fstop) {
                    ftmp = *fdata * lutscale;
                    a = (Int32)ftmp;
                    a = UT_MAX (0, UT_MIN (a, lutclip));
                    b = a + 1;
                    s = ftmp - a;
                    t = b - ftmp;
                    *(fdata++) = t * lut[a] + s * lut[b];
                }
            }
            break;
        }
        default: {
            UT_ErrFatal ("remap_values", "unknown pixel data format");
        }
    }
    return;
}

/* Copy channel "chan" of scan line "y" of image "img" into buffer "buf",
   converting the pixel data to FF_ImgFmtTypeFloat format.
   Function "scan_to_buf" observes the "wrap" flag as in IP_ComputeChannelDerivates. */

static void scan_to_buf
        (IP_ImageId img, FF_ImgChanType chan, Int32 y, 
         Float32 *buf, UT_Bool wrap)
{
    UInt8   *ubsrc;
    Float32 *fsrc, *fdest, *fstop;

    if (y < 0) {
        y = wrap? img->pimg.desc.height - 1: 0;
    }
    fdest = buf + 1;
    fstop = fdest + img->pimg.desc.width;
    switch (img->pimg.channel[chan]) {
        case FF_ImgFmtTypeNone: {
            while (fdest < fstop) {
                *(fdest++) = 0.0;
            }
            break;
        }
        case FF_ImgFmtTypeUByte: {
            ubsrc = FF_ImgUByteRow (&img->pimg, chan, y);
            while (fdest < fstop) {
                *(fdest++) = (1.0 / 255.0) * *(ubsrc++);
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            fsrc = FF_ImgFloatRow (&img->pimg, chan, y);
            while (fdest < fstop) {
                *(fdest++) = *(fsrc++);
            }
            break;
        }
    }
    buf[0] = buf[wrap? img->pimg.desc.width: 1];
    return;
}

/* Copy the contents of buffer "buf" into channel "chan" of scan line "y" of
   image "img", converting the format of the pixel data if necessary. */

static void buf_to_scan
        (Float32 *buf, IP_ImageId img, FF_ImgChanType chan, Int32 y, Int32 width)
{
    Int32   tmp;
    UInt8   *ubdest;
    Float32 *fdest, *fsrc, *fstop;

    fsrc = buf;
    fstop = buf + width;
    switch (img->pimg.channel[chan]) {
        case FF_ImgFmtTypeNone: {
            break;
        }
        case FF_ImgFmtTypeUByte: {
            ubdest = FF_ImgUByteRow (&img->pimg, chan, y);
            while (fsrc < fstop) {
                tmp = (Int32) (255.0 * *(fsrc++));
                *(ubdest++) = UT_MAX (0, UT_MIN (tmp, 255));
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            fdest = FF_ImgFloatRow (&img->pimg, chan, y);
            while (fsrc < fstop) {
                *(fdest++) = *(fsrc++);
            }
            break;
        }
    }
    return;
}

/* Compute differences between adjacent pixels in horizontal and in
   vertical direction. For an explanation of the parameters, see the
   comments on function IP_ComputeChannelDerivates. */

static UT_Bool hvdiff
        (IP_ImageId srcimg, FF_ImgChanType srcchan,
         IP_ImageId destimg, FF_ImgChanType hchan, FF_ImgChanType vchan,
         UT_Bool wrap)
{
    Float32 *scan0, *scan1, *diff, *src0, *src1, *dest, *stop;
    Int32   width, height, y;

    if (!(diff = UT_MemTempArray (srcimg->pimg.desc.width, Float32))) {
        return UT_False;
    }
    if (!(scan0 = UT_MemTempArray (srcimg->pimg.desc.width + 1, Float32)) ||
        !(scan1 = UT_MemTempArray (srcimg->pimg.desc.width + 1, Float32))) {
        return UT_False;
    }
    width = UT_MIN (srcimg->pimg.desc.width, destimg->pimg.desc.width);
    height = UT_MIN (srcimg->pimg.desc.height, destimg->pimg.desc.height);
    scan_to_buf (srcimg, srcchan, -1, scan0, wrap);
    for (y = 0; y < height; ++y) {
        scan_to_buf (srcimg, srcchan, y, scan1, wrap);
        stop = diff + width;
        dest = diff;
        src0 = scan1;
        src1 = scan1 + 1;
        while (dest < stop) {
            *(dest++) = 0.5 + *(src1++) - *(src0++);
        }
        buf_to_scan (diff, destimg, hchan, y, width);
        dest = diff;
        src0 = scan0 + 1;
        src1 = scan1 + 1;
        while (dest < stop) {
            *(dest++) = 0.5 + *(src1++) - *(src0++);
        }
        buf_to_scan (diff, destimg, vchan, y, width);
        UT_SWAP (scan0, scan1, Float32 *);
    }
    return UT_True;
}

/* Apply a three pixel wide horizontal median filter to all channels in image
   "img" for which the current drawing mask is UT_True. */

static void median_h (IP_ImageId img)
{
    Int32      x, y, chan;
    UInt8      ub0, ub1, ub2;
    Float32    f0, f1, f2;
    FF_ImgDesc *pimg;

    pimg = &img->pimg;
    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan]) {
            continue;
        }
        switch (img->pimg.channel[chan]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                for (y = 0; y < pimg->desc.height; ++y) {
                    ub0 = *FF_ImgUBytePixel (pimg, chan, 0, y);
                    ub1 = *FF_ImgUBytePixel (pimg, chan, 1, y);
                    for (x = 2; x < pimg->desc.width; ++x) {
                        ub2 = *FF_ImgUBytePixel (pimg, chan, x, y);
                        *FF_ImgUBytePixel (pimg, chan, x - 1, y) =
                            (ub0 > ub1?
                             (ub0 > ub2?
                              (ub1 > ub2? ub1: ub2)
                              :
                              ub0
                              )
                             :
                             (ub0 > ub2?
                              ub0
                              :
                              (ub1 > ub2? ub2: ub1)
                              )
                            );
                        ub0 = ub1;
                        ub1 = ub2;
                    }
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                for (y = 0; y < pimg->desc.height; ++y) {
                    f0 = *FF_ImgFloatPixel (pimg, chan, 0, y);
                    f1 = *FF_ImgFloatPixel (pimg, chan, 1, y);
                    for (x = 2; x < pimg->desc.width; ++x) {
                        f2 = *FF_ImgFloatPixel (pimg, chan, x, y);
                        *FF_ImgFloatPixel (pimg, chan, x - 1, y) =
                            (f0 > f1?
                             (f0 > f2?
                              (f1 > f2? f1: f2)
                              :
                              f0
                              )
                             :
                             (f0 > f2?
                              f0
                              :
                              (f1 > f2? f2: f1)
                              )
                            );
                        f0 = f1;
                        f1 = f2;
                    }
                }
                break;
            }
        }
    }
    return;
}

/* Apply a three pixel wide vertical median filter to all channels in image
   "img" for which the current drawing mask is UT_True. */

static void median_v (IP_ImageId img)
{
    Int32      x, y, chan;
    UInt8      ub0, ub1, ub2;
    Float32    f0, f1, f2;
    FF_ImgDesc *pimg;

    pimg = &img->pimg;
    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan]) {
            continue;
        }
        switch (img->pimg.channel[chan]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                for (x = 0; x < pimg->desc.width; ++x) {
                    ub0 = *FF_ImgUBytePixel (pimg, chan, x, 0);
                    ub1 = *FF_ImgUBytePixel (pimg, chan, x, 1);
                    for (y = 2; y < pimg->desc.height; ++y) {
                        ub2 = *FF_ImgUBytePixel (pimg, chan, x, y);
                        *FF_ImgUBytePixel (pimg, chan, x, y - 1) =
                            (ub0 > ub1?
                             (ub0 > ub2?
                              (ub1 > ub2? ub1: ub2)
                              :
                              ub0
                              )
                             :
                             (ub0 > ub2?
                              ub0
                              :
                              (ub1 > ub2? ub2: ub1)
                              )
                            );
                        ub0 = ub1;
                        ub1 = ub2;
                    }
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                for (x = 0; x < pimg->desc.width; ++x) {
                    f0 = *FF_ImgFloatPixel (pimg, chan, x, 0);
                    f1 = *FF_ImgFloatPixel (pimg, chan, x, 1);
                    for (y = 2; y < pimg->desc.height; ++y) {
                        f2 = *FF_ImgFloatPixel (pimg, chan, x, y);
                        *FF_ImgFloatPixel (pimg, chan, x, y - 1) =
                            (f0 > f1?
                             (f0 > f2?
                              (f1 > f2? f1: f2)
                              :
                              f0
                              )
                             :
                             (f0 > f2?
                              f0
                              :
                              (f1 > f2? f2: f1)
                              )
                            );
                        f0 = f1;
                        f1 = f2;
                    }
                }
                break;
            }
        }
    }
    return;
}

/* Apply a median filter to all channels in image sequence "srcimg1",
   "srcimg2", "srcimg3", for which the current drawing mask is UT_True. */

static void median_sequence
        (IP_ImageId srcimg1,
         IP_ImageId srcimg2,
         IP_ImageId srcimg3,
         IP_ImageId destimg)
{
    Int32      x, y, chan, itmp, width, height;
    Float32    src1 = 0.0, src2 = 0.0, src3 = 0.0, dest = 0.0;
    FF_ImgDesc *psrc1, *psrc2, *psrc3, *pdest;

    pdest = &destimg->pimg;
    psrc1 = &srcimg1->pimg;
    psrc2 = &srcimg2->pimg;
    psrc3 = &srcimg3->pimg;
    width = pdest->desc.width;
    width = UT_MIN (width, psrc1->desc.width);
    width = UT_MIN (width, psrc2->desc.width);
    width = UT_MIN (width, psrc3->desc.width);
    height = pdest->desc.height;
    height = UT_MIN (height, psrc1->desc.height);
    height = UT_MIN (height, psrc2->desc.height);
    height = UT_MIN (height, psrc3->desc.height);
    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan] || !pdest->channel[chan]) {
            continue;
        }
        for (y = 0; y < height; ++y) {
            for (x = 0; x < width; ++x) {
                switch (psrc1->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        src1 = 0.0;
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        src1 = (1.0 / 255.0) *
                            *FF_ImgUBytePixel (psrc1, chan, x, y);
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        src1 = *FF_ImgFloatPixel (psrc1, chan, x, y);
                        break;
                    }
                }
                switch (psrc2->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        src2 = 0.0;
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        src2 = (1.0 / 255.0) *
                            *FF_ImgUBytePixel (psrc2, chan, x, y);
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        src2 = *FF_ImgFloatPixel (psrc2, chan, x, y);
                        break;
                    }
                }
                switch (psrc3->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        src3 = 0.0;
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        src3 = (1.0 / 255.0) *
                            *FF_ImgUBytePixel (psrc3, chan, x, y);
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        src3 = *FF_ImgFloatPixel (psrc3, chan, x, y);
                        break;
                    }
                }
                dest =
                    (src1 > src2?
                     (src1 > src3?
                      (src2 > src3? src2: src3)
                      :
                      src1
                      )
                     :
                     (src1 > src3?
                      src1
                      :
                      (src2 > src3? src3: src2)
                      )
                    );
                switch (pdest->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        itmp = (Int32) (255.0 * dest);
                        *FF_ImgUBytePixel (pdest, chan, x, y) =
                            UT_MAX (0, UT_MIN (itmp, 255));
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        *FF_ImgFloatPixel (pdest, chan, x, y) = dest;
                        break;
                    }
                }
            }
        }
    }
    return;
}

/* Apply a three by three pixel dilatation/erosion filter to one
   scan line of a UByte channel of an image. */

static void dler_UByte_scan (UInt8 *s1, UInt8 *s2, UInt8 *s3, 
                             Int32 len, Int32 mode, UInt8 *dest)
{
    UInt8 ub;
    UInt8 *stop = s1 + len;

    if (mode == 1) {
        while (s1 < stop) {
            ub = 0;

            ub = UT_MAX (ub, UT_MAX (*(s1-1), UT_MAX (*s1, *(s1+1))));
            ub = UT_MAX (ub, UT_MAX (*(s2-1), UT_MAX (*s2, *(s2+1))));
            ub = UT_MAX (ub, UT_MAX (*(s3-1), UT_MAX (*s3, *(s3+1))));

            *dest++ = ub;
            s1++;
            s2++;
            s3++;
        }
    } else {
        while (s1 < stop) {
            ub = 255;

            ub = UT_MIN (ub, UT_MIN (*(s1-1), UT_MIN (*s1, *(s1+1))));
            ub = UT_MIN (ub, UT_MIN (*(s2-1), UT_MIN (*s2, *(s2+1))));
            ub = UT_MIN (ub, UT_MIN (*(s3-1), UT_MIN (*s3, *(s3+1))));

            *dest++ = ub;
            s1++;
            s2++;
            s3++;
        }
    }
}

/* Apply a three by three pixel dilatation/erosion filter to one
   scan line of a Float32 channel of an image. */

static void dler_Float_scan (Float32 *s1, Float32 *s2, Float32 *s3, 
                             Int32 len, Int32 mode, Float32 *dest)
{
    Float32 f;
    Float32 *stop = s1 + len;

    if (mode == 1) {
        while (s1 < stop) {
            f = 0.0;

            f = UT_MAX (f, UT_MAX (*(s1-1), UT_MAX (*s1, *(s1+1))));
            f = UT_MAX (f, UT_MAX (*(s2-1), UT_MAX (*s2, *(s2+1))));
            f = UT_MAX (f, UT_MAX (*(s3-1), UT_MAX (*s3, *(s3+1))));

            *dest++ = f;
            s1++;
            s2++;
            s3++;
        }
    } else {
        while (s1 < stop) {
            f = 1.0;

            f = UT_MIN (f, UT_MIN (*(s1-1), UT_MIN (*s1, *(s1+1))));
            f = UT_MIN (f, UT_MIN (*(s2-1), UT_MIN (*s2, *(s2+1))));
            f = UT_MIN (f, UT_MIN (*(s3-1), UT_MIN (*s3, *(s3+1))));

            *dest++ = f;
            s1++;
            s2++;
            s3++;
        }
    }
}

/* Apply a three by three pixel dilatation/erosion filter to all channels 
   in image "img" for which the current drawing mask is UT_True. */

static UT_Bool dilat_eros (IP_ImageId img, Int32 n, Int32 mode)
{
    Int32      i, x, y, chan;
    UInt8      *ubuf1 = NULL, *ubuf2 = NULL;
    Float32    *fbuf1 = NULL, *fbuf2 = NULL;
    FF_ImgDesc *pimg;

    pimg = &img->pimg;
    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan]) {
            continue;
        }
        switch (img->pimg.channel[chan]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                UInt8   *in_s1, *in_s2, *in_s3, *pix1, *pix2, *pix3, 
                        *buf1_ptr, *buf2_ptr, *out_p1, *out_p2;
                if (!ubuf1 || !ubuf2) {
                    if (!(ubuf1 = UT_MemTempArray (pimg->desc.width, UInt8)) ||
                        !(ubuf2 = UT_MemTempArray (pimg->desc.width, UInt8))) {
                        return UT_False;
                    }
                }
                for (i = 0; i < n; i++) {
                    in_s1 = FF_ImgUByteRow (pimg, chan, 0);
                    in_s2 = FF_ImgUByteRow (pimg, chan, 1);
                    in_s3 = FF_ImgUByteRow (pimg, chan, 2);
                    pix1 = in_s1 + 1; 
                    pix2 = in_s2 + 1;
                    pix3 = in_s3 + 1;
                    buf1_ptr = ubuf1;
                    buf2_ptr = ubuf2;
                    out_p1 = buf1_ptr + 1;
                    out_p2 = buf2_ptr + 1;
                    dler_UByte_scan (pix1, pix2, pix3, 
                                     pimg->desc.width-2, mode, out_p1);

                    for (y = 2; y < pimg->desc.height -1; ++y) {
                        in_s1 = in_s2;
                        in_s2 = in_s3;
                        pix1 = in_s1 + 1; 
                        pix2 = in_s2 + 1;
                        in_s3 = FF_ImgUByteRow (pimg, chan, y+1);
                        pix3 = in_s3 + 1;
                        dler_UByte_scan (pix1, pix2, pix3, 
                                         pimg->desc.width-2, mode, out_p2);

                        for (x = 1; x < pimg->desc.width -1; ++x) {
                            *FF_ImgUBytePixel (pimg, chan, x, y-1) = buf1_ptr[x];
                        }
                        UT_SWAP (buf1_ptr, buf2_ptr, UInt8 *);
                        out_p2 = buf2_ptr;
                    }
                    for (x = 1; x < pimg->desc.width -1; ++x) {
                        *FF_ImgUBytePixel (pimg, chan, x, y-1) = buf1_ptr[x];
                    }
                }
                break;
            }

            case FF_ImgFmtTypeFloat: {
                Float32 *in_s1, *in_s2, *in_s3, *pix1, *pix2, *pix3, 
                        *buf1_ptr, *buf2_ptr, *out_p1, *out_p2;
                if (!fbuf1 || !fbuf2) {
                    if (!(fbuf1 = UT_MemTempArray (pimg->desc.width, Float32)) ||
                        !(fbuf2 = UT_MemTempArray (pimg->desc.width, Float32))) {
                        return UT_False;
                    }
                }
                for (i = 0; i < n; i++) {
                    in_s1 = FF_ImgFloatRow (pimg, chan, 0);
                    in_s2 = FF_ImgFloatRow (pimg, chan, 1);
                    in_s3 = FF_ImgFloatRow (pimg, chan, 2);
                    pix1 = in_s1 + 1; 
                    pix2 = in_s2 + 1;
                    pix3 = in_s3 + 1;
                    buf1_ptr = fbuf1;
                    buf2_ptr = fbuf2;
                    out_p1 = buf1_ptr + 1;
                    out_p2 = buf2_ptr + 1;
                    dler_Float_scan (pix1, pix2, pix3, 
                                     pimg->desc.width-2, mode, out_p1);

                    for (y = 2; y < pimg->desc.height -1; ++y) {
                        in_s1 = in_s2;
                        in_s2 = in_s3;
                        pix1 = in_s1 + 1; 
                        pix2 = in_s2 + 1;
                        in_s3 = FF_ImgFloatRow (pimg, chan, y+1);
                        pix3 = in_s3 + 1;
                        dler_Float_scan (pix1, pix2, pix3, 
                                         pimg->desc.width-2, mode, out_p2);

                        for (x = 1; x < pimg->desc.width -1; ++x) {
                            *FF_ImgFloatPixel (pimg, chan, x, y-1) = buf1_ptr[x];
                        }
                        UT_SWAP (buf1_ptr, buf2_ptr, Float32 *);
                        out_p2 = buf2_ptr;
                    }
                    for (x = 1; x < pimg->desc.width -1; ++x) {
                        *FF_ImgFloatPixel (pimg, chan, x, y-1) = buf1_ptr[x];
                    }
                }
                break;
            }
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_ChangeChannelGamma
 *
 *      Usage:          Change one channel of an image by gamma filtering
 *                      and linear scaling of the pixel data.
 *
 *      Synopsis:       UT_Bool IP_ChangeChannelGamma(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              Float32 gamma, Float32 scale, Float32 offset)
 *
 *      Description:    Channel "channel" of every pixel in image "img" is changed:
 *                      The pixel's orignal value "v" is clipped (values less
 *                      than 0.0 are set to 0.0, values greater than 1.0 are
 *                      set to 1.0). The clipped value is gamma filtered, using
 *                      gamma factor "gamma" and then scaled by "scale". "offset"
 *                      is added to the scaled value. The result replaces the
 *                      pixels original value:
 *
 *                      v = pow (max (0, min (v, 1)), 1/gamma) * scale + offset
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_RemapChannelValues
 *                      IP_ComputeChannelDerivates
 *
 ***************************************************************************/

UT_Bool IP_ChangeChannelGamma
        (IP_ImageId img, FF_ImgChanType channel,
         Float32 gamma, Float32 scale, Float32 offset)
{
    UT_Bool success;

    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (gamma < 0.0) {
        UT_ErrSetNum (UT_ErrParamRange, str_notless, str_gamma, 0.0);
        return UT_False;
    }
    success = change_gamma (img, channel, gamma, scale, offset);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_RemapChannelValues
 *
 *      Usage:          Change one channel of an image by remapping the
 *                      pixel data with a lookup table.
 *
 *      Synopsis:       UT_Bool IP_RemapChannelValues(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              Int32 lutSize, const Float32 lutList[])
 *
 *      Description:    The original value "v" of channel "channel" of every
 *                      pixel in image "img" is mapped into a new value which
 *                      replaces the original pixel data:
 *
 *                              v = f (v * (lutSize - 1))
 *
 *                      "f" is a continuous and piecewise linear function so
 *                      that for all x in the range from 0 to (lutSize - 1)
 *
 *                              f (floor(x)) == lutList[floor(x)].
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_ChangeChannelGamma
 *                      IP_ComputeChannelDerivates
 *
 ***************************************************************************/

UT_Bool IP_RemapChannelValues
        (IP_ImageId img, FF_ImgChanType channel,
         Int32 lutSize, const Float32 lutList[])
{
    UT_Bool success;

    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (lutSize < 2) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_inotless, str_lutsize, 2);
        return UT_False;
    }
    remap_values (img, channel, lutSize, lutList);
    success = UT_True;
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_ComputeChannelDerivates
 *
 *      Usage:          Compute differences between adjacent pixels.
 *
 *      Synopsis:       UT_Bool IP_ComputeChannelDerivates(
 *                              IP_ImageId srcImg,
 *                              FF_ImgChanType srcChannel,
 *                              IP_ImageId destImg,
 *                              FF_ImgChanType horizontalChannel,
 *                              FF_ImgChanType verticalChannel,
 *                              UT_Bool wrap)
 *                      
 *      Description:    For each pixel in channel "srcChannel" of image "srcImg",
 *                      the differences between the pixel and its left and bottom
 *                      neighbors are computed.
 *                      The result is stored in the "horizontalChannel" and
 *                      "verticalChannel" channels of image "destImg":
 *
 *                          horizontalChannel (x, y) = 0.5 + srcChannel (x, y) - srcChannel (x-1, y)
 *                          verticalChannel   (x, y) = 0.5 + srcChannel (x, y) - srcChannel (x, y-1)
 *
 *                      The "wrap" flag determines the differences computed
 *                      at the left and bottom borders of the source image.
 *                      If "wrap" is set to UT_True, IP_ComputeChannelDerivates
 *                      assumes that
 *
 *                          srcChannel (-1, y) = srcChannel (w-1, y) and
 *                          srcChannel (x, -1) = srcChannel (x, h-1),
 *
 *                      where "w" and "h" are the width and height of "srcImg" in pixels.
 *
 *                      If "wrap" is set to UT_False, IP_ComputeChannelDerivates
 *                      assumes that
 *
 *                          srcChannel (-1, y) = srcChannel (0, y) and
 *                          srcChannel (x, -1) = srcChannel (x, 0).
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_ChangeChannelGamma
 *                      IP_RemapChannelValues
 *
 ***************************************************************************/

UT_Bool IP_ComputeChannelDerivates (IP_ImageId srcImg, FF_ImgChanType srcChannel,
                 IP_ImageId destImg, FF_ImgChanType horizontalChannel, FF_ImgChanType verticalChannel,
                 UT_Bool wrap)
{
    UT_MemState memstate;
    UT_Bool     success;

    if (srcChannel < 0 || srcChannel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, srcChannel,
                      IP_GetChannelTypeName (srcChannel));
        return UT_False;
    }
    if (horizontalChannel < 0 || horizontalChannel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, horizontalChannel,
                      IP_GetChannelTypeName (horizontalChannel));
        return UT_False;
    }
    if (verticalChannel < 0 || verticalChannel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, verticalChannel,
                      IP_GetChannelTypeName (verticalChannel));
        return UT_False;
    }
    if (srcImg->pimg.channel[srcChannel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, srcChannel,
                      IP_GetChannelTypeName (srcChannel));
        return UT_False;
    }
    if (destImg->pimg.channel[horizontalChannel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, horizontalChannel,
                      IP_GetChannelTypeName (horizontalChannel));
        return UT_False;
    }
    if (destImg->pimg.channel[verticalChannel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, verticalChannel,
                      IP_GetChannelTypeName (verticalChannel));
        return UT_False;
    }
    memstate = UT_MemRemember ();
    success = hvdiff (srcImg, srcChannel, destImg, horizontalChannel, verticalChannel, wrap);
    UT_MemRestore (memstate);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_Median
 *
 *      Usage:          Apply a median filter to an image.
 *
 *      Synopsis:       UT_Bool IP_Median(IP_ImageId img)
 *
 *      Description:    A three pixel wide median filter is applied to
 *                      image "img". First horizontally, then vertically.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_MedianSequence
 *                      IP_Blur
 *
 ***************************************************************************/

UT_Bool IP_Median (IP_ImageId img)
{
    UT_Bool success;

    median_h (img);
    median_v (img);
    success = UT_True;
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_MedianSequence
 *
 *      Usage:          Apply a median filter to an image sequence.
 *
 *      Synopsis:       UT_Bool IP_MedianSequence(
 *                              IP_ImageId srcImg1,
 *                              IP_ImageId srcImg2,
 *                              IP_ImageId srcImg3,
 *                              IP_ImageId destImg)
 *                      
 *      Description:    A median filter is applied to image sequence 
 *                      "srcImg1", "srcImg2", "srcImg3".
 *                      Corresponding pixel values in "srcImg1", "srcImg2" and
 *                      "srcImg3" are compared and the second-largest value is
 *                      stored in "destImg".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_Median
 *
 ***************************************************************************/

UT_Bool IP_MedianSequence
        (IP_ImageId srcImg1,
         IP_ImageId srcImg2,
         IP_ImageId srcImg3,
         IP_ImageId destImg)
{
    UT_Bool success;

    median_sequence (srcImg1, srcImg2, srcImg3, destImg);
    success = UT_True;
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_Dilatation
 *
 *      Usage:          Dilatation (expansion) of bright areas of an image.
 *
 *      Synopsis:       UT_Bool IP_Dilatation(
 *                              IP_ImageId img,
 *                              Int32 count)
 *
 *      Description:    A three by three pixel wide dilatation filter is 
 *                      applied "count" times to image "img".
 *
 *                      The dilatation filter expands bright areas in 
 *                      the image and eliminates dark pixels.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_Erosion
 *                      IP_Median
 *
 ***************************************************************************/

UT_Bool IP_Dilatation (IP_ImageId img, Int32 count)
{
    UT_MemState memstate;
    UT_Bool     success;

    if (count < 0) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_inotless, str_dilatationcount, 0);
        return UT_False;
    }
    memstate = UT_MemRemember ();
    success = dilat_eros (img, count, 1);
    UT_MemRestore (memstate);
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_Erosion
 *
 *      Usage:          Erosion (contraction) of bright areas of an image.
 *
 *      Synopsis:       UT_Bool IP_Erosion(
 *                              IP_ImageId img,
 *                              Int32 count)
 *
 *      Description:    A three by three pixel wide erosion filter is 
 *                      applied "count" times to image "img".
 *
 *                      The erosion filter contracts bright areas in 
 *                      the image and eliminates bright pixels.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All 
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_Dilatation
 *                      IP_Median
 *
 ***************************************************************************/

UT_Bool IP_Erosion (IP_ImageId img, Int32 count)
{
    UT_MemState memstate;
    UT_Bool     success;

    if (count < 0) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_inotless, str_erosioncount, 0);
        return UT_False;
    }
    memstate = UT_MemRemember ();
    success = dilat_eros (img, count, -1);
    UT_MemRestore (memstate);
    return success;
}

static UT_Bool threshold_filter (IP_ImageId src, IP_ImageId dest,
                                 Float32 t, UT_Bool overwrite)
{
    Int32 x, y, xmax, ymax, ithresh,
          vcenter, vbottom, vright;
    UInt8 *rbottom, *rcenter,
          *gbottom, *gcenter,
          *bbottom, *bcenter;
    FF_ImgDesc *psrc, *pdest;

    psrc  = &src->pimg;
    pdest = &dest->pimg;

    ithresh = (Int32) ((3.0 * 255.0) * t);
    xmax = src->pimg.desc.width  - 2;
    ymax = src->pimg.desc.height - 1;
    for (y = ymax; y >= 1; y--) {
        rbottom = FF_ImgUByteRow (psrc, FF_ImgChanTypeRed,   y-1);
        rcenter = FF_ImgUByteRow (psrc, FF_ImgChanTypeRed,   y);
        gbottom = FF_ImgUByteRow (psrc, FF_ImgChanTypeGreen, y-1);
        gcenter = FF_ImgUByteRow (psrc, FF_ImgChanTypeGreen, y);
        bbottom = FF_ImgUByteRow (psrc, FF_ImgChanTypeBlue,  y-1);
        bcenter = FF_ImgUByteRow (psrc, FF_ImgChanTypeBlue,  y);
        for (x = 1; x <= xmax; ++x) {
            vcenter = rcenter[x]   + gcenter[x]   + bcenter[x];
            vright  = rcenter[x+1] + gcenter[x+1] + bcenter[x+1] - vcenter;
            vbottom = rbottom[x]   + gbottom[x]   + bbottom[x]   - vcenter;
            if (ithresh < UT_ABS (vright) || ithresh < UT_ABS (vbottom)) {
                *FF_ImgUBytePixel (pdest, FF_ImgChanTypeRed,   x, y) = 255;
                *FF_ImgUBytePixel (pdest, FF_ImgChanTypeGreen, x, y) = 255;
                *FF_ImgUBytePixel (pdest, FF_ImgChanTypeBlue,  x, y) = 255;
            } else {
                if (overwrite) {
                    *FF_ImgUBytePixel (pdest, FF_ImgChanTypeRed,   x, y) = 0;
                    *FF_ImgUBytePixel (pdest, FF_ImgChanTypeGreen, x, y) = 0;
                    *FF_ImgUBytePixel (pdest, FF_ImgChanTypeBlue,  x, y) = 0;
                }
            }
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_Threshold
 *
 *      Usage:          Apply a threshold filter to an image.
 *
 *      Synopsis:       UT_Bool IP_Threshold 
 *                              (IP_ImageId srcImg, IP_ImageId destImg,
 *                               Float32 threshold, UT_Bool overwrite)
 *
 *      Description:    Each pixel of image "srcImg" greater than "threshold" is
 *                      set to white (255) in image "destImg". 
 *                      All other pixels in "destImg" are left unmodified,
 *                      if "overwrite" is "UT_False". 
 *                      If "overwrite" is "UT_True", these pixels are set to black (0).
 *
 *                      The "threshold" value must be in the range 0.0 .. 1.0.
 *
 *                      Note:
 *                      - Only RGB channels in UByte format are supported.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: RGB
 *                      Float format: No
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CutOff
 *
 ***************************************************************************/

UT_Bool IP_Threshold (IP_ImageId srcImg, IP_ImageId destImg, 
                      Float32 threshold, UT_Bool overwrite)
{
    UT_MemState memstate;
    UT_Bool     success;

    if (srcImg->pimg.channel[FF_ImgChanTypeRed]   != FF_ImgFmtTypeUByte ||
        srcImg->pimg.channel[FF_ImgChanTypeGreen] != FF_ImgFmtTypeUByte ||
        srcImg->pimg.channel[FF_ImgChanTypeBlue]  != FF_ImgFmtTypeUByte) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_rgb_ubyte);
        return UT_False;
    }

    memstate = UT_MemRemember ();
    success = threshold_filter (srcImg, destImg, threshold, overwrite);
    UT_MemRestore (memstate);
    return success;
}

static UT_Bool cutoff_filter (IP_ImageId img, Float32 tmin, Float32 tmax)
{
    Int32      x, y, xmax, ymax, chan;
    UInt8      t1, t2;
    UInt8      *upix;
    Float32    *fpix;
    FF_ImgDesc *pimg;

    pimg = &img->pimg;
    xmax = pimg->desc.width;
    ymax = pimg->desc.height;
    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan]) {
            continue;
        }
        switch (img->pimg.channel[chan]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                t1 = (UInt8) (tmin * 255.0);
                t2 = (UInt8) (tmax * 255.0);
                for (y = 0; y < ymax; ++y) {
                    for (x = 0; x < xmax; ++x) {
                        upix = FF_ImgUBytePixel (pimg, chan, x, y);
                        if (*upix <= t1) {
                            *upix = 0;
                        } else if (*upix >= t2) {
                            *upix = 255;
                        }
                    }
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                for (y = 0; y < ymax; ++y) {
                    for (x = 0; x < xmax; ++x) {
                        fpix = FF_ImgFloatPixel (pimg, chan, x, y);
                        if (*fpix <= tmin) {
                            *fpix = 0.0;
                        } else if (*fpix >= tmax) {
                            *fpix = 1.0;
                        }
                    }
                }
                break;
            }
            default: {
                UT_ErrFatal ("cutoff_filter", "unknown pixel data format");
            }
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_CutOff
 *
 *      Usage:          Apply a cut-off filter to an image.
 *
 *      Synopsis:       UT_Bool IP_CutOff( 
 *                              IP_ImageId img,
 *                              Float32 thresholdMin,
 *                              Float32 thresholdMax)
 *
 *      Description:    Each pixel of image "img" less than "thresholdMin" is
 *                      set to 0. Each pixel larger than "thresholdMax" is set
 *                      to 255 (for UByte channels) resp. 1.0 (for Float channels).
 *                      All other pixels are left unmodified.
 *                      The threshold values must be in the range [0.0, 1.0].
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if succesfull, else UT_False.
 *
 *      See also:       IP_Threshold
 *
 ***************************************************************************/

UT_Bool IP_CutOff (IP_ImageId img, Float32 thresholdMin, Float32 thresholdMax)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = cutoff_filter (img, thresholdMin, thresholdMax);
    UT_MemRestore (memstate);
    return success;
}

static UT_Bool markNonZeroPixels (IP_ImageId src, IP_ImageId dest,
                                  Float32 threshold, Int32 *numMarkedPixels)
{
    Int32      x, y, xmin, ymin;
    Int32      chan;
    Int32      numMarked = 0;
    UT_Bool    foundDiffPixel;
    FF_ImgDesc *psrc, *pdest;

    psrc  = &src->pimg;
    pdest = &dest->pimg;

    for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
        if (psrc->channel[chan] != pdest->channel[chan]) {
            UT_ErrSetNum (UT_ErrUnxpInput, str_diff_chans);
            return UT_False;
        }
    }

    xmin = UT_MIN (psrc->desc.width  - 1, pdest->desc.width  - 1);
    ymin = UT_MIN (psrc->desc.height - 1, pdest->desc.height - 1);

    for (y = ymin; y >= 0; y--) {
        for (x = 0; x <= xmin; x++) {
            foundDiffPixel = UT_False;
            for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
                if (!IP_StateDrawMask[chan]) {
                    continue;
                }
                switch (pdest->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        if ((UInt8)threshold < *FF_ImgUBytePixel (psrc, chan, x, y)) {
                            foundDiffPixel = UT_True;
                        }
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        if (threshold < *FF_ImgFloatPixel (psrc, chan, x, y)) {
                            foundDiffPixel = UT_True;
                        }
                        break;
                    }
                    default: {
                        UT_ErrFatal ("markNonZeroPixels", "unknown pixel data format");
                    }
                }
            }
            if (foundDiffPixel) {
                numMarked++;
            }
            for (chan = 0; chan < FF_NumImgChanTypes; chan++) {
                if (!IP_StateDrawMask[chan]) {
                    continue;
                }
                switch (pdest->channel[chan]) {
                    case FF_ImgFmtTypeNone: {
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        if (foundDiffPixel) {
                            *FF_ImgUBytePixel (pdest, chan, x, y) = IP_StateDrawColorUByte[chan];
                        } else {
                            *FF_ImgUBytePixel (pdest, chan, x, y) = *FF_ImgUBytePixel (psrc, chan, x, y);
                        }
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        if (foundDiffPixel) {
                            *FF_ImgFloatPixel (pdest, chan, x, y) = IP_StateDrawColorFloat[chan];
                        } else {
                            *FF_ImgFloatPixel (pdest, chan, x, y) = *FF_ImgFloatPixel (psrc, chan, x, y);
                        }
                        break;
                    }
                }
            }
        }
    }

    *numMarkedPixels = numMarked;
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_MarkNonZeroPixels
 *
 *      Usage:          Mark all non-zero pixels of an image.
 *
 *      Synopsis:       UT_Bool IP_MarkNonZeroPixels(
 *                              IP_ImageId srcImg,
 *                              IP_ImageId destImg,
 *                              Float32 threshold,
 *                              Int32 *numMarkedPixels)
 *
 *      Description:    Each pixel of image "srcImg" is checked, if its
 *                      value is larger than "threshold". If true, this pixel
 *                      is set in image "destImg" to the current draw color.
 *                      Otherwise the pixel value of "srcImg" is copied to "destImg".
 *                      The number of pixels marked with the current draw color is 
 *                      returned in "numMarkedPixels".
 *
 *                      Use this function in combination with IP_DifferenceImage
 *                      to check, if two images are identical.
 *
 *                      Tcl note: The function can be called in two ways.
 *                      1. The destination image is created by the Tcl wrapper
 *                         and returned by the function.
 *                      2. Direct mapping of the C function, where the destination
 *                         image has to be supplied by the caller. The caller is
 *                         responsible for correct image format and size.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_DifferenceImage
 *
 ***************************************************************************/

UT_Bool IP_MarkNonZeroPixels (IP_ImageId srcImg, IP_ImageId destImg, 
                              Float32 threshold, Int32 *numMarkedPixels)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = markNonZeroPixels (srcImg, destImg, threshold, numMarkedPixels);
    UT_MemRestore (memstate);
    return success;
}

static UT_Bool diffImgs (IP_ImageId src1, IP_ImageId src2, IP_ImageId dest)
{
    Int32      y, xmin, ymin;
    Int32      chan;
    FF_ImgDesc *psrc1, *psrc2, *pdest;

    psrc1 = &src1->pimg;
    psrc2 = &src2->pimg;
    pdest = &dest->pimg;

    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (psrc1->channel[chan] != psrc2->channel[chan] ||
            psrc1->channel[chan] != pdest->channel[chan]) {
            UT_ErrSetNum (UT_ErrUnxpInput, str_diff_chans);
            return UT_False;
        }
    }

    xmin = UT_MIN (psrc1->desc.width  - 1, psrc2->desc.width  - 1);
    xmin = UT_MIN (pdest->desc.width  - 1, xmin);

    ymin = UT_MIN (psrc1->desc.height - 1, psrc2->desc.height - 1);
    ymin = UT_MIN (pdest->desc.height - 1, ymin);

    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        if (!IP_StateDrawMask[chan]) {
            continue;
        }
        switch (pdest->channel[chan]) {
            case FF_ImgFmtTypeNone: {
                break;
            }
            case FF_ImgFmtTypeUByte: {
                UInt8 *src1data, *src2data;
                UInt8 *destdata, *deststop;
                for (y = ymin; y >= 0; y--) {
                    src1data = FF_ImgUByteRow (psrc1, chan, y);
                    src2data = FF_ImgUByteRow (psrc2, chan, y);
                    destdata = FF_ImgUByteRow (pdest, chan, y);
                    deststop = destdata + xmin + 1;
                    while (destdata < deststop) {
                        *destdata = UT_ABS (*src1data - *src2data);
                        src1data++;
                        src2data++;
                        destdata++;
                    }
                }
                break;
            }
            case FF_ImgFmtTypeFloat: {
                Float32 *src1data, *src2data;
                Float32 *destdata, *deststop;
                for (y = ymin; y >= 0; y--) {
                    src1data = FF_ImgFloatRow (psrc1, chan, y);
                    src2data = FF_ImgFloatRow (psrc2, chan, y);
                    destdata = FF_ImgFloatRow (pdest, chan, y);
                    deststop = destdata + xmin + 1;
                    while (destdata < deststop) {
                        *destdata = UT_ABS (*src1data - *src2data);
                        src1data++;
                        src2data++;
                        destdata++;
                    }
                }
                break;
            }
            default: {
                UT_ErrFatal ("diffImgs", "unknown pixel data format");
            }
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_DifferenceImage
 *
 *      Usage:          Generate a difference image of two images.
 *
 *      Synopsis:       UT_Bool IP_DifferenceImage(
 *                              IP_ImageId srcImg1,
 *                              IP_ImageId srcImg2,
 *                              IP_ImageId destImg)
 *
 *      Description:    The difference of all pixels of images "srcImg1"
 *                      and "srcImg2" is computed and stored in image "destImg":
 *
 *                          destImg[x,y] = abs (srcImg1[x,y] - srcImg2[x,y])
 *
 *                      Tcl note: The function can be called in two ways.
 *                      1. The destination image is created by the Tcl wrapper
 *                         and returned by the function.
 *                      2. Direct mapping of the C function, where the destination
 *                         image has to be supplied by the caller. The caller is
 *                         responsible for correct image format and size.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_MarkNonZeroPixels
 *
 ***************************************************************************/

UT_Bool IP_DifferenceImage (IP_ImageId srcImg1, IP_ImageId srcImg2, IP_ImageId destImg)
{
    UT_MemState memstate;
    UT_Bool     success;

    memstate = UT_MemRemember ();
    success = diffImgs (srcImg1, srcImg2, destImg);
    UT_MemRestore (memstate);
    return success;
}

/* Scan images "img1" and "img2" to see how similar they are. */

static void diff_channel (IP_ImageId img1, IP_ImageId img2, FF_ImgChanType chan,
                          Int32 less1[256], Int32 less2[256])
{
    UInt8 *pix1, *pix2, *stop;
    Int32         y, i, diff;

    for (i = 0; i < 256; i++) {
        less1[i] = 0;
        less2[i] = 0;
    }

    for (y=0; y<img1->pimg.desc.height; y++) {
        pix1 = FF_ImgUByteRow (&img1->pimg, chan, y);
        pix2 = FF_ImgUByteRow (&img2->pimg, chan, y);
        stop = pix1 + img1->pimg.desc.width;
        while (pix1 < stop) {
            if ((diff = *pix2 - *pix1) >= 0) {
                less1[diff]++;
            } else {
                less2[-diff]++;
            }
            pix1++;
            pix2++;
        }
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_DiffChannel
 *
 *      Usage:          Obtain the channel difference of two images.
 *
 *      Synopsis:       UT_Bool IP_DiffChannel(
 *                              IP_ImageId srcImg1,
 *                              IP_ImageId srcImg2, 
 *                              FF_ImgChanType channel, 
 *                              Int32 lessList1[256],
 *                              Int32 lessList2[256])
 *
 *      Description:    Channel "channel" of images "srcImg1" and "srcImg2" 
 *                      is read to see how much they differ.
 *
 *                      The following notation is used:
 *
 *                      0 < val < 256:
 *                          lessList1[val] == sum    "sum" pixels in image "srcImg1"
 *                                                   have a channel value which
 *                                                   is "val" less than the value
 *                                                   of image "srcImg2".
 *
 *                          lessList2[val] == sum    "sum" pixels in image "srcImg2"
 *                                                   have a channel value which
 *                                                   is "val" less than the value
 *                                                   of image "srcImg1".
 *
 *                      val == 0:
 *                          lessList1[val] == sum    "sum" pixels don't differ.
 *
 *                          lessList2[val]           always zero.
 *
 *                      Note: 
 *                          - Only channels in FF_ImgFmtTypeUByte are allowed.
 *                          - The two images must be equal in width and height.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: No
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_DifferenceImage
 *
 ***************************************************************************/

UT_Bool IP_DiffChannel (IP_ImageId srcImg1, IP_ImageId srcImg2, FF_ImgChanType channel,
                       Int32 lessList1[256], Int32 lessList2[256])
{
    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (srcImg1->pimg.channel[channel]    != FF_ImgFmtTypeUByte ||
        srcImg2->pimg.channel[channel] != FF_ImgFmtTypeUByte) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_chn_ubyte);
        return UT_False;
    }

    if (!((srcImg1->pimg.desc.height == srcImg2->pimg.desc.height) &&
          (srcImg1->pimg.desc.width  == srcImg2->pimg.desc.width))) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_equal);
        return UT_False;
    }

    diff_channel (srcImg1, srcImg2, channel, lessList1, lessList2);
    return UT_True;
}
