/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Image.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Declaration of the data types and functions
 *                      exported by the image processing library.
 *
 *      Additional documentation:
 *                      None.
 *
 **************************************************************************/

#ifndef __IP_IMAGE_H__
#define __IP_IMAGE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "UT_Compat.h"
#include "FF_Image.h"

/* Drawing modes */

typedef enum {
    IP_DrawModeNone    = -1,  /* Invalid draw mode */
    IP_DrawModeReplace =  0,  /* Replace old color with current color */
    IP_DrawModeAdd     =  1,  /* Add old color and current color */
    IP_DrawModeSub     =  2,  /* Subtract old color from current color */
    IP_DrawModeXor     =  3   /* Exclusive-or old color and current color */
} IP_DrawModeType;

#define IP_NumDrawModeTypes 4   /* Number of different drawing modes */

/* Pyramid filling modes for blending and warping */

typedef enum {
    IP_FillModeNone = -1,     /* Invalid fill mode */
    IP_FillModeFill =  0,     /* Fill with current color */
    IP_FillModeWrap =  1,     /* Tile with image */
    IP_FillModeClip =  2      /* Clip sampling coordinates */
} IP_FillModeType;

#define IP_NumFillModeTypes 3   /* Number of different fill modes */

/* Field interpolation modes. Used in function IP_Unlace. */

typedef enum {
    IP_UnlaceModeExtract = 0,   /* No field interpolation */
    IP_UnlaceModeReplicate,     /* Replicate the extracted scanlines */
    IP_UnlaceModeInterpolate    /* Interpolate between extracted scanlines */
} IP_UnlaceModeType;

#define IP_NumUnlaceModeTypes 3 /* Number of different field interp. modes */

/* An identifier for an image */

typedef struct IP_ImageStruct *IP_ImageId;

/* Functions */

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetFormat (const FF_ImgFmtType[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_GetFormat (FF_ImgFmtType[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetDrawMask (const UT_Bool[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_GetDrawMask (UT_Bool[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetDrawColor (const Float32[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_GetDrawColor (Float32[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetDrawMode (const IP_DrawModeType[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_GetDrawMode (IP_DrawModeType[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetColorCorrect
        (Float32, const CIE_hue *, const CIE_hue *,
         const CIE_hue *, const CIE_hue *);
extern POIMG_IMPORT_EXPORT void IP_GetColorCorrect
        (Float32 *, CIE_hue *, CIE_hue *, CIE_hue *, CIE_hue *);

extern POIMG_IMPORT_EXPORT Int32 IP_PushState (void);
extern POIMG_IMPORT_EXPORT Int32 IP_PopState (void);

extern POIMG_IMPORT_EXPORT UT_Bool IP_SetNumThreads (Int32);
extern POIMG_IMPORT_EXPORT Int32 IP_GetNumThreads (void);

extern POIMG_IMPORT_EXPORT UT_Bool IP_GetDrawModeType   (const char *, IP_DrawModeType *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetFillModeType   (const char *, IP_FillModeType *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetUnlaceModeType (const char *, IP_UnlaceModeType *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetChannelType    (const char *, FF_ImgChanType *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetChannelFormat  (const char *, FF_ImgFmtType *);

extern POIMG_IMPORT_EXPORT const char *IP_GetDrawModeTypeName   (IP_DrawModeType);
extern POIMG_IMPORT_EXPORT const char *IP_GetFillModeTypeName   (IP_FillModeType);
extern POIMG_IMPORT_EXPORT const char *IP_GetUnlaceModeTypeName (IP_UnlaceModeType);
extern POIMG_IMPORT_EXPORT const char *IP_GetChannelTypeName    (FF_ImgChanType);
extern POIMG_IMPORT_EXPORT const char *IP_GetChannelFormatName  (FF_ImgFmtType);

extern POIMG_IMPORT_EXPORT IP_ImageId IP_NewImage (Int32, Int32, Float32, const char *);
extern POIMG_IMPORT_EXPORT void IP_DeleteImage (IP_ImageId);

extern POIMG_IMPORT_EXPORT UT_Bool IP_HasChannel  (IP_ImageId, FF_ImgChanType);
extern POIMG_IMPORT_EXPORT void IP_GetImageInfo   (IP_ImageId, FF_ImgHdr *);
extern POIMG_IMPORT_EXPORT void IP_GetImageFormat (IP_ImageId, FF_ImgFmtType[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_WriteImage (IP_ImageId, const char *, const char *, const char *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_ReadImage (IP_ImageId, const char *, const char *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetFileInfo   (const char *, FF_ImgHdr *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetFileFormat (const char *, FF_ImgFmtType[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_WriteSimple (IP_ImageId, const char *, const char *, const char *);
extern POIMG_IMPORT_EXPORT IP_ImageId IP_NewImageFromFile (const char *, const char *);

extern POIMG_IMPORT_EXPORT UT_Bool IP_AddChannel    (IP_ImageId, FF_ImgChanType, FF_ImgFmtType);
extern POIMG_IMPORT_EXPORT UT_Bool IP_DeleteChannel (IP_ImageId, FF_ImgChanType);

extern POIMG_IMPORT_EXPORT void IP_DrawPixel (IP_ImageId, Int32, Int32);
extern POIMG_IMPORT_EXPORT void IP_DrawLine (IP_ImageId, Int32, Int32, Int32, Int32, Int32);
extern POIMG_IMPORT_EXPORT void IP_DrawRect (IP_ImageId, Int32, Int32, Int32, Int32, UT_Bool);
extern POIMG_IMPORT_EXPORT void IP_DrawEllipse (IP_ImageId, Int32, Int32, Int32, Int32, UT_Bool);
extern POIMG_IMPORT_EXPORT void IP_DrawCircle  (IP_ImageId, Int32, Int32, Int32, UT_Bool);
extern POIMG_IMPORT_EXPORT void IP_DrawBrushLine (IP_ImageId, IP_ImageId, Int32, Int32, Int32, Int32, Int32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_DrawBrush (IP_ImageId, IP_ImageId, Int32, Int32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_MergeBrush (IP_ImageId, IP_ImageId, IP_ImageId, Int32, Int32, Float32);
extern POIMG_IMPORT_EXPORT void IP_DrawText (IP_ImageId, Int32, Int32, const char *);
extern POIMG_IMPORT_EXPORT void IP_GetTextSize (const char *, Int32 *, Int32 *);
extern POIMG_IMPORT_EXPORT Int32 IP_GetTextScale (void);
extern POIMG_IMPORT_EXPORT UT_Bool IP_SetTextScale (Int32 scale);
extern POIMG_IMPORT_EXPORT void IP_BlankRect (IP_ImageId, Int32, Int32, Int32, Int32);
extern POIMG_IMPORT_EXPORT void IP_Blank (IP_ImageId);
extern POIMG_IMPORT_EXPORT void IP_CopyRect (IP_ImageId, IP_ImageId, Int32, Int32, Int32, Int32, Int32, Int32);
extern POIMG_IMPORT_EXPORT void IP_WrapRect (IP_ImageId, IP_ImageId, Int32, Int32, Int32, Int32, Int32, Int32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CopyChannel (IP_ImageId, IP_ImageId, FF_ImgChanType, FF_ImgChanType);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CopyImage (IP_ImageId srcimg, IP_ImageId destimg);

extern POIMG_IMPORT_EXPORT void IP_DrawAAPixel (IP_ImageId, Float32, Float32, const Float32[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_DrawAALine (IP_ImageId, Float32, Float32, Float32, Float32, const Float32[FF_NumImgChanTypes], const Float32[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT void IP_DrawAAText (IP_ImageId, Float32, Float32, const char *, const Float32[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_CompositeMatte
        (IP_ImageId, IP_ImageId, IP_ImageId, Float32, Float32,
         Float32, Float32, Float32, Float32, UT_Bool, IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CompositeColorMatte
        (IP_ImageId, IP_ImageId, IP_ImageId, Float32, Float32,
         Float32, Float32, Float32, Float32, UT_Bool, IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CompositeDepth (IP_ImageId, IP_ImageId, IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CrossDissolve (IP_ImageId, IP_ImageId, IP_ImageId, Float32);

extern POIMG_IMPORT_EXPORT UT_Bool IP_Interlace (IP_ImageId, IP_ImageId, IP_ImageId, UT_Bool);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Unlace (IP_ImageId, IP_ImageId, UT_Bool, IP_UnlaceModeType);

extern POIMG_IMPORT_EXPORT UT_Bool IP_WarpKeypoint
        (IP_ImageId, IP_ImageId, Int32, const Float32[], const Float32[],
         const Float32[], const Float32[], Int32, Float32, IP_FillModeType);
extern POIMG_IMPORT_EXPORT UT_Bool IP_BlendKeypoint
        (IP_ImageId, IP_ImageId, IP_ImageId, Int32, const Float32[], const Float32[],
         const Float32[], const Float32[], const Float32[], Int32, Float32, IP_FillModeType);
extern POIMG_IMPORT_EXPORT UT_Bool IP_WarpFunct
        (IP_ImageId, IP_ImageId, UT_Bool (*) (Float32, Float32, Float32 *, Float32 *),
         UT_Bool (*) (Float32, Float32, Float32 *, Float32 *, Float32 *, Float32 *),
         Int32, IP_FillModeType);
extern POIMG_IMPORT_EXPORT UT_Bool IP_BlendFunct
        (IP_ImageId, IP_ImageId, IP_ImageId,
         UT_Bool (*) (Float32, Float32, Float32 *, Float32 *),
         UT_Bool (*) (Float32, Float32, Float32 *, Float32 *, Float32 *, Float32 *),
         UT_Bool (*) (Float32, Float32, Float32 *, Float32 *),
         UT_Bool (*) (Float32, Float32, Float32 *, Float32 *, Float32 *, Float32 *),
         UT_Bool (*) (Float32, Float32, Float32 *),
         Int32, IP_FillModeType);

extern POIMG_IMPORT_EXPORT void  IP_ConvertUByteRow (IP_ImageId, Int32, UInt8 **, UInt8 **);
extern POIMG_IMPORT_EXPORT UInt8 *IP_GetUByteRow (IP_ImageId, FF_ImgChanType, Int32);
extern POIMG_IMPORT_EXPORT Float32 *IP_GetFloatRow (IP_ImageId, FF_ImgChanType, Int32);

extern POIMG_IMPORT_EXPORT UT_Bool IP_GetSample (const IP_ImageId, Float32, Float32, Float32[FF_NumImgChanTypes]);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetPixel  (const IP_ImageId, Int32, Int32, Float32[FF_NumImgChanTypes]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_ChangeChannelGamma (IP_ImageId, FF_ImgChanType, Float32, Float32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_RemapChannelValues (IP_ImageId, FF_ImgChanType, Int32, const Float32[]);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Blur (IP_ImageId, Float32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_DepthBlur (IP_ImageId, Float32, Float32, Float32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_ComputeChannelDerivates (IP_ImageId, FF_ImgChanType, IP_ImageId, FF_ImgChanType, FF_ImgChanType, UT_Bool);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Median (IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_MedianSequence (IP_ImageId, IP_ImageId, IP_ImageId, IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Dilatation (IP_ImageId, Int32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Erosion (IP_ImageId, Int32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_CutOff (IP_ImageId, Float32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Threshold (IP_ImageId, IP_ImageId, Float32, UT_Bool);
extern POIMG_IMPORT_EXPORT UT_Bool IP_MarkNonZeroPixels (IP_ImageId, IP_ImageId, Float32, Int32*);
extern POIMG_IMPORT_EXPORT UT_Bool IP_DifferenceImage (IP_ImageId, IP_ImageId, IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Flicker (IP_ImageId);

extern POIMG_IMPORT_EXPORT void IP_FlipDiagonal (IP_ImageId);
extern POIMG_IMPORT_EXPORT void IP_FlipHorizontal (IP_ImageId);
extern POIMG_IMPORT_EXPORT void IP_FlipVertical (IP_ImageId);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Rotate90 (IP_ImageId, IP_ImageId, Int32);

extern POIMG_IMPORT_EXPORT UT_Bool IP_ScaleRect
        (IP_ImageId, IP_ImageId, Int32, Int32, Int32, Int32, Int32, Int32, Int32, Int32,
         UT_Bool, UT_Bool);

extern POIMG_IMPORT_EXPORT UT_Bool IP_CreateChannelNoise (IP_ImageId, FF_ImgChanType, Int32, Int32, Int32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_JuliaSet (IP_ImageId, Float32, Float32, Float32, Float32, Float32, 
                                                   Int32, const UInt8 [256], const UInt8 [256], const UInt8 [256]);
extern POIMG_IMPORT_EXPORT UT_Bool IP_Mandelbrot (IP_ImageId, Float32, Float32, Float32, 
                                                     Int32, const UInt8 [256], const UInt8 [256], const UInt8 [256]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_GetChannelHistogram (IP_ImageId, FF_ImgChanType, Int32 [256]);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetChannelStats (IP_ImageId, FF_ImgChanType, Int32, Int32, Int32, Int32,
        Float64 *, Float64 *, UInt32 *, UT_Bool);
extern POIMG_IMPORT_EXPORT UT_Bool IP_GetChannelRange (IP_ImageId, FF_ImgChanType, Int32, Int32, Int32, Int32, Float32 *, Float32 *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_DiffChannel (IP_ImageId, IP_ImageId, FF_ImgChanType, Int32 [256], Int32 [256]);

extern POIMG_IMPORT_EXPORT UT_Bool IP_CorrelateChannel
        (IP_ImageId, IP_ImageId, FF_ImgChanType, Int32, Int32, Int32, Int32, Int32, Int32, Float32 *);
extern POIMG_IMPORT_EXPORT UT_Bool IP_FindChannelPattern
        (IP_ImageId, IP_ImageId, FF_ImgChanType, Int32, Int32, Int32, Int32, Int32, Int32,
         Int32, Int32, Float32 *, Float32 *, Float32 *);

extern POIMG_IMPORT_EXPORT UT_Bool IP_ShearVertical (IP_ImageId, Float32, Float32);
extern POIMG_IMPORT_EXPORT UT_Bool IP_ShearHorizontal (IP_ImageId, Float32, Float32);

extern POIMG_IMPORT_EXPORT UT_Bool IP_Stamp (IP_ImageId, IP_ImageId, Int32, Int32, Float32, Float32, Float32);

#ifdef __cplusplus
}
#endif

#endif /* __IP_IMAGE_H__ */
