/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Info.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Various functions to obtain information about images.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_GetChannelHistogram
 *                      IP_HasChannel
 *                      IP_GetImageInfo
 *                      IP_GetImageFormat
 m                      IP_GetFileInfo
 *                      IP_GetFileFormat
 *                      IP_GetChannelStats
 *                      IP_GetChannelRange
 *                      IP_GetDrawModeType
 *                      IP_GetFillModeType
 *                      IP_GetUnlaceModeType
 *                      IP_GetChannelType
 *                      IP_GetDrawModeTypeName
 *                      IP_GetFillModeTypeName
 *                      IP_GetUnlaceModeTypeName
 *                      IP_GetChannelTypeName
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"
#include "UT_Const.h"
#include "UT_Macros.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Misc.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#define str_InvalidDrawModeName   "Invalid draw mode name: %s."
#define str_InvalidFillModeName   "Invalid fill mode name: %s."
#define str_InvalidUnlaceModeName "Invalid unlace mode name: %s."

/* Possible draw mode names.
 * Note: This must be synchronized with the enumeration IP_DrawModeType in
 * header file IP_Image.h.
 */
static const char *drawModeTypeNames[] = {
    "REPLACE", "ADD", "SUB", "XOR"
};

/* Possible fill mode names.
 * Note: This must be synchronized with the enumeration IP_FillModeType in
 * header file IP_Image.h.
 */
static const char *fillModeTypeNames[] = {
    "FILL", "WRAP", "CLIP"
};

/* Possible unlace mode names.
 * Note: This must be synchronized with the enumeration IP_UnlaceModeType in
 * header file IP_Image.h.
 */
static const char *unlaceModeTypeNames[] = {
    "EXTRACT", "REPLICATE", "INTERPOLATE"
};

/* Generate a histogram of channel "chan" of image "img". 
   The histogram information is returned in "buf". 
*/

static void histo (IP_ImageId img, FF_ImgChanType chan, Int32 buf[256])
{
    UInt8 *ubdata, *ubstop;
    Float32 *fdata,  *fstop;
    Int32  y, i;

    for (i = 0; i < 256; i++) {
        buf[i] = 0;
    }

    switch (img->pimg.channel[chan]) { 
        case FF_ImgFmtTypeUByte: {
            for (y = 0; y < img->pimg.desc.height; ++y) {
                ubdata = FF_ImgUByteRow (&img->pimg, chan, y);
                ubstop = ubdata + img->pimg.desc.width;
                while (ubdata < ubstop) {
                    buf[*ubdata]++;
                    ++ubdata;
                }
            }
            break;
        }
        case FF_ImgFmtTypeFloat: {
            for (y = 0; y < img->pimg.desc.height; ++y) {
                fdata = FF_ImgFloatRow (&img->pimg, chan, y);
                fstop = fdata + img->pimg.desc.width;
                while (fdata < fstop) {
                    buf[UT_MIN (255, UT_MAX (0, (int)(*fdata * 255)))]++;
                    ++fdata;
                }
            }
            break;
        }
        default: {
            UT_ErrFatal ("histo", "invalid pixel format");
            break;
        }
    }
    return;
}

/* Scan channel "chan" of image "image" and return the arithmetic mean value
 * and the standard deviation.
 * The function does not check, if channel "chan" and the matte channel
 * are available and in the correct format.
 * This is in the responsibility of the calling procedure. 
 * See IP_GetChannelStats for a detailled description.
 */

static UT_Bool getChanStats (IP_ImageId img, FF_ImgChanType chan,
                             Int32 x1, Int32 y1, Int32 x2, Int32 y2,
                             Float64 *meanVal, Float64 *stdDev,
                             UInt32 *numPix, UT_Bool useMatte)
{
    Int32 y, width;
    Float32 *startChan = NULL, *stopChan = NULL;
    Float32 *startMatte = NULL;
    Float32 *tmpMatte, *tmpChan;
    FF_ImgDesc *pimg;
    UInt32  count = 0;
    Float64 sum = 0.0;

    width = img->pimg.desc.width;
    pimg  = &img->pimg;

    if (!(tmpChan  = UT_MemTempArray (width, Float32)) ||
        !(tmpMatte = UT_MemTempArray (width, Float32))) {
        return UT_False;
    }

    for (y=y1; y<=y2; y++) {
        if (useMatte) {
            switch (pimg->channel[FF_ImgChanTypeMatte]) {
                case FF_ImgFmtTypeUByte: {
                    IP_CopyUByte2Float (width, FF_ImgUByteRow (pimg, FF_ImgChanTypeMatte, y), tmpMatte);
                    startMatte = tmpMatte;
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    startMatte = FF_ImgFloatRow (pimg, FF_ImgChanTypeMatte, y);
                    break;
                }
                default: {
                    UT_ErrFatal ("getChanStats", "invalid pixel format");
                    break;
                }
            }
        }
        switch (pimg->channel[chan]) {
            case FF_ImgFmtTypeUByte: {
                IP_CopyUByte2FloatAsIs (width, FF_ImgUByteRow (pimg, chan, y), tmpChan);
                startChan = tmpChan;
                break;
            }
            case FF_ImgFmtTypeFloat: {
                startChan = FF_ImgFloatRow (pimg, chan, y);
                break;
            }
            default: {
                UT_ErrFatal ("getChanStats", "invalid pixel format");
                break;
            }
        }
        stopChan  = startChan + x2;
        startChan = startChan + x1;
        if (useMatte) {
            while (startChan <= stopChan) {
                if (*startMatte != 0.0) {
                    sum += (Float64)(*startChan * *startMatte);
                    count++;
                }
                startChan++;
                startMatte++;
            }
        } else {
            while (startChan <= stopChan) {
                sum += (Float64)*startChan;
                startChan++;
		count++;
            }
        }
    }
    *meanVal = (sum / count);
    *numPix = count;

    sum = 0.0;
    for (y=y1; y<=y2; y++) {
        if (useMatte) {
            switch (pimg->channel[FF_ImgChanTypeMatte]) {
                case FF_ImgFmtTypeUByte: {
                    IP_CopyUByte2Float (width, FF_ImgUByteRow (pimg, FF_ImgChanTypeMatte, y), tmpMatte);
                    startMatte = tmpMatte;
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    startMatte = FF_ImgFloatRow (pimg, FF_ImgChanTypeMatte, y);
                    break;
                }
                default: {
                    UT_ErrFatal ("getChanStats", "invalid pixel format");
                    break;
                }
            }
        }
        switch (pimg->channel[chan]) {
            case FF_ImgFmtTypeUByte: {
                IP_CopyUByte2FloatAsIs (width, FF_ImgUByteRow (pimg, chan, y), tmpChan);
                startChan = tmpChan;
                break;
            }
            case FF_ImgFmtTypeFloat: {
                startChan = FF_ImgFloatRow (pimg, chan, y);
                break;
            }
            default: {
                UT_ErrFatal ("getChanStats", "invalid pixel format");
                break;
            }
        }
        stopChan  = startChan + x2;
        startChan = startChan + x1;
        if (useMatte) {
            while (startChan <= stopChan) {
                if (*startMatte != 0.0) {
                    sum += UT_SQR (*meanVal - (Float64)(*startChan * *startMatte));
                }
                startChan++;
                startMatte++;
            }
        } else {
            while (startChan <= stopChan) {
                sum += UT_SQR (*meanVal - (Float64)*startChan);
                startChan++;
            }
        }
    }
    if (count == 1) {
        *stdDev = sum;
    } else {
        *stdDev = sqrt (sum / (count-1));
    }

    UT_MemFree (tmpChan);
    UT_MemFree (tmpMatte);
    return UT_True;
}

/* Scan channel "chan" of image "image" and return the minimum and maximum 
 * values in "minVal", resp. "maxVal". 
 * The channel has to be in UInt8 format.
 * The function does not check, if channel "chan" is of type FF_ImgFmtTypeUByte.
 * This is in the responsibility of the calling procedure. 
 */

static void getUByteRange (IP_ImageId img, FF_ImgChanType chan,
                           Int32 x1, Int32 y1, Int32 x2, Int32 y2,
                           UInt8 *minVal, UInt8 *maxVal)
{
    Int32 y;
    UInt8 low, high;
    UInt8 *startChan = NULL, *stopChan = NULL;
    FF_ImgDesc *pimg;

    low  = MaxUInt8;
    high = 0;
    pimg = &img->pimg;

    for (y=y1; y<=y2; y++) {
        startChan = FF_ImgUByteRow (pimg, chan, y); 
        stopChan  = startChan + x2;
        startChan = startChan + x1;
        while (startChan <= stopChan) {
            low  = UT_MIN (low,  *startChan);
            high = UT_MAX (high, *startChan);
            startChan++;
        }
    }
    *minVal = low;
    *maxVal = high;
    return;
}

/* Scan channel "chan" of image "image" and return the minimum and maximum 
 * values in "minVal", resp. "maxVal". 
 * The channel has to be in Float32 format.
 * The function does not check, if channel "chan" is of type FF_ImgFmtTypeFloat.
 * This is in the responsibility of the calling procedure. 
 */

static void getFloatRange (IP_ImageId img, FF_ImgChanType chan,
                           Int32 x1, Int32 y1, Int32 x2, Int32 y2,
                           Float32 *minVal, Float32 *maxVal)
{
    Int32 y;
    Float32 low, high;
    Float32 *startChan = NULL, *stopChan = NULL;
    FF_ImgDesc *pimg;

    low  =  MaxFloat32;
    high = -MaxFloat32;
    pimg = &img->pimg;

    for (y=y1; y<=y2; y++) {
        startChan = FF_ImgFloatRow (pimg, chan, y); 
        stopChan  = startChan + x2;
        startChan = startChan + x1;
        while (startChan <= stopChan) {
            low  = UT_MIN (low,  *startChan);
            high = UT_MAX (high, *startChan);
            startChan++;
        }
    }
    *minVal = low;
    *maxVal = high;
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelHistogram
 *
 *      Usage:          Obtain a histogram of one channel of an image.  
 *
 *      Synopsis:       UT_Bool IP_GetChannelHistogram(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              Int32 histogramList[256])
 *
 *      Description:    A histogram of channel "channel" of image "img" is
 *                      generated. The result is returned in "histogramList".
 *
 *                      A histogram shows the distribution of pixel values.
 *                      The range from 0.0 to 1.0 is scaled into the range
 *                      from 0 to 255; pixel values, which are less than 0
 *                      or greater than 255 after scaling, are clamped to
 *                      0 or 255. For every pixel, whose scaled value is "i",
 *                      histogramList[i] is incremented by 1.
 *                      For example, (histogramList[5] == 45) means that there
 *                      are 45 pixels in the image, whose scaled pixel value is 5.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetChannelStats
 *                      IP_GetChannelRange
 *
 ***************************************************************************/

UT_Bool IP_GetChannelHistogram (IP_ImageId img, FF_ImgChanType channel, Int32 histogramList[256])
{
    if (channel < 0 ||
        channel >= FF_NumImgChanTypes ||
        img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      FF_ImgGetChannelTypeName (channel));
        return UT_False;
    }
    histo (img, channel, histogramList);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelStats
 *
 *      Usage:          Obtain statistics of an image channel.
 *
 *      Synopsis:       UT_Bool IP_GetChannelStats(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              Int32 x1, Int32 y1,
 *                              Int32 x2, Int32 y2,
 *                              Float32 *meanValue,
 *                              Float32 *stdDev,
 *                              UInt32 *numPixels,
 *                              UT_Bool useMatte)
 *
 *      Description:    Channel "channel" of image "img" is scanned.
 *                      If the values for rectangle "x1, y1" and "x2, y2"
 *                      are set to a negative value (eg. -1), the whole
 *                      image is used for scanning. Otherwise only the
 *                      pixels within the rectangle are considered.
 *                      The arithmetic mean value of all pixels in the
 *                      channel is returned in parameter "meanValue".
 *                      The standard deviation is returned in "stdDev".
 *                      If "useMatte" is set to UT_True, only the pixels
 *                      with a matte value unequal to zero are taken into
 *                      consideration.
 *                      The number of pixels used to calculate the statistics
 *                      is returned in parameter "numPixels".
 *                      In the simple case, when "useMatte" is UT_False and
 *                      the whole image is considered, this value equals
 *                      to the product of the image width and height.
 *
 *                      Note: The mean value and standard deviation are
 *                            always returned as floating point value, even
 *                            if the channel is of type FF_ImgFmtUByte.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetChannelRange
 *                      IP_GetChannelHistogram
 *
 ***************************************************************************/

UT_Bool IP_GetChannelStats (IP_ImageId img, FF_ImgChanType channel, 
                            Int32 x1, Int32 y1, Int32 x2, Int32 y2,
                            Float64 *meanValue, Float64 *stdDev,
                            UInt32 *numPixels, UT_Bool useMatte)
{
    UT_Bool success = UT_False;

    if (channel < 0 ||
        channel >= FF_NumImgChanTypes ||
        img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    /* Check the format of the matte channel, if useMatte is set. */
    if (useMatte && img->pimg.channel[FF_ImgChanTypeMatte] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, FF_ImgChanTypeMatte,
                      IP_GetChannelTypeName (FF_ImgChanTypeMatte));
        return UT_False;
    }

    if (x1 < 0) {
        x1 = 0;
    }
    if (y1 < 0) {
        y1 = 0;
    }
    if (x2 < x1 || x2 >= img->pimg.desc.width) {
        x2 = img->pimg.desc.width-1;
    }
    if (y2 < y1 || y2 >= img->pimg.desc.height) {
        y2 = img->pimg.desc.height-1;
    }

    switch (img->pimg.channel[channel]) {
        case FF_ImgFmtTypeUByte:
        case FF_ImgFmtTypeFloat: {
            success = getChanStats (img, channel, x1, y1, x2, y2, meanValue, stdDev, numPixels, useMatte);
            break;
        }
        default: {
            UT_ErrFatal ("IP_GetChannelStats", "invalid pixel format");
            break;
        }
    }
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelRange
 *
 *      Usage:          Obtain the range of an image channel.
 *
 *      Synopsis:       UT_Bool IP_GetChannelRange(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              Int32 x1, Int32 y1,
 *                              Int32 x2, Int32 y2,
 *                              Float32 *minValue, Float32 *maxValue)
 *
 *      Description:    Channel "channel" of image "img" is scanned.
 *                      If the values for rectangle "x1, y1" and "x2, y2"
 *                      are set to a negative value (eg. -1), the whole
 *                      image is used for scanning. Otherwise only the
 *                      pixels within the rectangle are considered.
 *                      The minimum resp. maximum channel values are returned
 *                      in "minValue" resp. "maxValue".
 *
 *                      Note: The values are always returned in floating point
 *                      format.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetChannelStats
 *                      IP_GetChannelHistogram
 *
 ***************************************************************************/

UT_Bool IP_GetChannelRange (IP_ImageId img, FF_ImgChanType channel, 
                            Int32 x1, Int32 y1, Int32 x2, Int32 y2,
                            Float32 *minValue, Float32 *maxValue)
{
    UT_Bool success = UT_False;

    if (channel < 0 ||
        channel >= FF_NumImgChanTypes ||
        img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }

    if (x1 < 0) {
        x1 = 0;
    }
    if (y1 < 0) {
        y1 = 0;
    }
    if (x2 < x1 || x2 >= img->pimg.desc.width) {
        x2 = img->pimg.desc.width-1;
    }
    if (y2 < y1 || y2 >= img->pimg.desc.height) {
        y2 = img->pimg.desc.height-1;
    }

    switch (img->pimg.channel[channel]) {
        case FF_ImgFmtTypeUByte: {
            UInt8 imin, imax;
            getUByteRange (img, channel, x1, y1, x2, y2, &imin, &imax);
            *minValue = imin;
            *maxValue = imax;
            success = UT_True;
            break;
        }
        case FF_ImgFmtTypeFloat: {
            getFloatRange (img, channel, x1, y1, x2, y2, minValue, maxValue);
            success = UT_True;
            break;
        }
        default: {
            UT_ErrFatal ("IP_GetChannelRange", "invalid pixel format");
            success = UT_False;
            break;
        }
    }
    return success;
}

/***************************************************************************
 *[@e
 *      Name:           IP_HasChannel
 *
 *      Usage:          Check existence of a channel in an image.
 *
 *      Synopsis:       UT_Bool IP_HasChannel(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel)
 *
 *      Description:    Check, if channel "channel" exists in image "img".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if channel exists, else UT_False.
 *
 *      See also:       IP_AddChannel
 *                      IP_GetImageInfo
 *                      IP_GetImageFormat
 *
 ***************************************************************************/

UT_Bool IP_HasChannel (IP_ImageId img, FF_ImgChanType channel)
{
    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        return UT_False;
    }
    if (img->pimg.channel[channel] != FF_ImgFmtTypeNone) {
        return UT_True;
    }
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetImageInfo
 *
 *      Usage:          Read the descriptor of an image.
 *
 *      Synopsis:       void IP_GetImageInfo(
 *                           IP_ImageId img,
 *                           FF_ImgHdr *desc)
 *
 *      Description:    The image descriptor (width, height, aspect ratio,
 *                      gamma factor and color correction parameters) of
 *                      image "img" are written to "desc".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   None.
 *
 *      See also:       IP_GetImageFormat
 *                      IP_GetFileInfo
 *                      IP_HasChannel
 *
 ***************************************************************************/

void IP_GetImageInfo (IP_ImageId img, FF_ImgHdr *desc)
{
    if (desc) {
        *desc = img->pimg.desc;
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetImageFormat
 *
 *      Usage:          Read the pixel data format of an image.
 *
 *      Synopsis:       void IP_GetImageInfo(
 *                           IP_ImageId img,
 *                           FF_ImgFmtType formatList[FF_NumImgChanTypes])
 *
 *      Description:    The pixel data formats of the channels in "img"
 *                      are written to "formatList".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   None.
 *
 *      See also:       IP_GetImageInfo
 *                      IP_GetFileFormat
 *                      IP_HasChannel
 *
 ***************************************************************************/

void IP_GetImageFormat (IP_ImageId img, FF_ImgFmtType formatList[FF_NumImgChanTypes])
{
    Int32 i;

    if (formatList) {
        for (i = 0; i < FF_NumImgChanTypes; ++i) {
            formatList[i] = img->pimg.channel[i];
        }
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetFileInfo
 *
 *      Usage:          Read the image descriptor from a file.
 *
 *      Synopsis:       UT_Bool IP_GetFileInfo(
 *                              const char *fileName,
 *                              FF_ImgHdr *desc)
 *
 *      Description:    The header of pixel file "fileName" is read. Width,
 *                      height, aspect ratio, gamma factor and color correction
 *                      parameters of the image stored in the file are written
 *                      to "desc".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetFileFormat
 *                      IP_GetImageInfo
 *
 ***************************************************************************/

UT_Bool IP_GetFileInfo (const char *fileName, FF_ImgHdr *desc)
{
    FF_ImgFileDesc pxf;

    memset (&pxf, 0, sizeof (FF_ImgFileDesc));

    if (!FF_ImgOpenRead (fileName, &pxf, "")) {
        return UT_False;
    }
    if (desc) {
        *desc = pxf.desc;
    }
    FF_ImgClose (&pxf);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetFileFormat
 *
 *      Usage:          Read the image format from a file.
 *
 *      Synopsis:       UT_Bool IP_GetFileFormat(
 *                              const char *fileName,
 *                              FF_ImgFmtType formatList[FF_NumImgChanTypes])
 *
 *      Description:    The header of pixel file "fileName" is read. 
 *                      The pixel data formats of the channels stored in the
 *                      file are written to "formatList".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetFileInfo
 *                      IP_GetImageInfo
 *
 ***************************************************************************/

UT_Bool IP_GetFileFormat (const char *fileName, FF_ImgFmtType formatList[FF_NumImgChanTypes])
{
    FF_ImgFileDesc pxf;
    Int32          i;

    memset (&pxf, 0, sizeof (FF_ImgFileDesc));

    if (!FF_ImgOpenRead (fileName, &pxf, "")) {
        return UT_False;
    }
    if (formatList) {
        for (i = 0; i < FF_NumImgChanTypes; ++i) {
            formatList[i] = pxf.channel[i];
        }
    }
    FF_ImgClose (&pxf);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetDrawModeType
 *
 *      Usage:          Get the numeric value of a draw mode name.
 *
 *      Synopsis:       UT_Bool IP_GetDrawModeType(
 *                              const char *drawModeTypeName,
 *                              IP_DrawModeType *drawModeType)
 *
 *      Description:    The numeric value of draw mode name "drawModeTypeName"
 *                      is returned in "drawModeType".
 *
 *                      Possible draw mode names:
 *                          "REPLACE", "ADD", "SUB", "XOR"
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: N/A
 *                      Float format: N/A
 *
 *      Return value:   UT_True, if successful, otherwise UT_False.
 *
 *      See also:       IP_GetDrawModeTypeName
 *                      IP_GetChannelType
 *                      IP_GetFillModeType
 *                      IP_GetChannelFormat
 *                      IP_GetUnlaceModeType
 *
 ***************************************************************************/

UT_Bool IP_GetDrawModeType (const char *drawModeTypeName, IP_DrawModeType *drawModeType)
{
    Int32 i;

    if (IP_NumDrawModeTypes != sizeof(drawModeTypeNames)/sizeof(drawModeTypeNames[0])) {
        UT_ErrFatal ("IP_GetDrawModeType", "Inconsistent number of draw modes");
        return UT_False;
    }

    for (i=0; i<sizeof(drawModeTypeNames)/sizeof(drawModeTypeNames[0]); i++) {
        if (UT_StringEqual (drawModeTypeName, drawModeTypeNames[i])) {
            *drawModeType = (IP_DrawModeType) i;
            return UT_True;
        }
    }
    UT_ErrSetNum (UT_ErrParamInvalid, str_InvalidDrawModeName, drawModeTypeName);
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetDrawModeTypeName
 *
 *      Usage:          Get the name of a draw mode type.
 *
 *      Synopsis:       const char *IP_GetDrawModeTypeName(
 *                                  IP_DrawModeType drawModeType)
 *
 *      Description:    Possible draw mode type names:
 *                          "REPLACE", "ADD", "SUB", "XOR"
 *                          "UNKNOWN" for invalid draw mode type.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: N/A
 *                      Float format: N/A
 *
 *      Return value:   A pointer to the name of the draw mode type.
 *
 *      See also:       IP_GetDrawModeType
 *                      IP_GetChannelTypeName
 *                      IP_GetFillModeTypeName
 *                      IP_GetChannelFormatName
 *                      IP_GetUnlaceTypeName
 *
 ***************************************************************************/

const char *IP_GetDrawModeTypeName (IP_DrawModeType drawModeType)
{
    if (drawModeType < IP_DrawModeNone || drawModeType >= IP_NumDrawModeTypes) {
        return "UNKNOWN";
    }
    return drawModeTypeNames[drawModeType];
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetFillModeType
 *
 *      Usage:          Get the numeric value of a fill mode name.
 *
 *      Synopsis:       UT_Bool IP_GetFillModeType(
 *                              const char *fillModeTypeName,
 *                              IP_FillModeType *fillModeType)
 *
 *      Description:    The numeric value of fill mode name "fillModeTypeName"
 *                      is returned in "fillModeType".
 *
 *                      Possible fill mode names:
 *                          "FILL", "WRAP", "CLIP"
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: N/A
 *                      Float format: N/A
 *
 *      Return value:   UT_True, if successful, otherwise UT_False.
 *
 *      See also:       IP_GetFillModeTypeName
 *                      IP_GetChannelType
 *                      IP_GetDrawModeType
 *                      IP_GetChannelFormat
 *                      IP_GetUnlaceType
 *
 ***************************************************************************/

UT_Bool IP_GetFillModeType (const char *fillModeTypeName, IP_FillModeType *fillModeType)
{
    Int32 i;

    if (IP_NumFillModeTypes != sizeof(fillModeTypeNames)/sizeof(fillModeTypeNames[0])) {
        UT_ErrFatal ("IP_GetFillModeType", "Inconsistent number of fill modes");
        return UT_False;
    }

    for (i=0; i<sizeof(fillModeTypeNames)/sizeof(fillModeTypeNames[0]); i++) {
        if (UT_StringEqual (fillModeTypeName, fillModeTypeNames[i])) {
            *fillModeType = (IP_FillModeType) i;
            return UT_True;
        }
    }
    UT_ErrSetNum (UT_ErrParamInvalid, str_InvalidFillModeName, fillModeTypeName);
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetFillModeTypeName
 *
 *      Usage:          Get the name of a fill mode type.
 *
 *      Synopsis:       const char *IP_GetFillModeTypeName(
 *                                  IP_FillModeType fillModeType)
 *
 *      Description:    Possible fill mode type names:
 *                          "REPLACE", "ADD", "SUB", "XOR"
 *                          "UNKNOWN" for invalid fill mode type.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: N/A
 *                      Float format: N/A
 *
 *      Return value:   A pointer to the name of the fill mode type.
 *
 *      See also:       IP_GetFillModeType
 *                      IP_GetChannelTypeName
 *                      IP_GetDrawModeTypeName
 *                      IP_GetChannelFormatName
 *                      IP_GetUnlaceTypeName
 *
 ***************************************************************************/

const char *IP_GetFillModeTypeName (IP_FillModeType fillModeType)
{
    if (fillModeType < IP_FillModeNone || fillModeType >= IP_NumFillModeTypes) {
        return "UNKNOWN";
    }
    return fillModeTypeNames[fillModeType];
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetUnlaceModeType
 *
 *      Usage:          Get the numeric value of an unlace mode name.
 *
 *      Synopsis:       UT_Bool IP_GetUnlaceModeType
 *                              (const char *unlaceModeTypeName,
 *                              IP_UnlaceModeType *unlaceModeType)
 *
 *      Description:    The numeric value of unlace mode name
 *                      "unlaceModeTypeName" is returned in "unlaceModeType".
 *
 *                      Possible unlace mode names:
 *                          "EXTRACT", "REPLICATE", "INTERPOLATE"
 *
 *      Return value:   UT_True, if successful, otherwise UT_False.
 *
 *      See also:       IP_GetUnlaceModeTypeName
 *                      IP_GetChannelType
 *                      IP_GetDrawModeType
 *                      IP_GetFillModeType
 *                      IP_GetChannelFormat
 *
 ***************************************************************************/

UT_Bool IP_GetUnlaceModeType (const char *unlaceModeTypeName, IP_UnlaceModeType *unlaceModeType)
{
    Int32 i;

    if (IP_NumUnlaceModeTypes != sizeof(unlaceModeTypeNames)/sizeof(unlaceModeTypeNames[0])) {
        UT_ErrFatal ("IP_GetUnlaceModeType", "Inconsistent number of unlace modes");
        return UT_False;
    }

    for (i=0; i<sizeof(unlaceModeTypeNames)/sizeof(unlaceModeTypeNames[0]); i++) {
        if (UT_StringEqual (unlaceModeTypeName, unlaceModeTypeNames[i])) {
            *unlaceModeType = (IP_UnlaceModeType) i;
            return UT_True;
        }
    }
    UT_ErrSetNum (UT_ErrParamInvalid, str_InvalidUnlaceModeName, unlaceModeTypeName);
    return UT_False;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetUnlaceModeTypeName
 *
 *      Usage:          Get the name of an unlace mode type.
 *
 *      Synopsis:       const char *IP_GetUnlaceModeTypeName(
 *                                  IP_UnlaceModeType unlaceModeType)
 *
 *      Description:    Possible unlace mode type names:
 *                          "EXTRACT", "REPLICATE", "INTERPOLATE"
 *                          "UNKNOWN" for invalid unlace mode type.
 *
 *      Return value:   A pointer to the name of the unlace mode type.
 *
 *      See also:       IP_GetUnlaceModeType
 *                      IP_GetChannelTypeName
 *                      IP_GetDrawModeTypeName
 *                      IP_GetFillModeTypeName
 *                      IP_GetChannelFormatName
 *
 ***************************************************************************/

const char *IP_GetUnlaceModeTypeName (IP_UnlaceModeType unlaceModeType)
{
    if (unlaceModeType < IP_UnlaceModeExtract || unlaceModeType >= IP_NumUnlaceModeTypes) {
        return "UNKNOWN";
    }
    return unlaceModeTypeNames[unlaceModeType];
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelType
 *
 *      Usage:          Get the numeric value of a channel type name.
 *
 *      Synopsis:       UT_Bool IP_GetChannelType(
 *                              const char *channelTypeName, 
 *                              FF_ImgChanType *channelType)
 *
 *      Description:    The numeric value of channel type name
 *                      "channelTypeName" is returned in "channelType".
 *
 *                      Possible channel type names:
 *                          "NONE", "BW", "RED", "GREEN", "BLUE",
 *                          "MATTE", "REDMATTE", "GREENMATTE", "BLUEMATTE",
 *                          "HORINORMAL", "VERTNORMAL", "DEPTH",
 *                          "TEMPERATURE", "RADIANCE".
 *
 *      Return value:   UT_True, if successful, otherwise UT_False.
 *
 *      See also:       IP_GetChannelTypeName
 *                      IP_GetDrawModeType
 *                      IP_GetFillModeType
 *                      IP_GetChannelFormat
 *                      IP_GetUnlaceModeType
 *
 ***************************************************************************/

UT_Bool IP_GetChannelType (const char *channelTypeName, FF_ImgChanType *channelType)
{
    return FF_ImgGetChannelType (channelTypeName, channelType);
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelTypeName
 *
 *      Usage:          Get the name of a channel type.
 *
 *      Synopsis:       const char *FF_ImgGetChannelTypeName(
 *                                  FF_ImgChanType channelType)
 *
 *      Description:    Possible channel type names:
 *                          "NONE", "BW", "RED", "GREEN", "BLUE",
 *                          "MATTE", "REDMATTE", "GREENMATTE", "BLUEMATTE",
 *                          "HORINORMAL", "VERTNORMAL", "DEPTH",
 *                          "TEMPERATURE", "RADIANCE"
 *                          "UNKNOWN" for invalid channel type.
 *
 *      Return value:   A pointer to the name of the channel type.
 *
 *      See also:       IP_GetChannelType
 *                      IP_GetDrawModeTypeName
 *                      IP_GetFillModeTypeName
 *                      IP_GetChannelFormatName
 *                      IP_GetUnlaceModeTypeName
 *
 ***************************************************************************/

const char *IP_GetChannelTypeName (FF_ImgChanType channelType)
{
    return FF_ImgGetChannelTypeName (channelType);
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelFormat
 *
 *      Usage:          Get the numeric value of a channel format name.
 *
 *      Synopsis:       UT_Bool IP_GetChannelFormat(
 *                              const char *channelFormatName, 
 *                              FF_ImgFmtType *channelFormat)
 *
 *      Description:    The numeric value of channel format name
 *                      "channelFormatName" is returned in "channelFormat".
 *
 *                      Possible format names:
 *                          "NONE", "UBYTE", "FLOAT"
 *
 *      Return value:   UT_True, if successful, otherwise UT_False.
 *
 *      See also:       IP_GetChannelFormatName
 *                      IP_GetDrawModeType
 *                      IP_GetFillModeType
 *                      IP_GetChannelType 
 *                      IP_GetUnlaceModeType
 *
 ***************************************************************************/

UT_Bool IP_GetChannelFormat (const char *channelFormatName, FF_ImgFmtType *channelFormat)
{
    return FF_ImgGetFormatType (channelFormatName, channelFormat);
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetChannelFormatName
 *
 *      Usage:          Get the name of a channel format.
 *
 *      Synopsis:       const char *IP_GetChannelFormatName(
 *                                  FF_ImgFmtType channelFormat)
 *
 *      Description:    Possible channel format names:
 *                          "NONE", "UBYTE", "FLOAT"
 *
 *      Return value:   A pointer to the name of the channel format.
 *
 *      See also:       IP_GetChannelFormat
 *                      IP_GetDrawModeTypeName
 *                      IP_GetFillModeTypeName
 *                      IP_GetChannelTypeName
 *                      IP_GetUnlaceModeTypeName
 *
 ***************************************************************************/

const char *IP_GetChannelFormatName (FF_ImgFmtType channelFormat)
{
    return FF_ImgGetFormatTypeName (channelFormat);
}
