/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Interlace.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions to interlace two video fields into one
 *                      full frame and to divide interlaced frame into
 *                      two fields.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_Interlace
 *                      IP_Unlace
 *
 **************************************************************************/

#include <stdio.h>

#include "UT_Compat.h"
#include "UT_Macros.h"
#include "UT_Error.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

/* Interpolate between scanlines "s1" and "s2" and write the result
   to scanline "dest". "Width" gives the length of the scanlines. */

static void interp_UByte (Int32 width, UInt8 *s1, UInt8 *s2, UInt8 *dest)
{
    UInt8 *pd, *stop;

    pd = dest;
    stop = pd + width;

    while (pd < stop) {
        *pd = (UInt8)(((Int32)*s1 + (Int32)*s2) >> 1);
        s1++;
        s2++;
        pd++;
    }
    return;
}

static void interp_Float (Int32 width, Float32 *s1, Float32 *s2, Float32 *dest)
{
    Float32 *pd, *stop;

    pd = dest;
    stop = pd + width;

    while (pd < stop) {
        *pd = (*s1 + *s2) * 0.5;
        s1++;
        s2++;
        pd++;
    }
    return;
}
/***************************************************************************
 *[@e
 *      Name:           IP_Interlace
 *
 *      Usage:          Interlace two video fields into one frame.
 *
 *      Synopsis:       UT_Bool IP_Interlace(
 *                              IP_ImageId fieldImg1,
 *                              IP_ImageId fieldImg2,
 *                              IP_ImageId frameImg,
 *                              UT_Bool skipLines)
 *
 *      Description:    Images "fieldImg1" and "fieldImg2" are taken as the
 *                      first and second field of an interlaced video image.
 *                      The images are combined into a full video frame.
 *                      The result is stored in image "frameImg".
 *
 *                      If the "skipLines" flag is UT_True, only every second
 *                      scan line of "fieldImg1" and "fieldImg2" is used to
 *                      construct the interlaced frame. The field images must
 *                      have the same size as the frame image.
 *                      If the "skipLines" flag is UT_False, all scan lines
 *                      from "fieldImg1" and "fieldImg2" are used. The field
 *                      images must have half the height of the frame image.
 *
 *                      Note:
 *                      - For interlacing, "fieldImg1", "fieldImg2" and "frameImg"
 *                        are aligned at the upper left corner. The top scan line
 *                        in "frameImg" is always taken from image "fieldImg1".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_Unlace
 *                      IP_Flicker
 *
 ***************************************************************************/

UT_Bool IP_Interlace (IP_ImageId fieldImg1, IP_ImageId fieldImg2, IP_ImageId frameImg,
                      UT_Bool skipLines)
{
    Int32       chn, y, yfld, yfrm, width, height;
    IP_ImageId  fieldImg;

    width = frameImg->pimg.desc.width;
    width = UT_MIN (width, fieldImg1->pimg.desc.width);
    width = UT_MIN (width, fieldImg2->pimg.desc.width);
    if (skipLines) {
        height = frameImg->pimg.desc.height;
        height = UT_MIN (height, fieldImg1->pimg.desc.height);
        height = UT_MIN (height, fieldImg2->pimg.desc.height);
        for (y = 0; y < height; ++y) {
            yfrm = frameImg->pimg.desc.height - 1 - y;
            if (y & 1) {
                fieldImg = fieldImg2;
            } else {
                fieldImg = fieldImg1;
            }
            yfld = fieldImg->pimg.desc.height - 1 - y;
            for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
                if (IP_StateDrawMask[chn] &&
                    frameImg->pimg.channel[chn] &&
                    fieldImg->pimg.channel[chn]) {
                    switch (frameImg->pimg.channel[chn]) {
                        case FF_ImgFmtTypeUByte: {
                            switch (fieldImg->pimg.channel[chn]) {
                                case FF_ImgFmtTypeUByte: {
                                    IP_CopyUByte2UByte
                                        (width,
                                         FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgUByteRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                case FF_ImgFmtTypeFloat: {
                                    IP_CopyFloat2UByte
                                        (width,
                                         FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgUByteRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                default: {
                                    UT_ErrFatal ("IP_Interlace", "invalid channel type");
                                }
                            }
                            break;
                        }
                        case FF_ImgFmtTypeFloat: {
                            switch (fieldImg->pimg.channel[chn]) {
                                case FF_ImgFmtTypeUByte: {
                                    IP_CopyUByte2Float
                                        (width,
                                         FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgFloatRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                case FF_ImgFmtTypeFloat: {
                                    IP_CopyFloat2Float
                                        (width,
                                         FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgFloatRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                default: {
                                    UT_ErrFatal ("IP_Interlace", "invalid channel type");
                                }
                            }
                            break;
                        }
                        default: {
                            UT_ErrFatal ("IP_Interlace", "invalid channel type");
                        }
                    }
                }
            }
        }
    } else {
        height = frameImg->pimg.desc.height;
        height = UT_MIN (height, (fieldImg1->pimg.desc.height << 1));
        height = UT_MIN (height, (fieldImg2->pimg.desc.height << 1));
        for (y = 0; y < height; ++y) {
            yfrm = frameImg->pimg.desc.height - 1 - y;
            if (y & 1) {
                fieldImg = fieldImg2;
            } else {
                fieldImg = fieldImg1;
            }
            yfld = fieldImg->pimg.desc.height - 1 - (y >> 1);
            for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
                if (IP_StateDrawMask[chn] &&
                    frameImg->pimg.channel[chn] &&
                    fieldImg->pimg.channel[chn]) {
                    switch (frameImg->pimg.channel[chn]) {
                        case FF_ImgFmtTypeUByte: {
                            switch (fieldImg->pimg.channel[chn]) {
                                case FF_ImgFmtTypeUByte: {
                                    IP_CopyUByte2UByte
                                        (width,
                                         FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgUByteRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                case FF_ImgFmtTypeFloat: {
                                    IP_CopyFloat2UByte
                                        (width,
                                         FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgUByteRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                default: {
                                    UT_ErrFatal ("IP_Interlace", "invalid channel type");
                                }
                            }
                            break;
                        }
                        case FF_ImgFmtTypeFloat: {
                            switch (fieldImg->pimg.channel[chn]) {
                                case FF_ImgFmtTypeUByte: {
                                    IP_CopyUByte2Float
                                        (width,
                                         FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgFloatRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                case FF_ImgFmtTypeFloat: {
                                    IP_CopyFloat2Float
                                        (width,
                                         FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                         FF_ImgFloatRow (&frameImg->pimg, chn, yfrm));
                                    break;
                                }
                                default: {
                                    UT_ErrFatal ("IP_Interlace", "invalid channel type");
                                }
                            }
                            break;
                        }
                        default: {
                            UT_ErrFatal ("IP_Interlace", "invalid channel type");
                        }
                    }
                }
            }
        }
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_Unlace
 *
 *      Usage:          Extract one video field from an interlaced frame.
 *
 *      Synopsis:       UT_Bool IP_Unlace(
 *                              IP_ImageId frameImg,
 *                              IP_ImageId fieldImg, 
 *                              UT_Bool useOddRows,
 *                              IP_UnlaceModeType unlaceMode)
 *
 *      Description:    Every second scan line from image "frameImg" is copied into
 *                      image "fieldImg". Depending on whether the "useOddRows" flag
 *                      is set to "UT_True" or "UT_False", the odd-numbered or the
 *                      even-numbered scan lines are copied.
 *
 *                      "unlaceMode" specifies the field interpolation mode.
 *                      It can have the following values:
 *                          IP_UnlaceModeExtract: 
 *                              The scan lines of the selected field are extracted
 *                              without any modification. Image "fieldImg" should
 *                              have half the height of image "frameImg".
 *                          IP_UnlaceModeReplicate:
 *                              Each scan line of the selected field is copied
 *                              twice. Image "fieldImg" should have the same height
 *                              as image "frameImg".
 *                          IP_UnlaceModeInterpolate:
 *                              Each scanline of the selected field is copied
 *                              into "fieldImg", which should have the same height
 *                              as image "frameImg". The scanlines of the other
 *                              field are computed by interpolating between
 *                              the upper and lower scanlines of the selected
 *                              field.
 *
 *                      Note:
 *                      - For field extraction, "fieldImg" and "frameImg" are aligned
 *                        at the upper left corner. The top scan line in "frameImg"
 *                        is always considered even-numbered.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    Yes
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_Interlace
 *                      IP_Flicker
 *
 ***************************************************************************/

UT_Bool IP_Unlace (IP_ImageId frameImg, IP_ImageId fieldImg, UT_Bool useOddRows, 
                   IP_UnlaceModeType unlaceMode)
{
    Int32 chn, width, yfrm, yfld, yyfld, yfldinc, destinc;
    UT_Bool interp;

    if (unlaceMode < 0 || unlaceMode >= IP_NumUnlaceModeTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_fipmode);
        return UT_False;
    }

    interp = (unlaceMode != IP_UnlaceModeExtract);

    width = UT_MIN (frameImg->pimg.desc.width, fieldImg->pimg.desc.width);
    if (useOddRows && interp) {
        yyfld = yfld = fieldImg->pimg.desc.height - 2;
    } else {
        yyfld = yfld = fieldImg->pimg.desc.height - 1;
    }
    if (useOddRows) {
        yfrm = frameImg->pimg.desc.height - 2;
    } else {
        yfrm = frameImg->pimg.desc.height - 1;
    }

    yfldinc = (interp)? 2: 1;

    while (yfrm >= 0 && yfld >= 0) {
        for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
            if (IP_StateDrawMask[chn]) {
                switch (frameImg->pimg.channel[chn]) {
                    case FF_ImgFmtTypeNone: {
                        break;
                    }
                    case FF_ImgFmtTypeUByte: {
                        switch (fieldImg->pimg.channel[chn]) {
                            case FF_ImgFmtTypeNone: {
                                break;
                            }
                            case FF_ImgFmtTypeUByte: {
                                IP_CopyUByte2UByte
                                    (width,
                                     FF_ImgUByteRow (&frameImg->pimg, chn, yfrm),
                                     FF_ImgUByteRow (&fieldImg->pimg, chn, yfld));
                                break;
                            }
                            case FF_ImgFmtTypeFloat: {
                                IP_CopyUByte2Float
                                    (width,
                                     FF_ImgUByteRow (&frameImg->pimg, chn, yfrm),
                                     FF_ImgFloatRow (&fieldImg->pimg, chn, yfld));
                                break;
                            }
                            default: {
                                UT_ErrFatal ("IP_Unlace", "invalid channel type");
                            }
                        }
                        break;
                    }
                    case FF_ImgFmtTypeFloat: {
                        switch (fieldImg->pimg.channel[chn]) {
                            case FF_ImgFmtTypeNone: {
                                break;
                            }
                            case FF_ImgFmtTypeUByte: {
                                IP_CopyFloat2UByte
                                    (width,
                                     FF_ImgFloatRow (&frameImg->pimg, chn, yfrm),
                                     FF_ImgUByteRow (&fieldImg->pimg, chn, yfld));
                                break;
                            }
                            case FF_ImgFmtTypeFloat: {
                                IP_CopyFloat2Float
                                    (width,
                                     FF_ImgFloatRow (&frameImg->pimg, chn, yfrm),
                                     FF_ImgFloatRow (&fieldImg->pimg, chn, yfld));
                                break;
                            }
                            default: {
                                UT_ErrFatal ("IP_Unlace", "invalid channel type");
                            }

                        }
                        break;
                    }
                    default: {
                        UT_ErrFatal ("IP_Unlace", "invalid channel type");
                    }
                }
            }
        }
        yfrm -= 2;
        yfld -= yfldinc;
    }

    /* If only a field should be extracted without 
       any interpolation, we're done. */
    if (!interp) {
        return UT_True;
    }

    yfld = yyfld;
    destinc = useOddRows? 1: -1;

    while (yfld >= 0) {
        for (chn = 0; chn < FF_NumImgChanTypes; ++chn) {
            if (!IP_StateDrawMask[chn]) {
                continue;
            }
            switch (fieldImg->pimg.channel[chn]) {
                case FF_ImgFmtTypeNone: {
                    break;
                }
                case FF_ImgFmtTypeUByte: {
                    switch (unlaceMode) {
                        case IP_UnlaceModeInterpolate: {
                            /* If the extracted field is the odd one and the 
                               current scanline is the top most, or the 
                               extracted field is the even one and the current
                               scanline is the lowest, we fall through to 
                               copy mode. */
                            if (!((useOddRows && yfld == yyfld) ||
                                (!useOddRows && (yfld - 2 < 0)))) {
                                interp_UByte (width, 
                                  FF_ImgUByteRow (&fieldImg->pimg, chn, yfld + 2*destinc),
                                  FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                  FF_ImgUByteRow (&fieldImg->pimg, chn, yfld + destinc));
                                break;
                            }
                        }
                        case IP_UnlaceModeReplicate: {
                            IP_CopyUByte2UByte
                                (width,
                                 FF_ImgUByteRow (&fieldImg->pimg, chn, yfld),
                                 FF_ImgUByteRow (&fieldImg->pimg, chn, yfld + destinc));
                            break;
                        }
                        default: {
                            UT_ErrFatal ("IP_Unlace", "unknown unlace mode");
                        }
                    }
                    break;
                }
                case FF_ImgFmtTypeFloat: {
                    switch (unlaceMode) {
                        case IP_UnlaceModeInterpolate: {
                            /* See comment above. */
                            if (!((useOddRows && yfld == yyfld) || 
                                (!useOddRows && (yfld - 2 < 0)))) {
                                interp_Float (width,
                                  FF_ImgFloatRow (&fieldImg->pimg, chn, yfld + 2*destinc),
                                  FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                  FF_ImgFloatRow (&fieldImg->pimg, chn, yfld + destinc));
                                break;
                            }
                        }
                        case IP_UnlaceModeReplicate: {
                            IP_CopyFloat2Float
                                (width,
                                 FF_ImgFloatRow (&fieldImg->pimg, chn, yfld),
                                 FF_ImgFloatRow (&fieldImg->pimg, chn, yfld + destinc));
                            break;
                        }
                        default: {
                            UT_ErrFatal ("IP_Unlace", "unknown unlace mode");
                        }
                    }
                }
                break;
            }
        }
        yfld -= 2;
    }
    return UT_True;
}
