/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Io.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions to create and delete images as well
 *                      as read and write images.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_NewImage
 *                      IP_DeleteImage
 *                      IP_WriteImage
 *                      IP_ReadImage
 *                      IP_WriteSimple
 *                      IP_NewImageFromFile
 *                      IP_AddChannel
 *                      IP_DeleteChannel
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Error.h"
#include "UT_Memory.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

/***************************************************************************
 *[@e
 *      Name:           IP_NewImage
 *
 *      Usage:          Create a new image.
 *
 *      Synopsis:       IP_ImageId IP_NewImage(
 *                                 Int32 width, Int32 height,
 *                                 Float32 aspect,
 *                                 const char *options)
 *
 *      Description:    Memory for an image with "width" by "height" pixels,
 *                      and with an aspect ratio of "aspect" is allocated.
 *                      The current pixel data format defines the format
 *                      for the channels in the new image.
 *                      The values of the pixels in the image are undefined.
 *
 *                      Tcl note:
 *                      If "aspect" is not specified, it is set to width / height.
 *                      The "options" parameter is not exposed at
 *                      the Tcl level, but used internally by the C interface.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   An identifier for the new image if successful,
 *                      else NULL.
 *
 *      See also:       IP_DeleteImage
 *                      IP_NewImageFromFile
 *                      IP_NewImageFromPhoto
 *
 ***************************************************************************/

IP_ImageId IP_NewImage (Int32 width, Int32 height, Float32 aspect, const char *options)
{
    UT_MemState    memstate;
    IP_ImageStruct *img;
    Int32          i;
    UT_Bool        optWithAlpha;

    if (width < 1) {
        UT_ErrSetNum (UT_ErrParamRange, str_inotless, str_image_width, 1);
        return NULL;
    }
    if (height < 1) {
        UT_ErrSetNum (UT_ErrParamRange, str_inotless, str_image_height, 1);
        return NULL;
    }
    if (aspect < 0.0) {
        UT_ErrSetNum (UT_ErrParamRange, str_notless, str_image_aspect, 0.0);
        return NULL;
    }

    memstate = UT_MemRemember ();
    if (!(img = UT_MemTemp (IP_ImageStruct))) {
        UT_MemRestore (memstate);
        return NULL;
    }
    img->pimg.desc.width  = width;
    img->pimg.desc.height = height;
    img->pimg.desc.aspect = aspect;
    img->pimg.desc.red    = IP_StateHueRed;
    img->pimg.desc.green  = IP_StateHueGreen;
    img->pimg.desc.blue   = IP_StateHueBlue;
    img->pimg.desc.white  = IP_StateHueWhite;
    img->pimg.desc.gamma  = IP_StateGamma;
    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        img->pimg.channel[i] = IP_StateFormat[i];
    }

    /* Set default alpha flag and check options of -withalpha key. */
    optWithAlpha = UT_True;
    FF_ImgGetOptWithAlpha (options, &optWithAlpha);
    if (!optWithAlpha) {
        img->pimg.channel[FF_ImgChanTypeMatte] = FF_ImgFmtTypeNone;
    }

    img->pimg.firstscan = 0;
    img->pimg.lastscan  = height - 1;

    if (!FF_ImgNew (&img->pimg)) {
        UT_MemRestore (memstate);
        return NULL;
    }
    UT_MemSave (memstate);
    return img;
}

/***************************************************************************
 *[@e
 *      Name:           IP_DeleteImage
 *
 *      Usage:          Delete an image.
 *
 *      Synopsis:       void IP_DeleteImage(IP_ImageId img)
 *
 *      Description:    Free all memory used by image "img".
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   None.
 *
 *      See also:       IP_NewImage
 *                      IP_NewImageFromFile
 *                      IP_NewImageFromPhoto
 *
 ***************************************************************************/

void IP_DeleteImage (IP_ImageId img)
{
    FF_ImgDelete (&img->pimg);
    UT_MemFree (img);
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_WriteImage
 *
 *      Usage:          Write an image to a file.
 *
 *      Synopsis:       UT_Bool IP_WriteImage(
 *                              IP_ImageId img,
 *                              const char *fileName,
 *                              const char *format,
 *                              const char *options)
 *
 *      Description:    Image "img" is written to pixel file "fileName" using
 *                      pixel file format "format" and file format options "options".
 *                      The current pixel data format selects, which channels will
 *                      be written to the file and in which format.
 *
 *                      The possible option values are format dependent and are
 *                      identical to the values offered by the Img extension.
 *
 *                      The following write options are recognized:
 *                      -verbose <bool>              : POI RAW SGI TGA
 *                      -withalpha <bool>            : POI RAW SGI TGA
 *                      -compression <None|Rle>      : POI     SGI TGA
 *                      -scanorder <TopDown|BottomUp>:     RAW     TGA
 *                      -useheader <bool>            :     RAW
 *                      -byteorder <Intel|Motorola>  :     RAW
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_WriteSimple
 *                      IP_ReadImage
 *                      IP_NewImageFromFile
 *                      IP_NewImageFromPhoto
 *
 ***************************************************************************/

UT_Bool IP_WriteImage (IP_ImageId img,
                       const char *fileName,
                       const char *format,
                       const char *options)
{
    FF_ImgFileDesc pxf;
    Int32          i, istart, iend, istep;

    memset (&pxf, 0, sizeof (FF_ImgFileDesc));

    pxf.desc = img->pimg.desc;
    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        pxf.channel[i] = IP_StateFormat[i];
    }
    strncpy (pxf.format, format, FF_ImgMaxNameLen);
    pxf.format[FF_ImgMaxNameLen - 1] = '\0';
    if (!FF_ImgOpenWrite (fileName, &pxf, options)) {
        return UT_False;
    }
    if (!FF_ImgSetSrcImg (&pxf, &img->pimg, UT_False, UT_False)) {
        FF_ImgClose (&pxf);
        UT_ErrAppendStr (". ", str_cant_write, fileName);
        return UT_False;
    }
    if (pxf.scanorder == FF_ImgScanOrderTypeBottomUp) {
        istart = 0;
        iend = pxf.desc.height - 1;
        istep = 1;
    } else {
        istart = pxf.desc.height - 1;
        iend = 0;
        istep = -1;
    }
    for (i = istart; i != iend + istep; i += istep) {
        if (!FF_ImgWriteRow (&pxf, i, 0)) {
            FF_ImgClose (&pxf);
            UT_ErrAppendStr (". ", str_cant_write, fileName);
            return UT_False;
        }
    }
    FF_ImgClose (&pxf);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_ReadImage
 *
 *      Usage:          Read an image from a file.
 *
 *      Synopsis:       UT_Bool IP_ReadImage
 *                              (IP_ImageId img, 
 *                               const char *fileName,
 *                               const char *options)
 *
 *      Description:    The pixel data stored in image file "fileName" is
 *                      read into image "img" using options "options".
 *
 *                      If option "-scale" is true, the pixels in the file are
 *                      scaled, so that the width and height of the image in
 *                      the file becomes equal to width and height of "img".
 *
 *                      If "-scale" is false, the image in the file is copied
 *                      into "img". If "img" is larger than the image in the
 *                      file, the regions not covered by the image in the
 *                      file are filled with zeros.
 *
 *                      Default value of option "-scale" is false.
 *
 *                      Note that the image scaling algorithm has been
 *                      designed to transform the pixel data quickly on the
 *                      fly, while the data is read from the image file.
 *                      The quality of the scaled picture is not optimal: 
 *                      Aliasing may occur at high-contrast edges, especially
 *                      when the image is shrunk or blown up by some large
 *                      amount.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_NewImageFromFile
 *                      IP_NewImageFromPhoto
 *                      IP_WriteImage
 *
 ***************************************************************************/

UT_Bool IP_ReadImage (IP_ImageId img, const char *fileName, const char *options)
{
    FF_ImgFileDesc pxf;
    Int32          i, istart, iend, istep;
    UT_Bool        optScale, optGamma, optColor;

    memset (&pxf, 0, sizeof (FF_ImgFileDesc));

    /* Set default scale flag and check options of -scale key. */
    optScale = UT_False;
    FF_ImgGetOptBool (options, "-scale", &optScale);

    /* Set default gamma flag and check options of -gamma key. */
    optGamma = UT_False;
    FF_ImgGetOptBool (options, "-gamma", &optGamma);

    /* Set default color flag and check options of -color key. */
    optColor = UT_False;
    FF_ImgGetOptBool (options, "-color", &optColor);

    if (!FF_ImgOpenRead (fileName, &pxf, options)) {
        return UT_False;
    }

    if (!FF_ImgSetDestImg (&pxf, &img->pimg, optScale, optGamma, optColor)) {
        FF_ImgClose (&pxf);
        UT_ErrAppendStr (". ", str_cant_read, fileName);
        return UT_False;
    }
    if (pxf.scanorder == FF_ImgScanOrderTypeBottomUp) {
        istart = 0;
        if (optScale) {
            iend = img->pimg.desc.height - 1;
        } else {
            iend = pxf.desc.height - 1;
        }
        istep = 1;
    } else {
        iend = 0;
        if (optScale) {
            istart = img->pimg.desc.height - 1;
        } else {
            istart = pxf.desc.height - 1;
        }
        istep = -1;
    }
    for (i = istart; i != iend + istep; i += istep) {
        if (!FF_ImgReadRow (&pxf, i, 0)) {
            FF_ImgClose (&pxf);
            UT_ErrAppendStr (". ", str_cant_read, fileName);
            return UT_False;
        }
    }
    FF_ImgClose (&pxf);
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_WriteSimple
 *
 *      Usage:          Write an image to a file - simple version.
 *
 *      Synopsis:       UT_Bool IP_WriteSimple(
 *                              IP_ImageId img,
 *                              const char *fileName,
 *                              const char *format,
 *                              const char *options)
 *
 *      Description:    Image "img" is written to pixel file "fileName" using
 *                      file format "format" and format options "options".
 *                      The pixel data format in the file will be the same as
 *                      in "img" for all channels.
 *
 *                      The possible option values are format dependent and are
 *                      identical to the values offered by the Img extension.
 *                      See IP_WriteImage for write option details.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_NewImageFromFile
 *                      IP_ReadImage
 *                      IP_WriteImage
 *
 ***************************************************************************/

UT_Bool IP_WriteSimple
        (IP_ImageId img,
         const char *fileName,
         const char *format,
         const char *options)
{
    FF_ImgFmtType imgFmt[FF_NumImgChanTypes];

    if ( IP_PushState() < 0) {
        return UT_False;
    }
    IP_GetImageFormat (img, imgFmt);
    IP_SetFormat (imgFmt);
    if (!IP_WriteImage (img, fileName, format, options)) {
        IP_PopState();
        return UT_False;
    }
    IP_PopState();
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_NewImageFromFile
 *
 *      Usage:          Read an image from a file.
 *
 *      Synopsis:       IP_ImageId IP_NewImageFromFile(
 *                                 const char *fileName,
 *                                 const char *options)
 *
 *      Description:    A new image with the same width, height, aspect
 *                      ratio and pixel data formats as the image stored in
 *                      file "fileName" is created.
 *                      The pixel data in the file is copied into the new
 *                      image without scaling and without gamma or color
 *                      correction.
 *
 *                      The possible option values are format dependent and are
 *                      identical to the values offered by the Img extension.
 *
 *                      The following read options are recognized by all formats:
 *                      -verbose   <bool>
 *                      -withalpha <bool>
 *
 *                      The following read options are recognized by the RAW format:
 *                      -scanorder <TopDown|BottomUp>
 *                      -useheader <bool>
 *                      -skipbytes <bool>
 *                      -byteorder <Intel|Motorola>
 *                      -width     <integer>
 *                      -height    <integer>
 *                      -numchan   <integer>
 *                      -pixeltype <double|float|int|short|byte>
 *
 *                      Note, that the mapping options of the Img extension
 *                      "-map", "-min", "-max", "-gamma", "-saturation", "-cutoff"
 *                      are not supported, as the poImg extension supports
 *                      floating point images.
 *                      To display floating point images using photo images,
 *                      see the IP_AsPhoto or poImgUtil::AsPhoto command.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   An identifier for the image read from the file
 *                      if successful, else NULL.
 *
 *      See also:       IP_NewImage
 *                      IP_NewImageFromPhoto
 *                      IP_ReadImage
 *                      IP_WriteImage
 *                      IP_AsPhoto
 *                      poImgUtil::AsPhoto
 *
 ***************************************************************************/

IP_ImageId IP_NewImageFromFile (const char *fileName, const char *options)
{
    IP_ImageId     img = NULL;
    FF_ImgFileDesc pxf;
    Int32          i, istart, iend, istep;

    memset (&pxf, 0, sizeof (FF_ImgFileDesc));

    if (!FF_ImgOpenRead (fileName, &pxf, options)) {
        return NULL;
    }

    if ( IP_PushState() < 0) {
        return NULL;
    }
    if (!IP_SetFormat (pxf.channel)) {
        IP_PopState();
        return NULL;
    }
    if (!IP_SetColorCorrect (pxf.desc.gamma,
                             &pxf.desc.red, &pxf.desc.green,
                             &pxf.desc.blue, &pxf.desc.white)) {
        IP_PopState();
        return NULL;
    }

    img = IP_NewImage (pxf.desc.width, pxf.desc.height, pxf.desc.aspect, options);
    if (!img) {
        IP_PopState();
        return NULL;
    }
    IP_PopState();

    if (!FF_ImgSetDestImg (&pxf, &img->pimg, UT_False, UT_False, UT_False)) {
        FF_ImgClose (&pxf);
        UT_ErrAppendStr (". ", str_cant_read, fileName);
        return NULL;
    }
    if (pxf.scanorder == FF_ImgScanOrderTypeBottomUp) {
        istart = 0;
        iend = pxf.desc.height - 1;
        istep = 1;
    } else {
        iend = 0;
        istart = pxf.desc.height - 1;
        istep = -1;
    }
    for (i = istart; i != iend + istep; i += istep) {
        if (!FF_ImgReadRow (&pxf, i, 0)) {
            FF_ImgClose (&pxf);
            UT_ErrAppendStr (". ", str_cant_read, fileName);
            return NULL;
        }
    }
    FF_ImgClose (&pxf);
    return img;
}

/***************************************************************************
 *[@e
 *      Name:           IP_AddChannel
 *
 *      Usage:          Add a channel to an image.
 *
 *      Synopsis:       UT_Bool IP_AddChannel(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel,
 *                              FF_ImgFmtType formatType)
 *
 *      Description:    Add a new channel "channel" to image "img". 
 *                      The new channel is using pixel format "formatType".
 *                      The contents of the new channel are not initialized.
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_DeleteChannel
 *
 ***************************************************************************/

UT_Bool IP_AddChannel (IP_ImageId img, FF_ImgChanType channel, FF_ImgFmtType formatType)
{
    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (img->pimg.channel[channel] != FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (formatType < FF_ImgFmtTypeNone || formatType >= FF_NumImgFmtTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgfmt, formatType,
                      IP_GetChannelFormatName (formatType));
        return UT_False;
    }
    return FF_ImgAddChannel (&img->pimg, channel, formatType);
}

/***************************************************************************
 *[@e
 *      Name:           IP_DeleteChannel
 *
 *      Usage:          Delete a channel from an image.
 *
 *      Synopsis:       UT_Bool IP_DeleteChannel(
 *                              IP_ImageId img,
 *                              FF_ImgChanType channel)
 *
 *      Description:    Free the memory used by channel "channel"
 *                      of image "img". The pixel format of the
 *                      deleted channel is set to FF_ImgFmtTypeNone. 
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    No
 *                      UByte format: All
 *                      Float format: All
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_AddChannel
 *
 ***************************************************************************/

UT_Bool IP_DeleteChannel (IP_ImageId img, FF_ImgChanType channel)
{
    if (channel < 0 || channel >= FF_NumImgChanTypes) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    if (img->pimg.channel[channel] == FF_ImgFmtTypeNone) {
        UT_ErrSetNum (UT_ErrParamInvalid, str_imgchan, channel,
                      IP_GetChannelTypeName (channel));
        return UT_False;
    }
    FF_ImgDeleteChannel (&img->pimg, channel);
    return UT_True;
}
