/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_Stamp.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Function to stamp a logo onto an image.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_Stamp
 *
 **************************************************************************/

#include <stdio.h>

#include "UT_Compat.h"
#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Parallel.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

#if 0
    #define DEBUG_LOCAL
#endif

/***************************************************************************
 *[@e
 *      Name:           IP_Stamp
 *
 *      Usage:          Stamp an image on top of another image.
 *
 *      Synopsis:       UT_Bool IP_Stamp(
 *                              IP_ImageId img, IP_ImageId stampImg,
 *                              Int32 x, Int32 y,
 *                              Float32 sr, Float32 sg, Float32 sb)
 *
 *      Description:    Image "stampImg" is copied onto image "img" 
 *                      considering the matte channel of "stampImg". 
 *                      If "stampImg" does not have a matte channel, 
 *                      "stampImg" completely replaces the content of "img".
 *                      The lower left corner of "stampImg" is placed at
 *                      pixel position (x, y) in image "img". The red,
 *                      green and blue channels of image "stampImg" are
 *                      scaled according to the values specified in
 *                      "sr", "sg" and "sb". The scale values must be in
 *                      the range [0.0, 1.0].
 *
 *      States:         State settings influencing functionality:
 *                      Draw mask:    No
 *                      Draw mode:    No
 *                      Draw color:   No
 *                      Threading:    Yes
 *                      UByte format: RGB(A)
 *                      Float format: RGB(A)
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_CompositeMatte
 *                      IP_ChangeChannelGamma
 *                      IP_SetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_Stamp (IP_ImageId img, IP_ImageId stampImg, Int32 x, Int32 y,
                  Float32 sr, Float32 sg, Float32 sb)
{
    IP_ImageId stamp2Img, cmpImg;
    FF_ImgHdr imgDesc, stampDesc;
    FF_ImgFmtType stampfmt[FF_NumImgChanTypes] = { FF_ImgFmtTypeNone };
    FF_ImgFmtType pixfmt[FF_NumImgChanTypes]   = { FF_ImgFmtTypeNone };
    UT_Bool drawmask[FF_NumImgChanTypes]       = { UT_False };
    Float32 drawcolor[FF_NumImgChanTypes]      = { 1.0 };
    Int32 lw, lh, iw, ih;
    Int32 chan;

    if (IP_PushState() < 0) {
        return UT_False;
    }

    IP_GetImageInfo   (img,  &imgDesc);
    IP_GetImageInfo   (stampImg, &stampDesc);
    IP_GetImageFormat (stampImg, stampfmt);
    iw = imgDesc.width;
    ih = imgDesc.height;
    lw = stampDesc.width;
    lh = stampDesc.height;

    (void) iw; /* iw only used for debug output. Avoid warning. */
    #if defined (DEBUG_LOCAL)
        printf ("Image size: (%d %d) Stamp size: (%d %d)\n", iw, ih, lw, lh);
    #endif

    for (chan = 0; chan < FF_NumImgChanTypes; ++chan) {
        pixfmt[chan] = stampfmt[chan];
        if (stampfmt[FF_ImgChanTypeMatte] == FF_ImgFmtTypeNone) {
            pixfmt[FF_ImgChanTypeMatte] = FF_ImgFmtTypeUByte;
        }

        #if defined (DEBUG_LOCAL)
            printf ("Channel %d: StampFmt: %d PixFmt: %d\n",
                    chan, stampfmt[chan], pixfmt[chan]);
        #endif
    }
    IP_SetFormat (pixfmt);

    if (!(stamp2Img = IP_NewImage (lw, lh, stampDesc.aspect, NULL)) ||
        !(cmpImg    = IP_NewImage (lw, lh, stampDesc.aspect, NULL))) {
        IP_PopState();
        return UT_False;
    }

    if (!IP_CopyImage (stampImg, stamp2Img)) {
        IP_DeleteImage (cmpImg);
        IP_DeleteImage (stamp2Img);
        IP_PopState();
        return UT_False;
    }

    /* If stampImg does not supply a matte channel, we generate a white matte. */
    if (stampfmt[FF_ImgChanTypeMatte] == FF_ImgFmtTypeNone) {
        #if defined (DEBUG_LOCAL)
            printf ("Generating white matte\n");
        #endif
        drawmask[FF_ImgChanTypeMatte] = UT_True;
        IP_SetDrawMask (drawmask);
        drawcolor[FF_ImgChanTypeMatte] = 1.0;
        IP_SetDrawColor (drawcolor);
        IP_DrawRect (stamp2Img, 0, 0, lw, lh, UT_True);
    }

    drawmask[FF_ImgChanTypeRed]   = UT_True;
    drawmask[FF_ImgChanTypeGreen] = UT_True;
    drawmask[FF_ImgChanTypeBlue]  = UT_True;
    drawmask[FF_ImgChanTypeMatte] = UT_True;
    IP_SetDrawMask (drawmask);
    
    if (!IP_ChangeChannelGamma (stamp2Img, FF_ImgChanTypeRed,   1.0, sr, 0.0) ||
        !IP_ChangeChannelGamma (stamp2Img, FF_ImgChanTypeGreen, 1.0, sg, 0.0) ||
        !IP_ChangeChannelGamma (stamp2Img, FF_ImgChanTypeBlue,  1.0, sb, 0.0)) {
        IP_DeleteImage (cmpImg);
        IP_DeleteImage (stamp2Img);
        IP_PopState();
        return UT_False;
    }

    IP_Blank (cmpImg);
    IP_CopyRect (img, cmpImg, x, ih - (y + lh), x + lw, ih - y, 0, 0);

    if (!IP_CompositeMatte (stamp2Img, cmpImg, cmpImg, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                            UT_True, stamp2Img)) {
        IP_DeleteImage (cmpImg);
        IP_DeleteImage (stamp2Img);
        IP_PopState();
        return UT_False;
    }

    drawmask[FF_ImgChanTypeMatte] = UT_False;
    IP_SetDrawMask (drawmask);

    IP_CopyRect (cmpImg, img, 0, 0, lw - 1, lh - 1, x, ih - (y + lh));

    IP_DeleteImage (cmpImg);
    IP_DeleteImage (stamp2Img);

    IP_PopState ();

    return UT_True;
}
