/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         ImageProcessing
 *      Filename:       IP_State.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions to change the state of the image processing
 *                      module.
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *                      IP_SetFormat
 *                      IP_GetFormat
 *                      IP_SetDrawMask
 *                      IP_GetDrawMask
 *                      IP_SetDrawColor
 *                      IP_GetDrawColor
 *                      IP_SetDrawMode
 *                      IP_GetDrawMode
 *                      IP_SetColorCorrect
 *                      IP_GetColorCorrect
 *                      IP_PushState
 *                      IP_PopState
 *                      IP_SetNumThreads
 *                      IP_GetNumThreads
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>

#include "UT_Compat.h"
#include "UT_Const.h"
#include "UT_Macros.h"
#include "UT_Error.h"

#include "FF_Image.h"

#include "IP_Image.h"
#include "IP_ImagePrivate.h"

/* Current pixel data format */
FF_ImgFmtType IP_StateFormat[FF_NumImgChanTypes] = {
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeBrightness */
   FF_ImgFmtTypeUByte,  /* FF_ImgChanTypeRed */
   FF_ImgFmtTypeUByte,  /* FF_ImgChanTypeGreen */
   FF_ImgFmtTypeUByte,  /* FF_ImgChanTypeBlue */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeMatte */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeRedMatte */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeGreenMatte */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeBlueMatte */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeHoriSnr */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeVertSnr */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeDepth */
   FF_ImgFmtTypeNone,   /* FF_ImgChanTypeTemperature */
   FF_ImgFmtTypeNone    /* FF_ImgChanTypeRadiance */
};

/* Current draw mask */
UT_Bool IP_StateDrawMask[FF_NumImgChanTypes] = {
    UT_True,    /* FF_ImgChanTypeBrightness */
    UT_True,    /* FF_ImgChanTypeRed */
    UT_True,    /* FF_ImgChanTypeGreen */
    UT_True,    /* FF_ImgChanTypeBlue */
    UT_True,    /* FF_ImgChanTypeMatte */
    UT_True,    /* FF_ImgChanTypeRedMatte */
    UT_True,    /* FF_ImgChanTypeGreenMatte */
    UT_True,    /* FF_ImgChanTypeBlueMatte */
    UT_True,    /* FF_ImgChanTypeHoriSnr */
    UT_True,    /* FF_ImgChanTypeVertSnr */
    UT_True,    /* FF_ImgChanTypeDepth */
    UT_True,    /* FF_ImgChanTypeTemperature */
    UT_True     /* FF_ImgChanTypeRadiance */
};

/* Current draw color, UByte version */
UInt8 IP_StateDrawColorUByte[FF_NumImgChanTypes] = {
    0,          /* FF_ImgChanTypeBrightness */
    0,          /* FF_ImgChanTypeRed */
    0,          /* FF_ImgChanTypeGreen */
    0,          /* FF_ImgChanTypeBlue */
    0,          /* FF_ImgChanTypeMatte */
    0,          /* FF_ImgChanTypeRedMatte */
    0,          /* FF_ImgChanTypeGreenMatte */
    0,          /* FF_ImgChanTypeBlueMatte */
    0,          /* FF_ImgChanTypeHoriSnr */
    0,          /* FF_ImgChanTypeVertSnr */
    0,          /* FF_ImgChanTypeDepth */
    0,          /* FF_ImgChanTypeTemperature */
    0           /* FF_ImgChanTypeRadiance */
};

/* Current draw color, Float32 version */
Float32 IP_StateDrawColorFloat[FF_NumImgChanTypes] = {
    0.0,        /* FF_ImgChanTypeBrightness */
    0.0,        /* FF_ImgChanTypeRed */
    0.0,        /* FF_ImgChanTypeGreen */
    0.0,        /* FF_ImgChanTypeBlue */
    0.0,        /* FF_ImgChanTypeMatte */
    0.0,        /* FF_ImgChanTypeRedMatte */
    0.0,        /* FF_ImgChanTypeGreenMatte */
    0.0,        /* FF_ImgChanTypeBlueMatte */
    0.0,        /* FF_ImgChanTypeHoriSnr */
    0.0,        /* FF_ImgChanTypeVertSnr */
    0.0,        /* FF_ImgChanTypeDepth */
    0.0,        /* FF_ImgChanTypeTemperature */
    0.0         /* FF_ImgChanTypeRadiance */
};

/* Current draw mode */
IP_DrawModeType IP_StateDrawMode[FF_NumImgChanTypes] = {
    IP_DrawModeReplace,         /* FF_ImgChanTypeBrightness */
    IP_DrawModeReplace,         /* FF_ImgChanTypeRed */
    IP_DrawModeReplace,         /* FF_ImgChanTypeGreen */
    IP_DrawModeReplace,         /* FF_ImgChanTypeBlue */
    IP_DrawModeReplace,         /* FF_ImgChanTypeMatte */
    IP_DrawModeReplace,         /* FF_ImgChanTypeRedMatte */
    IP_DrawModeReplace,         /* FF_ImgChanTypeGreenMatte */
    IP_DrawModeReplace,         /* FF_ImgChanTypeBlueMatte */
    IP_DrawModeReplace,         /* FF_ImgChanTypeHoriSnr */
    IP_DrawModeReplace,         /* FF_ImgChanTypeVertSnr */
    IP_DrawModeReplace,         /* FF_ImgChanTypeDepth */
    IP_DrawModeReplace,         /* FF_ImgChanTypeTemperature */
    IP_DrawModeReplace          /* FF_ImgChanTypeRadiance */
};

/* Current gamma factor */
Float32 IP_StateGamma = 1.0;

/* Current monitor primary red, green and blue, and monitor white point */
CIE_hue IP_StateHueRed =        {0.670f, 0.330f};
CIE_hue IP_StateHueGreen =      {0.210f, 0.710f};
CIE_hue IP_StateHueBlue =       {0.140f, 0.080f};
CIE_hue IP_StateHueWhite =      {0.313f, 0.329f};

/* Stack for PushState and PopState functions. */
#define MAX_STACK 10
static FF_ImgFmtType   sStackFormat[MAX_STACK][FF_NumImgChanTypes];
static UT_Bool         sStackDrawMask[MAX_STACK][FF_NumImgChanTypes];
static Float32         sStackDrawColor[MAX_STACK][FF_NumImgChanTypes];
static IP_DrawModeType sStackDrawMode[MAX_STACK][FF_NumImgChanTypes];
static Int32           sStackIndex = -1;

/* Current number of threads used for image processing. Default is serial processing. */
Int32 IP_StateNumThreads = 0;

/* Current scale factor of vector text */
Int32 IP_TextScale = 1;

/***************************************************************************
 *[@e
 *      Name:           IP_SetFormat
 *
 *      Usage:          Set the current pixel data format.
 *
 *      Synopsis:       UT_Bool IP_SetFormat( 
 *                              const FF_ImgFmtType formatList[FF_NumImgChanTypes])
 *
 *      Description:    Define the current pixel data format for all channels,
 *                      i.e. the format to be used for the different channels
 *                      when a new image is created.
 *                      When memory is allocated for channel number "i" of an
 *                      image, pixel data format formatList[i] is used.
 *
 *                      Enumeration FF_ImgFmtType has the following members:
 *                      FF_ImgFmtTypeNone : 0  None. channel does not exist
 *                      FF_ImgFmtTypeUByte: 1  Unsigned 8-bit integers
 *                      FF_ImgFmtTypeFloat: 2  32-bit floating point numbers
 *
 *                      By default, the current pixel data format is
 *                      FF_ImgFmtTypeNone for all channels except for
 *                      FF_ImgChanTypeRed, FF_ImgChanTypeGreen and 
 *                      FF_ImgChanTypeBlue, for which
 *                      the default is FF_ImgFmtTypeUByte.
 *
 *                      Tcl note: If "formatList" is not specified, the
 *                      current pixel data format is reset to the default values.
 *                      If "formatList" has less elements than NUMCHAN
 *                      (FF_NumImgChanTypes), the missing elements are filled
 *                      with zeros.
 *                      The format enumeration values can be accessed in
 *                      Tcl using the following global variables:
 *                      NONE UBYTE FLOAT
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetFormat
 *                      IP_SetDrawMask
 *                      IP_SetDrawColor
 *                      IP_SetDrawMode
 *                      IP_SetColorCorrect
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

UT_Bool IP_SetFormat (const FF_ImgFmtType formatList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        if (formatList[i] < 0 || formatList[i] >= FF_NumImgFmtTypes) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_pixformat);
            return UT_False;
        }
    }
    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        IP_StateFormat[i] = formatList[i];
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetFormat
 *
 *      Usage:          Get the current pixel data format.
 *
 *      Synopsis:       void IP_GetFormat(
 *                           FF_ImgFmtType formatList[FF_NumImgChanTypes])
 *
 *      Description:    Get the current pixel data format in "formatList".
 *
 *      Return value:   None.
 *
 *      See also:       IP_SetFormat
 *                      IP_GetDrawMask
 *                      IP_GetDrawColor
 *                      IP_GetDrawMode
 *                      IP_GetColorCorrect
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

void IP_GetFormat (FF_ImgFmtType formatList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        formatList[i] = IP_StateFormat[i];
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_SetDrawMask
 *
 *      Usage:          Set the current draw mask.
 *
 *      Synopsis:       UT_Bool IP_SetDrawMask(
 *                              const UT_Bool drawMaskList[FF_NumImgChanTypes])
 *
 *      Description:    Define the current draw mask, i.e. the mask selecting
 *                      which channels will be affected by functions drawing
 *                      into an image:
 *                      Drawing into channel number "i" is allowed only if
 *                      drawMaskList[i] is UT_True.
 *
 *                      By default, the current draw mask is UT_True for
 *                      all channels.
 *
 *                      Tcl note: If "drawMaskList" is not specified, the
 *                      current draw mask is reset to the default values.
 *                      If "drawMaskList" has less elements than NUMCHAN
 *                      (FF_NumImgChanTypes), the missing elements are filled
 *                      with zeros.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetDrawMask
 *                      IP_SetDrawColor
 *                      IP_SetDrawMode
 *                      IP_SetColorCorrect
 *                      IP_SetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

UT_Bool IP_SetDrawMask (const UT_Bool drawMaskList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        IP_StateDrawMask[i] = !!drawMaskList[i];
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetDrawMask
 *
 *      Usage:          Get the current draw mask.
 *
 *      Synopsis:       void IP_GetDrawMask(
 *                           UT_Bool drawMaskList[FF_NumImgChanTypes])
 *
 *      Description:    Get the current draw mask in "drawMaskList".
 *
 *      Return value:   None.
 *
 *      See also:       IP_SetDrawMask
 *                      IP_GetDrawColor
 *                      IP_GetDrawMode
 *                      IP_GetColorCorrect
 *                      IP_GetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

void IP_GetDrawMask (UT_Bool drawMaskList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        drawMaskList[i] = IP_StateDrawMask[i];
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_SetDrawColor
 *
 *      Usage:          Set the current draw color.
 *
 *      Synopsis:       UT_Bool IP_SetDrawColor(
 *                              const Float32 drawColorList[FF_NumImgChanTypes])
 *
 *      Description:    Define the current draw color for all channels,
 *                      i.e. the color used when drawing into an image
 *                      takes place: The current draw color for
 *                      channel number "i" is set to "drawColorList[i]".
 *
 *                      Drawing colors should be in the range from 0.0 to 1.0
 *                      for all channels except FF_ImgChanTypeHoriSnr,
 *                      FF_ImgChanTypeVertSnr, FF_ImgChanTypeDepth,
 *                      FF_ImgChanTypeTemperature and FF_ImgChanTypeRadiance.
 *                      For FF_ImgChanTypeHoriSnr and FF_ImgChanTypeVertSnr,
 *                      draw colors must be in the range from -1.0 to +1.0.
 *                      For FF_ImgChanTypeDepth, draw colors must be 
 *                      greater than 0.0, as depth information is stored as the
 *                      reciprocal value.
 *
 *                      By default, the current draw color is 0.0 for all
 *                      channels.
 *
 *                      Tcl note: If "drawColorList" is not specified, the
 *                      current draw color is reset to the default values.
 *                      If "drawColorList" has less elements than NUMCHAN
 *                      (FF_NumImgChanTypes), the missing elements are filled
 *                      with zeros.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetDrawColor
 *                      IP_SetDrawMask
 *                      IP_SetDrawMode
 *                      IP_SetColorCorrect
 *                      IP_SetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

UT_Bool IP_SetDrawColor (const Float32 drawColorList[FF_NumImgChanTypes])
{
    Int32 i, iclr;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        iclr = (Int32)(drawColorList[i] * 255.0 + 0.5);
        IP_StateDrawColorUByte[i] = UT_MAX (0, UT_MIN (iclr, 255));
        IP_StateDrawColorFloat[i] = drawColorList[i];
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetDrawColor
 *
 *      Usage:          Get the current draw color.
 *
 *      Synopsis:       void IP_GetDrawColor(
 *                           Float32 drawColorList[FF_NumImgChanTypes])
 *
 *      Description:    Get the current draw color in "drawColorList".
 *
 *      Return value:   None.
 *
 *      See also:       IP_SetDrawColor
 *                      IP_GetDrawMask
 *                      IP_GetDrawMode
 *                      IP_GetColorCorrect
 *                      IP_GetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

void IP_GetDrawColor (Float32 drawColorList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        drawColorList[i] = IP_StateDrawColorFloat[i];
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_SetDrawMode
 *
 *      Usage:          Set the current draw mode.
 *
 *      Synopsis:       UT_Bool IP_SetDrawMode(
 *                              const IP_DrawModeType drawModeList[FF_NumImgChanTypes])
 *
 *      Description:    Define the current draw mode for all channels,
 *                      i.e. how the current draw color is combined with
 *                      the orignal color of pixels when drawing into an
 *                      image takes place: "drawModeList[i]" is the 
 *                      current draw mode for channel number "i".
 *
 *                      Enumeration IP_DrawModeType has the following members:
 *                      IP_DrawModeReplace: 0  Replace old color with current color
 *                      IP_DrawModeAdd    : 1  Add old color and current color
 *                      IP_DrawModeSub    : 2  Subtract old color from current color
 *                      IP_DrawModeXor    : 3  Exclusive-or old color and current color

 *                      By default, the current draw mode is
 *                      IP_DrawModeReplace for all channels.
 *
 *                      Tcl note: If "drawModeList" is not specified, the
 *                      current draw mode is reset to the default values.
 *                      If "drawModeList" has less elements than NUMCHAN
 *                      (FF_NumImgChanTypes), the missing elements are filled
 *                      with zeros.
 *                      The draw mode enumeration values can be accessed in
 *                      Tcl using the following global variables:
 *                      REPLACE ADD SUB XOR
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetDrawMode
 *                      IP_SetDrawMask
 *                      IP_SetDrawColor
 *                      IP_SetColorCorrect
 *                      IP_SetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

UT_Bool IP_SetDrawMode (const IP_DrawModeType drawModeList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        if (drawModeList[i] < 0 || drawModeList[i] >= IP_NumDrawModeTypes) {
            UT_ErrSetNum (UT_ErrParamInvalid, str_drawmode);
            return UT_False;
        }
    }
    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        IP_StateDrawMode[i] = drawModeList[i];
    }
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetDrawMode
 *
 *      Usage:          Get the current draw mode.
 *
 *      Synopsis:       void IP_GetDrawMode(
 *                           IP_DrawModeType drawModeList[FF_NumImgChanTypes])
 *
 *      Description:    Get the current draw mode in "drawModeList".
 *
 *      Return value:   None.
 *
 *      See also:       IP_SetDrawMode
 *                      IP_GetDrawMask
 *                      IP_GetDrawColor
 *                      IP_GetColorCorrect
 *                      IP_GetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

void IP_GetDrawMode (IP_DrawModeType drawModeList[FF_NumImgChanTypes])
{
    Int32 i;

    for (i = 0; i < FF_NumImgChanTypes; ++i) {
        drawModeList[i] = IP_StateDrawMode[i];
    }
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_SetColorCorrect
 *
 *      Usage:          Set the current color correction parameters.
 *
 *      Synopsis:       UT_Bool IP_SetColorCorrect(
 *                              Float32 gamma,
 *                              const CIE_hue *red,
 *                              const CIE_hue *green,
 *                              const CIE_hue *blue,
 *                              const CIE_hue *white)
 *
 *      Description:    Define the current gamma and color correction 
 *                      parameters, i.e. parameters written into
 *                      the header of any newly created images.
 *
 *                      The parameters and their default values are:
 *
 *                      gamma   1.0             Gamma factor of the frame
 *                                              buffer / monitor combination
 *                                              which will be used to display
 *                                              images
 *
 *                      red     (0.670, 0.330)  The location of the monitor's
 *                                              pure red in the CIE x-y color
 *                                              diagram; default is NTSC
 *                                              standard primary red.
 *
 *                      green   (0.210, 0.710)  The location of the monitor's
 *                                              pure green; default is NTSC
 *                                              standard primary green.
 *
 *                      blue    (0.140, 0.080)  The location of the monitor's
 *                                              pure blue; default is NTSC
 *                                              standard primary blue.
 *
 *                      white   (0.313, 0.329)  The location of the monitor's
 *                                              white point; default is CIE
 *                                              illuminant D6500, i.e. the
 *                                              color of a black body at a
 *                                              temperature of 6500 Kelvin.
 *
 *                      Tcl note: If no parameter is specified, the current
 *                      color correction values are reset to the default values.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetColorCorrect
 *                      IP_SetDrawMask
 *                      IP_SetDrawMode
 *                      IP_SetDrawColor
 *                      IP_SetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

UT_Bool IP_SetColorCorrect
        (Float32 gamma,
         const CIE_hue *red,
         const CIE_hue *green,
         const CIE_hue *blue,
         const CIE_hue *white)
{
    if (gamma < 0.0) {
         UT_ErrSetNum (UT_ErrParamRange, str_notless, str_gamma, 0.0);
        return UT_False;
    }
    if ((red->x < 0.0) || (red->x > 1.0) ||
        (red->y < 0.0) || (red->y > 1.0) ||
        (green->x < 0.0) || (green->x > 1.0) ||
        (green->y < 0.0) || (green->y > 1.0) ||
        (blue->x < 0.0) || (blue->x > 1.0) ||
        (blue->y < 0.0) || (blue->y > 1.0) ||
        (white->x < 0.0) || (white->x > 1.0) ||
        (white->y < 0.0) || (white->y > 1.0)) {
        UT_ErrSetNum (UT_ErrParamRange, str_range_inclusive,
                     str_cie_hue, 0.0, 1.0);
        return UT_False;
    }
    IP_StateGamma = gamma;
    IP_StateHueRed = *red;
    IP_StateHueGreen = *green;
    IP_StateHueBlue = *blue;
    IP_StateHueWhite = *white;
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetColorCorrect
 *
 *      Usage:          Get the current color correction parameters.
 *
 *      Synopsis:       void IP_GetColorCorrect(
 *                           Float32 *gamma,
 *                           CIE_hue *red,
 *                           CIE_hue *green,
 *                           CIE_hue *blue,
 *                           CIE_hue *white)
 *
 *      Description:    Get the current gamma and color correction parameters.
 *
 *      Return value:   None.
 *
 *      See also:       IP_SetColorCorrect
 *                      IP_GetDrawMask
 *                      IP_GetDrawMode
 *                      IP_GetDrawColor
 *                      IP_GetFormat
 *                      IP_PushState
 *                      IP_PopState
 *
 ***************************************************************************/

void IP_GetColorCorrect (Float32 *gamma, CIE_hue *red, CIE_hue *green,
                         CIE_hue *blue, CIE_hue *white)
{
    *gamma = IP_StateGamma;
    *red   = IP_StateHueRed;
    *green = IP_StateHueGreen;
    *blue  = IP_StateHueBlue;
    *white = IP_StateHueWhite;
    return;
}

/***************************************************************************
 *[@e
 *      Name:           IP_PushState
 *
 *      Usage:          Push current state onto stack.
 *
 *      Synopsis:       Int32 IP_PushState (void)
 *
 *      Description:    Push current state onto stack.
 *                      The state consists of the following members:
 *                          Pixel format.
 *                          Draw mask.
 *                          Draw color.
 *                          Draw mode.
 *
 *      Return value:   The current stack index starting with zero.
 *                      If the maximum stack depth is exceed, -1 is returned.
 *                      The maximum stack depth is 10.
 *
 *      See also:       IP_PopState
 *                      IP_GetFormat
 *                      IP_GetDrawMask
 *                      IP_GetDrawColor
 *                      IP_GetDrawMode
 *
 ***************************************************************************/

Int32 IP_PushState (void)
{
    if (sStackIndex >= MAX_STACK-1) {
         /* Maximum stack depth exceeded */
        return -1;
    }
    sStackIndex++;

    IP_GetFormat    (sStackFormat[sStackIndex]);
    IP_GetDrawMask  (sStackDrawMask[sStackIndex]);
    IP_GetDrawColor (sStackDrawColor[sStackIndex]);
    IP_GetDrawMode  (sStackDrawMode[sStackIndex]);
    return sStackIndex;
}

/***************************************************************************
 *[@e
 *      Name:           IP_PopState
 *
 *      Usage:          Pop current state from stack.
 *
 *      Synopsis:       Int32 IP_PopState (void)
 *
 *      Description:    Pop current state from stack.
 *                      The state consists of the following members:
 *                          Pixel format.
 *                          Draw mask.
 *                          Draw color.
 *                          Draw mode.
 *
 *      Return value:   The current stack index starting with zero.
 *                      If more IP_PopState functions have been called than 
 *                      preceeding IP_PushState functions, -1 is returned.
 *
 *      See also:       IP_PushState
 *                      IP_SetFormat
 *                      IP_SetDrawMask
 *                      IP_SetDrawColor
 *                      IP_SetDrawMode
 *
 ***************************************************************************/

Int32 IP_PopState (void)
{
    if (sStackIndex < 0) {
         /* State stack is empty. */
        return -1;
    }

    IP_SetFormat    (sStackFormat[sStackIndex]);
    IP_SetDrawMask  (sStackDrawMask[sStackIndex]);
    IP_SetDrawColor (sStackDrawColor[sStackIndex]);
    IP_SetDrawMode  (sStackDrawMode[sStackIndex]);
    sStackIndex--;
    return sStackIndex;
}

/***************************************************************************
 *[@e
 *      Name:           IP_SetNumThreads
 *
 *      Usage:          Set the number of threads used for parallel rocessing.
 *
 *      Synopsis:       UT_Bool IP_SetNumThreads (Int32 numThreads)
 *
 *      Description:    Attempt to allocate "numThreads" threads for parallel
 *                      operations on images. 
 *                      If "numThreads" is set to zero (which is the default),
 *                      serial processing is used.
 *
 *                      Some operating systems do not support the thread
 *                      concurrency mechanism on which the parallel image
 *                      operations are based. On these systems, all image
 *                      processing is done serially, regardless of the
 *                      number of threads specified with IP_SetNumThreads.
 *
 *      Return value:   UT_True if successful, else UT_False.
 *
 *      See also:       IP_GetNumThreads
 *
 ***************************************************************************/

UT_Bool IP_SetNumThreads (Int32 numThreads)
{
    if (numThreads < 0) {
         UT_ErrSetNum (UT_ErrParamInvalid, str_inotless, str_numthread, 0);
        return UT_False;
    }
    IP_StateNumThreads = numThreads;
    return UT_True;
}

/***************************************************************************
 *[@e
 *      Name:           IP_GetNumThreads
 *
 *      Usage:          Get the number of threads used for parallel processing.
 *
 *      Synopsis:       Int32 IP_GetNumThreads (void)
 *
 *      Description:    Return the number of threads used for parallel processing.
 *
 *      Return value:   Return the current number of threads allocated
 *                      for parallel processing.
 *
 *      See also:       IP_SetNumThreads
 *
 ***************************************************************************/
Int32 IP_GetNumThreads (void)
{
     return IP_StateNumThreads;
}
