/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         Utilities
 *      Filename:       UTT_SwatchIf.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Functions for the Tcl standard function library which
 *                      use the system clock to simulate stop watches which run
 *                      in real time.
 *
 *                      This module implements a new command "poSwatch" to
 *                      generate a swatch and miscellaneous swatch operations.
 *
 *      Additional documentation:
 *                      None.
 *                      
 *      Exported functions:
 *
 **************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "UT_Compat.h"

#include "UT_Error.h"
#include "UT_Memory.h"
#include "UT_Swatch.h"

#include <tcl.h>
#include "UTT_TclIf.h"

#include "UTT_SwatchTclIf.h"

static UT_Bool
    initialized = UT_False;

UT_Bool UTT_SwatchNew (OBJPARAMLIST)
{
    UT_Swatch sw;

    UTT_ParamHelp ("NewSwatch");

    if (!(sw = UT_SwatchNew ())) {
        return UT_False;
    }
    UT_SwatchStop  (sw);
    UT_SwatchReset (sw);
    *clientData = sw;
    return UT_True;
}

UT_Bool UTT_SwatchStop (ELEMPARAMLIST)
{
    UT_Swatch sw = (UT_Swatch) clientData;

    UTT_ParamHelp ("Stop");

    UT_SwatchStop (sw);
    return UT_True;
}

UT_Bool UTT_SwatchRun (ELEMPARAMLIST)
{
    UT_Swatch sw = (UT_Swatch) clientData;

    UTT_ParamHelp ("Run");

    UT_SwatchRun (sw);
    return UT_True;
}

UT_Bool UTT_SwatchReset (ELEMPARAMLIST)
{
    UT_Swatch sw = (UT_Swatch) clientData;

    UTT_ParamHelp ("Reset");

    UT_SwatchReset (sw);
    return UT_True;
}

UT_Bool UTT_SwatchLookup (ELEMPARAMLIST)
{
    UT_Swatch sw = (UT_Swatch) clientData;

    UTT_ParamHelp ("Lookup");

    UTT_SetFixFloat64Result (UT_SwatchLookup (sw));
    return UT_True;
}

/* Delete a Tcl swatch object. */

static void DeleteSwatch (ClientData clientData)
{
    UT_Swatch sw = (UT_Swatch) clientData;

    UT_SwatchDelete (sw);
    return;
}

static UTT_TclObjFunc swatchfunc[] =
{
    {UTT_SwatchNew,    "NewSwatch"},
};

static UTT_TclElemFunc swatchelemfunc[] =
{
    {UTT_SwatchStop,   "Stop"},
    {UTT_SwatchRun,    "Run"},
    {UTT_SwatchReset,  "Reset"},
    {UTT_SwatchLookup, "Lookup"},
};

static int SwatchElementCmd (TCLPARAMLIST)
{
    Int32 i;

    if (argc < 2) {
        Tcl_AppendResult (interp, "Available poSwatch options:\n", (char *)NULL);
        for (i=0; i< sizeof (swatchelemfunc) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, "  ", swatchelemfunc[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }

    if (argc == 2 && argv[1][0] == '?') {
        for (i=0; i< sizeof (swatchelemfunc) / sizeof (UTT_TclElemFunc); i++) {
            Tcl_AppendResult (interp, swatchelemfunc[i].name, " ", (char *)NULL);
        }
        return TCL_OK;
    }

    for (i=0; i< sizeof (swatchelemfunc) / sizeof (UTT_TclElemFunc); i++) {
        if (strcmp (argv[1], swatchelemfunc[i].name) == 0) {
            if (!swatchelemfunc[i].func (clientData, interp, argc, argv)) {
                if (strcmp (Tcl_GetStringResult (interp), "") == 0) {
                    Tcl_AppendResult (interp, UT_ErrStr, " (", UT_ErrAddStr, ")", (char *)NULL); 
                }
                return TCL_ERROR;
            } else {
                return TCL_OK;
            }
        }
    }

    Tcl_AppendResult (interp, "Bad poSwatch option \"", argv[1], "\". Must be:\n", (char *)NULL); 
    for (i=0; i< sizeof (swatchelemfunc) / sizeof (UTT_TclElemFunc); i++) {
        Tcl_AppendResult (interp, "  ", swatchelemfunc[i].name, "\n", (char *)NULL);
    }
    return TCL_ERROR;
}

static int SwatchCmd (TCLPARAMLIST)
{
    Int32 i;
    UT_Bool found = UT_False;
    UT_Swatch sw = NULL;
    char swatchName[20];

    static UInt32 swatchNum = 1;

    if (argc < 2) {
        Tcl_AppendResult (interp, "Available poSwatch subcommands:\n", (char *)NULL);
        for (i=0; i< sizeof (swatchfunc) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, "  ", swatchfunc[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }

    if (argc == 2 && argv[1][0] == '?') {
        for (i=0; i< sizeof (swatchfunc) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, swatchfunc[i].name, " ", (char *)NULL);
        }
        return TCL_OK;
    }

    for (i=0; i< sizeof (swatchfunc) / sizeof (UTT_TclObjFunc); i++) {
        if (strcmp (argv[1], swatchfunc[i].name) == 0) {
            if (!swatchfunc[i].func ((ClientData *)&sw, interp, argc, argv)) {
                if (strcmp (Tcl_GetStringResult (interp), "") == 0) {
                    Tcl_AppendResult (interp, UT_ErrStr, " (", UT_ErrAddStr, ")", (char *)NULL);
                }
                return TCL_ERROR;
            } else {
                found = UT_True;
                break;
            }
        }
    }
    if (!found) {
        Tcl_AppendResult (interp, "Bad poSwatch subcommand \"", argv[1], "\". Must be:\n", (char *)NULL);
        for (i=0; i< sizeof (swatchfunc) / sizeof (UTT_TclObjFunc); i++) {
            Tcl_AppendResult (interp, "  ", swatchfunc[i].name, "\n", (char *)NULL);
        }
        return TCL_ERROR;
    }
    if (sw) {
        sprintf (swatchName, "poSwatch%d", swatchNum);
        Tcl_CreateCommand (interp, swatchName, (Tcl_CmdProc *)SwatchElementCmd,
                           (ClientData) sw, (Tcl_CmdDeleteProc *)DeleteSwatch);
        Tcl_SetResult (interp, swatchName, TCL_VOLATILE);
        swatchNum++;
    }

    return TCL_OK;
}

int UTT_InitTclSwatchLib (Tcl_Interp *interp)
{
    if (initialized) {
        return TCL_OK;
    }
    Tcl_CreateCommand (interp, "poSwatch", (Tcl_CmdProc *) SwatchCmd,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) DeleteSwatch);
    initialized = UT_True;
    return TCL_OK;
}
