/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         Utilities
 *      Filename:       UTT_VecIf.c
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    The interface between the vector and matrix 
 *                      library and the Tcl interpreter.
 *                      The functions listed below deal with "vectors",
 *                      "transformation matrices" and "coordinate systems".
 *                      These are defined as follows:
 *
 *                      - A vector is an array of two or three floating point
 *                        numbers (2D or 3D vector). A vector represents either
 *                        a point in space or a direction.
 *
 *                      - A transformation matrix is an array of 6 or 12
 *                        floating point numbers (2D or 3D transformation
 *                        matrix). A matrix can represent any combination
 *                        of translations, rotations and scalings.
 *
 *                      - A coordinate system is an array of 6 or 12 floating
 *                        point numbers (2D or 3D coordinate system).
 *                        The first two (three) numbers represent the offset
 *                        of the coordinate system's origin relative to some
 *                        other coordinate system, called its "parent".
 *                        The next four (nine) numbers are interpreted as two
 *                        (three) vectors representing the direction of the
 *                        coordinate system's axes relative to its parent.
 *
 *                      - The following abbreviations are used:
 *                        UT_Vec2D      array of  2 floats
 *                        UT_Vec3D      array of  3 floats
 *                        UT_Tfm2D      array of  6 floats
 *                        UT_Tfm3D      array of 12 floats
 *                        UT_CoSys2D    array of  6 floats
 *                        UT_CoSys3D    array of 12 floats
 *
 *      Additional documentation:
 *                      None.
 *
 *      Exported functions:
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "UT_Compat.h"

#include "UT_Error.h"
#include "UT_Vector.h"

#include <tcl.h>
#include "UTT_TclIf.h"

#include "UTT_Private.h"
#include "UTT_VecTclIf.h"

UT_Bool UTT_VecAngle2D (TCLPARAMLIST)
{
    UT_Vec2D v1, v2;

    UTT_ParamHelp ("VecAngle2D vec1 vec2");

    UTT_GetFixVec2D (1, v1);
    UTT_GetFixVec2D (2, v2);
    UTT_SetFixFloat64Result (UT_VecAngle2D (v1, v2));
    return UT_True;
}

UT_Bool UTT_VecAngle3D (TCLPARAMLIST)
{
    UT_Vec3D v1, v2, nrm;

    UTT_ParamHelp ("VecAngle3D vec1 vec2 vecNormal");

    UTT_GetFixVec3D (1, v1);
    UTT_GetFixVec3D (2, v2);
    UTT_GetFixVec3D (3, nrm);
    UTT_SetFixFloat64Result (UT_VecAngle3D (v1, v2, nrm));
    return UT_True;
}

UT_Bool UTT_VecDistPointLine2D (TCLPARAMLIST)
{
    UT_Vec2D a, b, d, r;
    Float64  f;

    UTT_ParamHelp ("VecDistPointLine2D aVec bVec dVec f:out vec:out");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UTT_GetFixVec2D (3, d);
    UT_VecDistPointLine2D (a, b, d, &f, r);
    UTT_SetFixFloat64 (4, f);
    UTT_SetFixVec2D (5, r);
    return UT_True;
}

UT_Bool UTT_VecDistPointLine3D (TCLPARAMLIST)
{
    UT_Vec3D a, b, d, r;
    Float64  f;

    UTT_ParamHelp ("VecDistPointLine3D aVec bVec dVec f:out vec:out");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UTT_GetFixVec3D (3, d);
    UT_VecDistPointLine3D (a, b, d, &f, r);
    UTT_SetFixFloat64 (4, f);
    UTT_SetFixVec3D (5, r);
    return UT_True;
}

UT_Bool UTT_VecPointInTria2D (TCLPARAMLIST)
{
    UT_Vec2D t1, t2, t3, p;

    UTT_ParamHelp ("VecPointInTria2D vec1 vec2 vec3");

    UTT_GetFixVec2D (1, t1);
    UTT_GetFixVec2D (2, t2);
    UTT_GetFixVec2D (3, t3);
    UTT_GetFixVec2D (4, p);
    UTT_SetFixInt32Result (UT_VecPointInTria2D (t1, t2, t3, p));
    return UT_True;
}

UT_Bool UTT_VecPointInTria3D (TCLPARAMLIST)
{
    UT_Vec3D t1, t2, t3, p;

    UTT_ParamHelp ("VecPointInTria3D vec1 vec2 vec3 pVec");

    UTT_GetFixVec3D (1, t1);
    UTT_GetFixVec3D (2, t2);
    UTT_GetFixVec3D (3, t3);
    UTT_GetFixVec3D (4, p);
    UTT_SetFixInt32Result (UT_VecPointInTria3D (t1, t2, t3, p));
    return UT_True;
}

UT_Bool UTT_VecIsectPlaneLine2D (TCLPARAMLIST)
{
    UT_Vec2D a, n, b, d;
    Float64  f;
    UT_Bool  success;

    UTT_ParamHelp ("VecIsectPlaneLine2D aVec nVec bVec dVec f:out");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, n);
    UTT_GetFixVec2D (3, b);
    UTT_GetFixVec2D (4, d);
    success = UT_VecIsectPlaneLine2D (a, n, b, d, &f);
    UTT_SetFixInt32Result (success);
    if (success) {
        UTT_SetFixFloat64 (5, f);
    }
    return UT_True;
}

UT_Bool UTT_VecIsectPlaneLine3D (TCLPARAMLIST)
{
    UT_Vec3D a, n, b, d;
    Float64  f;
    UT_Bool  success;

    UTT_ParamHelp ("VecIsectPlaneLine3D aVec nVec bVec dVec f:out");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, n);
    UTT_GetFixVec3D (3, b);
    UTT_GetFixVec3D (4, d);
    success = UT_VecIsectPlaneLine3D (a, n, b, d, &f);
    UTT_SetFixInt32Result (success);
    if (success) {
        UTT_SetFixFloat64 (5, f);
    }
    return UT_True;
}

UT_Bool UTT_VecDistLineLine3D (TCLPARAMLIST)
{
    UT_Vec3D a1, b1, a2, b2;
    Float64  d, s1, s2;
    UT_Bool  success;

    UTT_ParamHelp ("VecDistLineLine3D a1Vec b1Vec a2Vec b2Vec d:out s1:out s2:out");

    UTT_GetFixVec3D (1, a1);
    UTT_GetFixVec3D (2, b1);
    UTT_GetFixVec3D (3, a2);
    UTT_GetFixVec3D (4, b2);
    success = UT_VecDistLineLine3D (a1, b1, a2, b2, &d, &s1, &s2);
    UTT_SetFixInt32Result (success);
    if (success) {
        UTT_SetFixFloat64 (5, d);
        UTT_SetFixFloat64 (6, s1);
        UTT_SetFixFloat64 (7, s2);
    }
    return UT_True;
}

UT_Bool UTT_VecIsectPlanePlane3D (TCLPARAMLIST)
{
    UT_Vec3D n1, n2, dir, a;
    Float64  d1, d2;
    UT_Bool  success;

    UTT_ParamHelp ("VecIsectPlanePlane3D n1Vec d1Vec n2Vec d2Vec dir:out a:out");

    UTT_GetFixVec3D   (1, n1);
    UTT_GetFixFloat64 (2, d1);
    UTT_GetFixVec3D   (3, n2);
    UTT_GetFixFloat64 (4, d2);
    success = UT_VecIsectPlanePlane3D (n1, d1, n2, d2, dir, a);
    UTT_SetFixInt32Result (success);
    if (success) {
        UTT_SetFixVec3D (5, dir);
        UTT_SetFixVec3D (6, a);
    }
    return UT_True;
}

UT_Bool UTT_TfmCopy2D (TCLPARAMLIST)
{
    UT_Tfm2D tmp;

    UTT_ParamHelp ("TfmCopy2D tfm tfm:out");

    UTT_GetFixTfm2D (1, tmp);
    UTT_SetFixTfm2D (2, tmp);
    return UT_True;
}

UT_Bool UTT_TfmCopy3D (TCLPARAMLIST)
{
    UT_Tfm3D tmp;

    UTT_ParamHelp ("TfmCopy3D tfm tfm:out");

    UTT_GetFixTfm3D (1, tmp);
    UTT_SetFixTfm3D (2, tmp);
    return UT_True;
}

UT_Bool UTT_TfmBuildIdent2D (TCLPARAMLIST)
{
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmBuildIdent2D tfm:out");

    UT_TfmBuildIdent2D (T);
    UTT_SetFixTfm2D (1, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildIdent3D (TCLPARAMLIST)
{
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildIdent3D tfm:out");

    UT_TfmBuildIdent3D (T);
    UTT_SetFixTfm3D (1, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildTrans2D (TCLPARAMLIST)
{
    UT_Vec2D v;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmBuildTrans2D vec tfm:out");

    UTT_GetFixVec2D (1, v);
    UT_TfmBuildTrans2D (v, T);
    UTT_SetFixTfm2D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildTrans3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildTrans3D vec tfm:out");

    UTT_GetFixVec3D (1, v);
    UT_TfmBuildTrans3D (v, T);
    UTT_SetFixTfm3D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildRot2D (TCLPARAMLIST)
{
    Float64   ang;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmBuildRot2D angle tfm:out");

    UTT_GetFixFloat64 (1, ang);
    UT_TfmBuildRot2D (ang, T);
    UTT_SetFixTfm2D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildRot3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildRot3D vec tfm:out");

    UTT_GetFixVec3D (1, v);
    UT_TfmBuildRot3D (v, T);
    UTT_SetFixTfm3D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildScale2D (TCLPARAMLIST)
{
    UT_Vec2D v;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmBuildScale2D vec tfm:out");

    UTT_GetFixVec2D (1, v);
    UT_TfmBuildScale2D (v, T);
    UTT_SetFixTfm2D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildScale3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildScale3D vec tfm:out");

    UTT_GetFixVec3D (1, v);
    UT_TfmBuildScale3D (v, T);
    UTT_SetFixTfm3D (2, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildRotArb3D (TCLPARAMLIST)
{
    UT_Vec3D axis;
    Float32  angle;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildRotArb3D axisVec angle tfm:out");

    UTT_GetFixVec3D     (1, axis);
    UTT_GetFixFloat32   (2, angle);
    UT_TfmBuildRotArb3D (axis, angle, T);
    UTT_SetFixTfm3D     (3, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildRotArbLos3D (TCLPARAMLIST)
{
    UT_Vec3D origin, axis;
    Float32  angle;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildRotArbLos3D originVec axisVec angle tfm:out");

    UTT_GetFixVec3D        (1, origin);
    UTT_GetFixVec3D        (2, axis);
    UTT_GetFixFloat32      (3, angle);
    UT_TfmBuildRotArbLos3D (origin, axis, angle, T);
    UTT_SetFixTfm3D        (4, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildLookAt3D (TCLPARAMLIST)
{
    UT_Vec3D eye, at, up;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildLookAt3D eyeVec atVec upVec tfm:out");

    UTT_GetFixVec3D (1, eye);
    UTT_GetFixVec3D (2, at);
    UTT_GetFixVec3D (3, up);
    UT_TfmBuildLookAt3D (eye, at, up, T);
    UTT_SetFixTfm3D (4, T);
    return UT_True;
}

UT_Bool UTT_TfmBuildLookAtVec3D (TCLPARAMLIST)
{
    UT_Vec3D eye, at, up;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmBuildLookAtVec3D eyeVec atVec upVec tfm:out");

    UTT_GetFixVec3D (1, eye);
    UTT_GetFixVec3D (2, at);
    UTT_GetFixVec3D (3, up);
    UT_TfmBuildLookAtVec3D (eye, at, up, T);
    UTT_SetFixTfm3D (4, T);
    return UT_True;
}

UT_Bool UTT_TfmConcat2D (TCLPARAMLIST)
{
    UT_Tfm2D A, B, R;

    UTT_ParamHelp ("TfmConcat2D tfm1 tfm2 tfm:out");

    UTT_GetFixTfm2D (1, A);
    UTT_GetFixTfm2D (2, B);
    UT_TfmConcat2D (A, B, R);
    UTT_SetFixTfm2D (3, R);
    return UT_True;
}

UT_Bool UTT_TfmConcat3D (TCLPARAMLIST)
{
    UT_Tfm3D A, B, R;

    UTT_ParamHelp ("TfmConcat3D tfm1 tfm2 tfm:out");

    UTT_GetFixTfm3D (1, A);
    UTT_GetFixTfm3D (2, B);
    UT_TfmConcat3D (A, B, R);
    UTT_SetFixTfm3D (3, R);
    return UT_True;
}

UT_Bool UTT_TfmConcatTrans2D (TCLPARAMLIST)
{
    UT_Vec2D v;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmConcatTrans2D vec tfm tfm:out");

    UTT_GetFixVec2D (1, v);
    UTT_GetFixTfm2D (2, T);
    UT_TfmConcatTrans2D (v, T);
    UTT_SetFixTfm2D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmConcatTrans3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmConcatTrans3D vec tfm tfm:out");
 
    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmConcatTrans3D (v, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmMultTrans3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmMultTrans3D vec tfm tfm:out");
 
    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmMultTrans3D (v, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmConcatRot2D (TCLPARAMLIST)
{
    Float32 ang;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmConcatRot2D angle tfm tfm:out");

    UTT_GetFixFloat32 (1, ang);
    UTT_GetFixTfm2D (2, T);
    UT_TfmConcatRot2D (ang, T);
    UTT_SetFixTfm2D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmConcatRot3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmConcatRot3D vec tfm tfm:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmConcatRot3D (v, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmMultRot3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmMultRot3D vec tfm tfm:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmMultRot3D (v, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmConcatScale2D (TCLPARAMLIST)
{
    UT_Vec2D v;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmConcatScale2D vec tfm tfm:out");

    UTT_GetFixVec2D (1, v);
    UTT_GetFixTfm2D (2, T);
    UT_TfmConcatScale2D (v, T);
    UTT_SetFixTfm2D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmConcatScale3D (TCLPARAMLIST)
{
    UT_Vec3D v;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmConcatScale3D vec tfm tfm:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmConcatScale3D (v, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmInvert2D (TCLPARAMLIST)
{
    UT_Tfm2D A, R;

    UTT_ParamHelp ("TfmInvert2D tfm tfm:out");

    UTT_GetFixTfm2D (1, A);
    UTT_SetFixInt32Result (UT_TfmInvert2D (A, R));
    UTT_SetFixTfm2D (2, R);
    return UT_True;
}

UT_Bool UTT_TfmInvert3D (TCLPARAMLIST)
{
    UT_Tfm3D A, R;

    UTT_ParamHelp ("TfmInvert3D tfm tfm:out");

    UTT_GetFixTfm3D (1, A);
    UTT_SetFixInt32Result (UT_TfmInvert3D (A, R));
    UTT_SetFixTfm3D (2, R);
    return UT_True;
}

UT_Bool UTT_TfmTranspose3D (TCLPARAMLIST)
{
    UT_Tfm3D A, R;

    UTT_ParamHelp ("TfmTranspose3D tfm tfm:out");

    UTT_GetFixTfm3D (1, A);
    UT_TfmTranspose3D (A, R);
    UTT_SetFixTfm3D (2, R);
    return UT_True;
}

UT_Bool UTT_TfmCoord2D (TCLPARAMLIST)
{
    UT_CoSys2D a, b;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmCoord2D aCosys bCosys tfm:out");

    UTT_GetFixCoSys2D (1, a);
    UTT_GetFixCoSys2D (2, b);
    UT_TfmCoord2D (&a, &b, T);
    UTT_SetFixTfm2D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmCoord3D (TCLPARAMLIST)
{
    UT_CoSys3D a, b;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmCoord3D aCosys bCosys tfm:out");

    UTT_GetFixCoSys3D (1, a);
    UTT_GetFixCoSys3D (2, b);
    UT_TfmCoord3D (&a, &b, T);
    UTT_SetFixTfm3D (3, T);
    return UT_True;
}

UT_Bool UTT_TfmInitCoord2D (TCLPARAMLIST)
{
    UT_CoSys2D a;

    UTT_ParamHelp ("TfmInitCoord2D cosys:out");

    UT_TfmInitCoord2D (&a);
    UTT_SetFixCoSys2D (1, a);
    return UT_True;
}

UT_Bool UTT_TfmInitCoord3D (TCLPARAMLIST)
{
    UT_CoSys3D a;

    UTT_ParamHelp ("TfmInitCoord3D cosys:out");

    UT_TfmInitCoord3D (&a);
    UTT_SetFixCoSys3D (1, a);
    return UT_True;
}

UT_Bool UTT_TfmApply2D (TCLPARAMLIST)
{
    UT_Vec2D v, r;
    UT_Tfm2D T;

    UTT_ParamHelp ("TfmApply2D vec tfm vec:out");

    UTT_GetFixVec2D (1, v);
    UTT_GetFixTfm2D (2, T);
    UT_TfmApply2D (v, T, r);
    UTT_SetFixVec2D (3, r);
    return UT_True;
}

UT_Bool UTT_TfmApply3D (TCLPARAMLIST)
{
    UT_Vec3D v, r;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmApply3D vec tfm vec:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmApply3D (v, T, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_TfmApplyNormal3D (TCLPARAMLIST)
{
    UT_Vec3D v, r;
    UT_Tfm3D T;

    UTT_ParamHelp ("TfmApplyNormal3D vec tfm vec:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixTfm3D (2, T);
    UT_TfmApplyNormal3D (v, T, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_VecCopy2D (TCLPARAMLIST)
{
    UT_Vec2D tmp;

    UTT_ParamHelp ("VecCopy2D vec vec:out");

    UTT_GetFixVec2D (1, tmp);
    UTT_SetFixVec2D (2, tmp);
    return UT_True;
}

UT_Bool UTT_VecCopy3D (TCLPARAMLIST)
{
    UT_Vec3D tmp;

    UTT_ParamHelp ("VecCopy3D vec vec:out");

    UTT_GetFixVec3D (1, tmp);
    UTT_SetFixVec3D (2, tmp);
    return UT_True;
}

UT_Bool UTT_VecAdd2D (TCLPARAMLIST)
{
    UT_Vec2D a, b, r;

    UTT_ParamHelp ("VecAdd2D aVec bVec vec:out");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UT_VecAdd2D (a, b, r);
    UTT_SetFixVec2D (3, r);
    return UT_True;
}

UT_Bool UTT_VecAdd3D (TCLPARAMLIST)
{
    UT_Vec3D a, b, r;

    UTT_ParamHelp ("VecAdd3D aVec bVec vec:out");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UT_VecAdd3D (a, b, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_VecSub2D (TCLPARAMLIST)
{
    UT_Vec2D a, b, r;

    UTT_ParamHelp ("VecSub2D aVec bVec vec:out");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UT_VecSub2D (a, b, r);
    UTT_SetFixVec2D (3, r);
    return UT_True;
}

UT_Bool UTT_VecSub3D (TCLPARAMLIST)
{
    UT_Vec3D a, b, r;

    UTT_ParamHelp ("VecSub3D aVec bVec vec:out");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UT_VecSub3D (a, b, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_VecScale2D (TCLPARAMLIST)
{
    Float32 m;
    UT_Vec2D v;

    UTT_ParamHelp ("VecScale2D scale vec vec:out");

    UTT_GetFixFloat32 (1, m);
    UTT_GetFixVec2D (2, v);

    UT_VecScale2D (m, v, v);

    UTT_SetFixVec2D (3, v);
    return UT_True;
}

UT_Bool UTT_VecScale3D (TCLPARAMLIST)
{
    Float32 m;
    UT_Vec3D v;

    UTT_ParamHelp ("VecScale3D scale vec vec:out");

    UTT_GetFixFloat32 (1, m);
    UTT_GetFixVec3D (2, v);

    UT_VecScale3D (m, v, v);

    UTT_SetFixVec3D (3, v);
    return UT_True;
}

UT_Bool UTT_VecCrossProd2D (TCLPARAMLIST)
{
    UT_Vec2D a, b;

    UTT_ParamHelp ("VecCrossProd2D aVec bVec");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UTT_SetFixFloat64Result (UT_VecCrossProd2D (a, b));
    return UT_True;
}

UT_Bool UTT_VecCrossProd3D (TCLPARAMLIST)
{
    UT_Vec3D a, b, r;

    UTT_ParamHelp ("VecCrossProd3D aVec bVec vec:out");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UT_VecCrossProd3D (a, b, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_VecDotProd2D (TCLPARAMLIST)
{
    UT_Vec2D a, b;

    UTT_ParamHelp ("VecDotProd2D aVec bVec");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UTT_SetFixFloat64Result (UT_VecDotProd2D (a, b));
    return UT_True;
}

UT_Bool UTT_VecDotProd3D (TCLPARAMLIST)
{
    UT_Vec3D a, b;

    UTT_ParamHelp ("VecDotProd3D aVec bVec");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UTT_SetFixFloat64Result (UT_VecDotProd3D (a, b));
    return UT_True;
}

UT_Bool UTT_VecLength2D (TCLPARAMLIST)
{
    UT_Vec2D v;

    UTT_ParamHelp ("VecLength2D vec");

    UTT_GetFixVec2D (1, v);
    UTT_SetFixFloat64Result (UT_VecLength2D (v));
    return UT_True;
}

UT_Bool UTT_VecLength3D (TCLPARAMLIST)
{
    UT_Vec3D v;

    UTT_ParamHelp ("VecLength3D vec");

    UTT_GetFixVec3D (1, v);
    UTT_SetFixFloat64Result (UT_VecLength3D (v));
    return UT_True;
}

UT_Bool UTT_VecDist2D (TCLPARAMLIST)
{
    UT_Vec2D a, b;

    UTT_ParamHelp ("VecDist2D aVec bVec");

    UTT_GetFixVec2D (1, a);
    UTT_GetFixVec2D (2, b);
    UTT_SetFixFloat64Result (UT_VecDist2D (a, b));
    return UT_True;
}

UT_Bool UTT_VecDist3D (TCLPARAMLIST)
{
    UT_Vec3D a, b;

    UTT_ParamHelp ("VecDist3D aVec bVec");

    UTT_GetFixVec3D (1, a);
    UTT_GetFixVec3D (2, b);
    UTT_SetFixFloat64Result (UT_VecDist3D (a, b));
    return UT_True;
}

UT_Bool UTT_VecUnit2D (TCLPARAMLIST)
{
    UT_Vec2D v;

    UTT_ParamHelp ("VecUnit2D vec vec:out");

    UTT_GetFixVec2D (1, v);
    UTT_SetFixInt32Result (UT_VecUnit2D (v));
    UTT_SetFixVec2D (2, v);
    return UT_True;
}

UT_Bool UTT_VecUnit3D (TCLPARAMLIST)
{
    UT_Vec3D v;

    UTT_ParamHelp ("VecUnit3D vec vec:out");

    UTT_GetFixVec3D (1, v);
    UTT_SetFixInt32Result (UT_VecUnit3D (v));
    UTT_SetFixVec3D (2, v);
    return UT_True;
}

UT_Bool UTT_VecLinComb2D (TCLPARAMLIST)
{
    UT_Vec2D v1, v2, r;
    Float32 a, b;

    UTT_ParamHelp ("VecLinComb2D v1Vec v2Vec a b vec:out");

    UTT_GetFixVec2D   (1, v1);
    UTT_GetFixVec2D   (2, v2);
    UTT_GetFixFloat32 (3, a);
    UTT_GetFixFloat32 (4, b);
    UT_VecLinComb2D (v1, v2, a, b, r);
    UTT_SetFixVec2D   (5, r);
    return UT_True;
}

UT_Bool UTT_VecLinComb3D (TCLPARAMLIST)
{
    UT_Vec3D v1, v2, r;
    Float32 a, b;

    UTT_ParamHelp ("VecLinComb3D v1Vec v2Vec a b vec:out");

    UTT_GetFixVec3D   (1, v1);
    UTT_GetFixVec3D   (2, v2);
    UTT_GetFixFloat32 (3, a);
    UTT_GetFixFloat32 (4, b);
    UT_VecLinComb3D (v1, v2, a, b, r);
    UTT_SetFixVec3D   (5, r);
    return UT_True;
}

UT_Bool UTT_VecReflect2D (TCLPARAMLIST)
{
    UT_Vec2D v, n, r;

    UTT_ParamHelp ("VecReflect2D vVec nVec vec:out");

    UTT_GetFixVec2D (1, v);
    UTT_GetFixVec2D (2, n);
    UT_VecReflect2D (v, n, r);
    UTT_SetFixVec2D (3, r);
    return UT_True;
}

UT_Bool UTT_VecReflect3D (TCLPARAMLIST)
{
    UT_Vec3D v, n, r;

    UTT_ParamHelp ("VecReflect3D vVec nVec vec:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixVec3D (2, n);
    UT_VecReflect3D (v, n, r);
    UTT_SetFixVec3D (3, r);
    return UT_True;
}

UT_Bool UTT_VecDecompose2D (TCLPARAMLIST)
{
    UT_Vec2D v, n, m, p;

    UTT_ParamHelp ("VecDecompose2D vVec nVec mVec:out pVec:out");

    UTT_GetFixVec2D (1, v);
    UTT_GetFixVec2D (2, n);
    UT_VecDecompose2D (v, n, m, p);
    UTT_SetFixVec2D (3, m);
    UTT_SetFixVec2D (4, p);
    return UT_True;
}

UT_Bool UTT_VecDecompose3D (TCLPARAMLIST)
{
    UT_Vec3D v, n, m, p;

    UTT_ParamHelp ("VecDecompose3D vVec nVec mVec:out pVec:out");

    UTT_GetFixVec3D (1, v);
    UTT_GetFixVec3D (2, n);
    UT_VecDecompose3D (v, n, m, p);
    UTT_SetFixVec3D (3, m);
    UTT_SetFixVec3D (4, p);
    return UT_True;
}
