/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         Utilities
 *      Filename:       UT_Compat.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Machine independent type definitions.
 *
 *      Additional documentation:
 *                      None.
 *
 **************************************************************************/

#ifndef __UT_COMPAT_H__
#define __UT_COMPAT_H__

#include "UT_Export.h"

/*
    - Some data types which should be the same precision on all machines:

        Int8            Signed 8 bit integer
        UInt8           Unsigned 8 bit integer
        Int16           Signed 16 bit integer
        UInt16          Unsigned 16 bit integer
        Int32           Signed 32 bit integer
        UInt32          Unsigned 32 bit integer
        Int64           Signed 64 bit integer
        UInt64          Unsigned 64 bit integer
        Float32         32 bit floating point
        Float64         64 bit floating point
        UT_Bool         Boolean data type

    - Memory alignment factor: Set MEM_ALIGN to n if every n'th address
      in the program's address space is suitable for storing all data
      types.  This will be used by the memory management library.

    - The operating system. Currently the following types are supported:

        OS_IS_UNIX      Linux or MacOS
        OS_IS_WIN32     Microsoft Windows

    - The maximum length of a complete file name with access path and the
      maximum length of each component of a file access path:

        MAXPATH         typically 4096 on Unix systems

    - If the operating system supports multiple program execution threads
      within a single shared address space, so that the UT_ThreadExec
      function can be implemented as defined in our parallelism library,
      then the following macro is defined:

        HAVE_THREADS

<*/

/* Check for Microsoft compiler */
#ifdef _MSC_VER
    #define CC_IS_MSC
#endif

typedef unsigned char      UInt8;
typedef unsigned short     UInt16;
typedef unsigned int       UInt32;
typedef unsigned long long UInt64;

typedef signed char        Int8;
typedef short              Int16;
typedef int                Int32;
typedef long long          Int64;

typedef float  Float32;
typedef double Float64;

#define MAXPATH 4096
#define MEM_ALIGN 8               /* Memory alignment factor */

typedef enum {
    UT_False = 0,
    UT_True  = 1
} UT_Bool;

#if defined (TCL_THREADS)
    #define HAVE_THREADS          /* Multithreading is supported */
    #define HAVE_PTHREADS         /* Multithreading with pthreads is supported */
#endif

#if defined (Linux)
    #define OS_IS_UNIX            /* Operating system is Linux */

#elif defined (Darwin)
    #define OS_IS_UNIX            /* Operating System is MacOS */

#elif defined (CC_IS_MSC)
    #define OS_IS_WIN32           /* Operating system is Windows. */

#elif defined (mingw)
    #define OS_IS_WIN32           /* Operating system is Windows using MSYS/MinGW. */

#else
        #error "Basic definitions for your operating system missing here"
#endif

#ifdef _MSC_VER
    #if _MSC_VER <= 1800 /* VS 2013 and older do not have snprintf */
        #define UT_SNPRINTF _snprintf
    #else
        #define UT_SNPRINTF snprintf
    #endif
#else
    #define UT_SNPRINTF snprintf
#endif

/* Marker for exported symbols. Normal operating systems don't need a 
   special handling for that. Only Windows needs some trick here.
*/
#if defined (OS_IS_WIN32)
    #define UT_DSO_EXPORT  __declspec( dllexport ) 
    #define UT_DSO_IMPORT  __declspec( dllimport ) 
#else
    #define UT_DSO_EXPORT
    #define UT_DSO_IMPORT
#endif

#endif /* __UT_COMPAT_H__ */
