/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         Utilities
 *      Filename:       UT_Error.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Declaration of the error and signal handling functions.
 *
 *      Additional documentation:
 *                      None.
 *
 **************************************************************************/

#ifndef __UT_ERROR_H__
#define __UT_ERROR_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "UT_Compat.h"

/* Descriptor for a set of error messages */

typedef struct UT_ErrBaseNode {
    const char *const *msg;      /* Error message array */
    Int32 n;                     /* Number of entries in msg */
    Int32 base;                  /* Error code corresponding to msg[0] */
    struct UT_ErrBaseNode *next; /* Successor in linked list */
} UT_ErrBase;

/* Various generally-used error codes */

extern POIMG_IMPORT_EXPORT UT_ErrBase UT_ErrList;

/* Error codes used by various libraries */

#define UT_ErrNone              (UT_ErrList.base +  0)
#define UT_ErrUnknownCode       (UT_ErrList.base +  1)
#define UT_ErrUnxpInput         (UT_ErrList.base +  2)
#define UT_ErrUnexpEof          (UT_ErrList.base +  3)
#define UT_ErrNotPoFile         (UT_ErrList.base +  4)
#define UT_ErrPoType            (UT_ErrList.base +  5)
#define UT_ErrIncompatible      (UT_ErrList.base +  6)
#define UT_ErrParamRange        (UT_ErrList.base +  7)
#define UT_ErrParamInvalid      (UT_ErrList.base +  8)
#define UT_ErrNotSupported      (UT_ErrList.base +  9)

#if defined (OS_IS_UNIX) || \
    defined (OS_IS_WIN32)

    /* Operating system error codes */

    #define UT_ErrNotOwner      (UT_ErrList.base + 10)
    #define UT_ErrNoFile        (UT_ErrList.base + 11)
    #define UT_ErrNoProcess     (UT_ErrList.base + 12)
    #define UT_ErrIntr          (UT_ErrList.base + 13)
    #define UT_ErrIoErr         (UT_ErrList.base + 14)
    #define UT_ErrNoDev         (UT_ErrList.base + 15)
    #define UT_ErrTooBig        (UT_ErrList.base + 16)
    #define UT_ErrExec          (UT_ErrList.base + 17)
    #define UT_ErrBadFile       (UT_ErrList.base + 18)
    #define UT_ErrChild         (UT_ErrList.base + 19)
    #define UT_ErrAgain         (UT_ErrList.base + 20)
    #define UT_ErrNoMem         (UT_ErrList.base + 21)
    #define UT_ErrPermission    (UT_ErrList.base + 22)
    #define UT_ErrAddress       (UT_ErrList.base + 23)
    #define UT_ErrNoBlock       (UT_ErrList.base + 24)
    #define UT_ErrBusy          (UT_ErrList.base + 25)
    #define UT_ErrExists        (UT_ErrList.base + 26)
    #define UT_ErrXdevive       (UT_ErrList.base + 27)
    #define UT_ErrNotDir        (UT_ErrList.base + 28)
    #define UT_ErrIsDir         (UT_ErrList.base + 29)
    #define UT_ErrInvalid       (UT_ErrList.base + 30)
    #define UT_ErrNumFiles      (UT_ErrList.base + 31)
    #define UT_ErrOpenFiles     (UT_ErrList.base + 32)
    #define UT_ErrNoTty         (UT_ErrList.base + 33)
    #define UT_ErrTextBusy      (UT_ErrList.base + 34)
    #define UT_ErrFileTooBig    (UT_ErrList.base + 35)
    #define UT_ErrFsFull        (UT_ErrList.base + 36)
    #define UT_ErrSeekPipe      (UT_ErrList.base + 38)
    #define UT_ErrFsReadonly    (UT_ErrList.base + 38)
    #define UT_ErrNumLinks      (UT_ErrList.base + 39)
    #define UT_ErrPipe          (UT_ErrList.base + 40)
    #define UT_ErrDomain        (UT_ErrList.base + 41)
    #define UT_ErrRange         (UT_ErrList.base + 42)

    /* Xenix / SystemV-Unix specials */

    #define UT_ErrFsNotClean    (UT_ErrList.base + 43)
    #define UT_ErrDeadlock      (UT_ErrList.base + 44)
    #define UT_ErrInvSem        (UT_ErrList.base + 45)
    #define UT_ErrNoSem         (UT_ErrList.base + 46)
    #define UT_ErrIsSem         (UT_ErrList.base + 47)
    #define UT_ErrNoMessage     (UT_ErrList.base + 48)
    #define UT_ErrRmIdent       (UT_ErrList.base + 49)
    #define UT_ErrNoLock        (UT_ErrList.base + 50)
    #define UT_ErrUnknown       (UT_ErrList.base + 51)

    /* BSD-Unix specials */

    #define UT_ErrWouldBlock    (UT_ErrList.base + 52)
    #define UT_ErrInProgress    (UT_ErrList.base + 53)
    #define UT_ErrAlready       (UT_ErrList.base + 54)
    #define UT_ErrNotSock       (UT_ErrList.base + 55)
    #define UT_ErrDestAddrReq   (UT_ErrList.base + 56)
    #define UT_ErrMsgSize       (UT_ErrList.base + 57)
    #define UT_ErrPrototype     (UT_ErrList.base + 58)
    #define UT_ErrNoProtoOpt    (UT_ErrList.base + 59)
    #define UT_ErrProtNoSupp    (UT_ErrList.base + 60)
    #define UT_ErrSockNoSupp    (UT_ErrList.base + 61)
    #define UT_ErrOpNoSupp      (UT_ErrList.base + 62)
    #define UT_ErrPfNoSupp      (UT_ErrList.base + 63)
    #define UT_ErrAfNoSupp      (UT_ErrList.base + 64)
    #define UT_ErrAddrInUse     (UT_ErrList.base + 65)
    #define UT_ErrAddrNotAvail  (UT_ErrList.base + 66)
    #define UT_ErrNetDown       (UT_ErrList.base + 67)
    #define UT_ErrNetUnreach    (UT_ErrList.base + 68)
    #define UT_ErrNetReset      (UT_ErrList.base + 69)
    #define UT_ErrConnAbort     (UT_ErrList.base + 70)
    #define UT_ErrConnReset     (UT_ErrList.base + 71)
    #define UT_ErrNoBufs        (UT_ErrList.base + 72)
    #define UT_ErrIsConn        (UT_ErrList.base + 73)
    #define UT_ErrNotConn       (UT_ErrList.base + 74)
    #define UT_ErrShutdown      (UT_ErrList.base + 75)
    #define UT_ErrTimedOut      (UT_ErrList.base + 76)
    #define UT_ErrConnRefused   (UT_ErrList.base + 77)
    #define UT_ErrLoop          (UT_ErrList.base + 78)
    #define UT_ErrNameTooLong   (UT_ErrList.base + 79)
    #define UT_ErrNotEmpty      (UT_ErrList.base + 80)

#else

    #error "Operating system error codes missing here"
#endif

extern POIMG_IMPORT_EXPORT void UT_ErrInit (void);
extern POIMG_IMPORT_EXPORT void UT_ErrSupply (const char * const * const, Int32, UT_ErrBase *);
extern POIMG_IMPORT_EXPORT void UT_ErrPrintStr (const char *);
extern POIMG_IMPORT_EXPORT void UT_ErrPrintAddStr (void);
extern POIMG_IMPORT_EXPORT void UT_ErrFatal (const char *, const char *);
extern POIMG_IMPORT_EXPORT Int32   UT_ErrFromOs (void);

extern POIMG_IMPORT_EXPORT void UT_ErrSetNum (Int32, const char *, ...);
extern POIMG_IMPORT_EXPORT void UT_ErrAppendStr (const char *, const char *, ...);

/* UT_ErrNum, UT_ErrStr and UT_ErrAddStr may only be changed by UT_ErrSetNum
   and UT_ErrAppendStr. Don't alter their values directly! */

#define UT_ErrAddStrLen 1000    /* Maximum length of UT_ErrAddStr string */

extern POIMG_IMPORT_EXPORT Int32 UT_ErrNum;             /* Error code from last call to UT_ErrSetNum */
extern POIMG_IMPORT_EXPORT const char *UT_ErrStr;       /* Error message corresponding to UT_ErrNum */
extern POIMG_IMPORT_EXPORT char UT_ErrAddStr[UT_ErrAddStrLen + 1];      /* Text explaining UT_ErrStr */
extern POIMG_IMPORT_EXPORT const char *UT_ErrMathExStr; /* Type of most recent arithmetic exception */

#ifdef __cplusplus
}
#endif

#endif /* __UT_ERROR_H__ */
