/**************************************************************************
 *{@C
 *      Copyright:      1988-2025 Paul Obermeier (obermeier@poSoft.de)
 *
 *      Module:         Utilities
 *      Filename:       UT_FileHeader.h
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    Definition of the data types and declaration of the
 *                      functions dealing with the file header used in poSoft
 *                      files.
 *
 *      Additional documentation:
 *                      None.
 *
 **************************************************************************/

#ifndef __UT_FILEHEADER_H__
#define __UT_FILEHEADER_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "UT_Compat.h"

/* When adding entries in any of the three typedefs, note the following:
   - Always update the entry "#define UT_Num*Types".
   - Always update the corresponding type names in UT_FileHeader.c
*/

/* Type of data contained in the file. */
typedef enum  {
    UT_FileTypeModel     = 0,
    UT_FileTypeImage     = 1,
    UT_FileTypeCurve     = 2,
    UT_FileTypeParticle  = 3,
    UT_FileTypeGrid      = 4
} UT_FileType;

#define UT_NumFileTypes 5

/* Sub types for dealing with variations of file types. */
typedef enum  {
    UT_FileSubTypePoModel      =  0,
    UT_FileSubTypeIrModel      =  1,
    UT_FileSubTypeAwModel      =  2,
    UT_FileSubTypePoImage      =  3,
    UT_FileSubTypeRawImage     =  4,
    UT_FileSubTypeSpline2D     =  5,
    UT_FileSubTypeWayPoint     =  6,
    UT_FileSubTypePoParticle   =  7,
    UT_FileSubTypeModtran      =  8,
    UT_FileSubTypeDensity      =  9,
    UT_FileSubTypeNastranModel = 10 
} UT_FileSubType;

#define UT_NumFileSubTypes 11

/* Type of data coding. */
typedef enum  {
    UT_FileCodingTypeAscii          = 0,
    UT_FileCodingTypePortableBinary = 1,
    UT_FileCodingTypeIntel          = 2,
    UT_FileCodingTypeMotorola       = 3
} UT_FileCodingType;

#define UT_NumFileCodingTypes 4

/* Structure for file header information. */
typedef struct {
    UInt16            majorVersion;
    UInt16            minorVersion;
    UT_FileType       fileType;
    UT_FileSubType    fileSubType;
    UT_FileCodingType fileCodingType;
} UT_FileHeader;

/* Functions to read and write file headers. */
extern POIMG_IMPORT_EXPORT UT_Bool UT_FileHeaderRead  (FILE *fp, UT_FileHeader *hdr);
extern POIMG_IMPORT_EXPORT UT_Bool UT_FileHeaderWrite (FILE *fp, UT_FileType fileType,
                                          UT_FileSubType fileSubType,
                                          UT_FileCodingType fileCodingType);
extern POIMG_IMPORT_EXPORT UT_Bool UT_FileHeaderWriteStruct (FILE *fp, const UT_FileHeader *hdr);

extern POIMG_IMPORT_EXPORT const char *UT_GetFileTypeName (UT_FileType fileType);
extern POIMG_IMPORT_EXPORT const char *UT_GetFileSubTypeName (UT_FileSubType fileSubType);
extern POIMG_IMPORT_EXPORT const char *UT_GetFileCodingTypeName (UT_FileCodingType fileCodingType);

extern POIMG_IMPORT_EXPORT UT_Bool 
    UT_GetFileType       (const char *typeName, UT_FileType       *type),
    UT_GetFileSubType    (const char *typeName, UT_FileSubType    *subType),
    UT_GetFileCodingType (const char *typeName, UT_FileCodingType *codingType);

extern POIMG_IMPORT_EXPORT void UT_PrintFileTypeNames (void);
extern POIMG_IMPORT_EXPORT void UT_PrintFileSubTypeNames (void);
extern POIMG_IMPORT_EXPORT void UT_PrintFileCodingTypeNames (void);
extern POIMG_IMPORT_EXPORT void UT_PrintFileVersion (void);

#ifdef __cplusplus
}
#endif

#endif /* __UT_FILEHEADER_H__ */
