namespace eval poImgTest {

    variable ns [namespace current]

    # Procs used for constraints.
    namespace export haveTk86 haveTk90
    namespace export havePkgImg havePkgMem
    namespace export havePhotoSupport

    # Utility procs.
    namespace export photoSize photosIdent
    namespace export poImgSize poImgsIdent
    namespace export poImgInfo poImgsCount
    namespace export getChanMap
    namespace export rawImgInfo
    namespace export fileRead filesIdent
    namespace export intListsIdent floatListsIdent

    namespace export createOnePixelRGB
    namespace export createUniform createUniformGrey createUniformRGB
    namespace export createScanlinesRGB
    namespace export createColorbar3Chan
    namespace export createColorbar4Chan
    namespace export createBrushImg

    # Debug procs.
    namespace export poImgView photoView
    namespace export poImgsList
    namespace export useVerbose

    # RAW image data procs.
    namespace export rawTestImgCheck rawTestImgCheckStats
    namespace export getByteOrderFromRawFilename
    namespace export getScanOrderFromRawFilename
    namespace export getPixelTypeFromRawFilename
    namespace export getFormatFromRawFilename

    variable sTopCount 1

    # The following variable may be changed manually to
    # enbale usage of "-verbose" option when reading/writing images.
    variable sVerbose false

    proc _versionCompare { version1 version2 } {
        # A wrapper around package vcompare to handle alpha or beta versions
        # containing "a" or "b", ex. 8.7a4.
        set version(1) $version1
        set version(2) $version2
        set versionList(1) [split $version1 "."]
        set versionList(2) [split $version2 "."]
        foreach num { 1 2 } {
            if { [llength $versionList($num)] == 2 } {
                set major [lindex $versionList($num) 0]
                set minor [lindex $versionList($num) 1]
                if { ! [string is integer -strict $minor] } {
                    lassign [split $minor "ab"] minor patch
                    set version($num) [format "%d.%d.%d" $major $minor $patch]
                }
            }
        }
        return [package vcompare $version(1) $version(2)]
    }

    proc haveTk86 {} {
        return [expr [_versionCompare "8.6" $::tk_patchLevel] <= 0]
    }

    proc haveTk90 {} {
        return [expr [_versionCompare "9.0" $::tk_patchLevel] <= 0]
    }

    proc havePkgImg {} {
        set retVal [catch {package require Img}]
        catch {package require img::dted}
        catch {package require img::flir}
        catch {package require img::raw}
        return [expr ! $retVal]
    }

    proc havePkgMem {} {
        set retVal [catch {package require poMemory}]
        return [expr ! $retVal]
    }

    proc havePhotoSupport {} {
        return [poImageState HasPhotoSupport]
    }

    proc poImgSize { poImg } {
        if { $poImg eq "" } {
            return [list]
        }
        $poImg GetImageInfo w h
        return [list $w $h]
    }

    proc poImgInfo { poImg numChans } {
        if { $poImg eq "" } {
            return [list]
        }
        $poImg GetImageInfo w h
        $poImg GetImageFormat fmt

        set chanList [list]
        switch -exact -- $numChans {
            1 { set chanList [list $::BRIGHTNESS] }
            2 { set chanList [list $::BRIGHTNESS $::MATTE] }
            3 { set chanList [list $::RED $::GREEN $::BLUE] }
            4 { set chanList [list $::RED $::GREEN $::BLUE $::MATTE] }
        }

        # Check, if all channels not included in chanList are OFF.
        for { set chan 0 } { $chan < $::NUMCHAN } { incr chan } {
            if { $chan ni $chanList } {
                if { [lindex $fmt $chan] != $::OFF } {
                    return [list]
                }
            }
        }

        set retList [list $w $h]
        foreach chan $chanList {
            lappend retList [lindex $fmt $chan]
        }
        return $retList
    }

    proc rawImgInfo { poImg numChans withAlpha expFmt } {
        if { $poImg eq "" } {
            return [list]
        }
        $poImg GetImageInfo w h
        $poImg GetImageFormat fmt
        if { ($numChans == 2 || $numChans == 4) && !$withAlpha } {
            incr numChans -1
        }
        if { $numChans == 1 } {
            set retVal [expr [lindex $fmt $::BRIGHTNESS] == $expFmt]
        } elseif { $numChans == 2 } {
            set retVal [expr [lindex $fmt $::BRIGHTNESS] == $expFmt && [lindex $fmt $::MATTE] == $expFmt]
        } elseif { $numChans == 3 } {
            set retVal [expr [lindex $fmt $::RED] == $expFmt && [lindex $fmt $::GREEN] == $expFmt && [lindex $fmt $::BLUE] == $expFmt]
        } elseif { $numChans == 4 } {
            set retVal [expr [lindex $fmt $::RED] == $expFmt && [lindex $fmt $::GREEN] == $expFmt && [lindex $fmt $::BLUE] == $expFmt && [lindex $fmt $::MATTE] == $expFmt]
        } else {
            set retVal 0
        }
        return [list $w $h $retVal]
    }

    proc poImgsCount {} {
        set nameList [info commands poImage*]
        set count 0
        foreach name $nameList {
            if { $name ne "poImage" && $name ne "poImageState" } {
                incr count
            }
        }
        return $count
    }

    proc poImgsList {} {
        set nameList [info commands poImage*]
        foreach name $nameList {
            if { $name ne "poImage" && $name ne "poImageState" } {
                puts "$name"
            }
        }
    }

    proc useVerbose { { onOff "" } } {
        variable sVerbose
        
        if { $onOff ne "" } {
            set sVerbose $onOff
        } else {
            return $sVerbose
        }
    }

    proc poImgsIdent { poImg1 poImg2 { threshold 0 } } {
        if { $poImg1 eq "" || $poImg2 eq "" } {
            return false
        }

        $poImg1 GetImageInfo w1 h1
        $poImg2 GetImageInfo w2 h2
        if { $w1 != $w2 && $h1 != $h2 } {
            return false
        }

        $poImg1 GetImageFormat fmt1
        $poImg2 GetImageFormat fmt2
        if { ! [intListsIdent $fmt1 $fmt2] } {
            return false
        }

        set diffImg [$poImg1 DifferenceImage $poImg2]
        set markImg [$diffImg MarkNonZeroPixels $threshold numMarked]

        poImgUtil DeleteImage $markImg $diffImg

        if { [poImageState HasPhotoSupport] && $threshold == 0 } {
            # Additionally compare the poImgs by transfering to photo images.
            set phImg1 [image create photo]
            set phImg2 [image create photo]
            $poImg1 AsPhoto $phImg1 [getChanMap $poImg1]
            $poImg2 AsPhoto $phImg2 [getChanMap $poImg2]
            set photosAreIdent [photosIdent $phImg1 $phImg2]
            image delete $phImg1 $phImg2
        } else {
            set photosAreIdent true
        }

        if { $numMarked == 0 && $photosAreIdent } {
            return true
        }
        return false
    }

    proc createOnePixelRGB { fmt w h  x y  r g b } {
        poImageState PushState
        poImgUtil SetFormatAll $::OFF
        poImgUtil SetFormatRGB $fmt $fmt $fmt
        set poImg [poImage NewImage $w $h]
        $poImg Blank
        poImgUtil SetDrawColorRGB $r $g $b
        $poImg DrawPixel $x $y
        poImageState PopState
        return $poImg
    }

    proc createUniformRGB { fmt w h  r g b } {
        poImageState PushState
        poImgUtil SetFormatAll $::OFF
        poImgUtil SetFormatRGB $fmt $fmt $fmt
        set poImg [poImage NewImage $w $h]
        poImgUtil SetDrawColorRGB $r $g $b
        $poImg DrawRect 0 0 $w $h true
        poImageState PopState
        return $poImg
    }

    proc createUniformGrey { fmt w h  value } {
        poImageState PushState
        poImgUtil SetFormatAll $::OFF
        poImgUtil SetFormatChan $::BRIGHTNESS $fmt
        set poImg [poImage NewImage $w $h]
        poImgUtil SetDrawColorChan $::BRIGHTNESS $value
        $poImg DrawRect 0 0 $w $h true
        poImageState PopState
        return $poImg
    }

    proc createUniform { chan fmt w h  value } {
        poImageState PushState
        poImgUtil SetFormatAll $::OFF
        poImgUtil SetFormatChan $chan $fmt
        set poImg [poImage NewImage $w $h]
        poImgUtil SetDrawColorChan $chan $value
        $poImg DrawRect 0 0 $w $h true
        poImageState PopState
        return $poImg
    }

    proc createScanlinesRGB { fmt w h  r1 g1 b1  r2 g2 b2 } {
        poImageState PushState
        poImgUtil SetFormatAll $::OFF
        poImgUtil SetFormatRGB $fmt $fmt $fmt
        set poImg [poImage NewImage $w $h]
        poImgUtil SetDrawColorRGB $r1 $g1 $b1
        for { set y 1 } { $y < $w } { incr y 2 } {
            $poImg DrawLine 0 $y $w $y
        }
        poImgUtil SetDrawColorRGB $r2 $g2 $b2
        for { set y 0 } { $y < $w } { incr y 2 } {
            $poImg DrawLine 0 $y $w $y
        }
        poImageState PopState
        return $poImg
    }

    proc createColorbar3Chan {} {
        return [poImage NewImageFromFile "testIn/Colorbar-3chan-compr.tga"]
    }

    proc createColorbar4Chan {} {
        return [poImage NewImageFromFile "testIn/Colorbar-4chan-compr.tga"]
    }

    proc createBrushImg {} {
        return [poImage NewImageFromFile "testIn/star.bsh"]
    }

    proc photoSize { phImg } {
        return [list [image width  $phImg] [image height $phImg]]
    }

    proc photosIdentTk86 { phImg1 phImg2 } {
        set w1 [image width  $phImg1]
        set h1 [image height $phImg1]
        set w2 [image width  $phImg2]
        set h2 [image height $phImg2]
        if { $w1 != $w2 && $h1 != $h2 } {
            return false
        }
        for { set y 0 } { $y < $h1 } { incr y } {
            for { set x 0 } { $x < $w1 } { incr x } {
                set left  [$phImg1 get $x $y]
                set right [$phImg2 get $x $y]

                set dr [expr { [lindex $right 0] - [lindex $left 0] }]
                if { $dr != 0 } { return false }

                set dg [expr { [lindex $right 1] - [lindex $left 1] }]
                if { $dg != 0 } { return false }

                set db [expr { [lindex $right 2] - [lindex $left 2] }]
                if { $db != 0 } { return false }
            }
        }
        return true
    }

    proc photosIdentTk90 { phImg1 phImg2 } {
        set w1 [image width  $phImg1]
        set h1 [image height $phImg1]
        set w2 [image width  $phImg2]
        set h2 [image height $phImg2]
        if { $w1 != $w2 && $h1 != $h2 } {
            return false
        }
        for { set y 0 } { $y < $h1 } { incr y } {
            for { set x 0 } { $x < $w1 } { incr x } {
                set left  [$phImg1 get $x $y -withalpha]
                set right [$phImg2 get $x $y -withalpha]

                set dr [expr { [lindex $right 0] - [lindex $left 0] }]
                if { $dr != 0 } { return false }

                set dg [expr { [lindex $right 1] - [lindex $left 1] }]
                if { $dg != 0 } { return false }

                set db [expr { [lindex $right 2] - [lindex $left 2] }]
                if { $db != 0 } { return false }

                set da [expr { [lindex $right 3] - [lindex $left 3] }]
                if { $da != 0 } { return false }
            }
        }
        return true
    }

    proc photosIdent { phImg1 phImg2 } {
        if { [haveTk90] } {
            return [photosIdentTk90 $phImg1 $phImg2]
        } else {
            return [photosIdentTk86 $phImg1 $phImg2]
        }
    }

    proc fileRead { fileName } {
        set fp [open $fileName r]
        fconfigure $fp -translation binary
        set imgData [read -nonewline $fp]
        close $fp
        return $imgData
    }

    proc filesIdent { fileName1 fileName2 } {
        if { [file size $fileName1] != [file size $fileName2] } {
            return false
        }
        set data1 [fileRead $fileName1]
        set data2 [fileRead $fileName2]
        if { [string equal $data1 $data2] } {
            return true
        }
        return false
    }

    proc intListsIdent { expected value { msg "intListsIdent" } } {
        # Check, if two lists with integer values are identical.
        #
        # expected   - Expected list.
        # value      - Test list.
        # msg        - Message for test case.
        #
        # Returns true, if both lists are identical.
        # If the check fails, return false and print message prepended with `"Error:"`.

        if { [llength $expected] != [llength $value] } {
            puts "Error: $msg (List length differ. Expected: [llength $expected] Have: [llength $value])"
            return false
        }
        set index 0
        foreach exp $expected val $value {
            if { $exp != $val } {
                puts "Error: $msg (Values differ at index $index. Expected: $exp Have: $val)"
                return false
            }
            incr index
        }
        return true
    }

    proc _abs { a } {
        if { $a >= 0.0 } {
            return $a
        } else {
            return [expr {-1.0 * $a}]
        }
    }

    proc floatListsIdent { expected value { msg "floatListsIdent" } { delta 1.0E-6 } } {
        # Check, if two lists with floating point values are identical.
        #
        # expected   - Expected list.
        # value      - Test list.
        # msg        - Message for test case.
        # delta      - Precision for floating point comparison.
        #
        # Returns true, if both lists are identical.
        # If the check fails, return false and print message prepended with `"Error:"`.

        if { [llength $expected] != [llength $value] } {
            puts "Error: $msg (List length differ. Expected: [llength $expected] Have: [llength $value])"
            return false
        }
        set index 0
        foreach exp $expected val $value {
            if { [_abs [expr {$exp - $val}]] > $delta } {
                puts "Error: $msg (Values differ at index $index. Expected: $exp Have: $val)"
                return false
            }
            incr index
        }
        return true
    }

    proc photoView { phImg args } {
        variable ns
        variable sTopCount

        set opts [dict create \
            -zoom  1 \
            -title "$phImg" \
        ]

        foreach { key value } $args {
            if { [dict exists $opts $key] } {
                if { $value eq "" } {
                    error "photoView: No value specified for key \"$key\"."
                }
                dict set opts $key $value
            } else {
                error "photoView: Unknown option \"$key\" specified."
            }
        }

        wm withdraw .
        set top .t_$sTopCount
        incr sTopCount
        toplevel $top
        set phZoom [image create photo]
        $phZoom copy $phImg -zoom [dict get $opts "-zoom"]
        label $top.l -image $phZoom -background magenta
        pack $top.l
        wm title $top [dict get $opts "-title"]
        bind $top <Escape> "set ${ns}::sContinue 1 ; destroy $top"
        focus -force $top
        update
        vwait ${ns}::sContinue
    }

    proc getChanMap { poImg } {
        $poImg GetImageFormat fmtList
        set chanMap [list]
        if { [lindex $fmtList $::RED] && [lindex $fmtList $::GREEN] && [lindex $fmtList $::BLUE] } {
             if { [lindex $fmtList $::MATTE] } {
                set chanMap [list $::RED $::GREEN $::BLUE $::MATTE]
            } else {
                set chanMap [list $::RED $::GREEN $::BLUE]
            }
        } elseif { [lindex $fmtList $::BRIGHTNESS] } {
            if { [lindex $fmtList $::MATTE] } {
                return [list $::BRIGHTNESS $::BRIGHTNESS $::BRIGHTNESS $::MATTE]
            } else {
                return [list $::BRIGHTNESS $::BRIGHTNESS $::BRIGHTNESS]
            }
        } elseif { [lindex $fmtList $::MATTE] } {
            set chanMap [list $::MATTE $::MATTE $::MATTE]
        } elseif { [lindex $fmtList $::RED] } {
            set chanMap [list $::RED $::RED $::RED]
        } elseif { [lindex $fmtList $::GREEN] } {
            set chanMap [list $::GREEN $::GREEN $::GREEN]
        } elseif { [lindex $fmtList $::BLUE] } {
            set chanMap [list $::BLUE $::BLUE $::BLUE]
        } elseif { [lindex $fmtList $::DEPTH] } {
            set chanMap [list $::DEPTH $::DEPTH $::DEPTH]
        } elseif { [lindex $fmtList $::TEMPERATURE] } {
            set chanMap [list $::TEMPERATURE $::TEMPERATURE $::TEMPERATURE]
        } elseif { [lindex $fmtList $::RADIANCE] } {
            set chanMap [list $::RADIANCE $::RADIANCE $::RADIANCE]
        }
        return $chanMap
    }

    proc poImgView { poImg args } {
        set opts [dict create \
            -zoom     1 \
            -title    "$poImg" \
            -chanmap  [getChanMap $poImg] \
            -valuemap "1 0 1" \
        ]
        foreach { key value } $args {
            if { [dict exists $opts $key] } {
                if { $value eq "" } {
                    error "poImgView: No value specified for key \"$key\"."
                }
                dict set opts $key $value
            } else {
                error "poImgView: Unknown option \"$key\" specified."
            }
        }

        set phImg [image create photo]
        $poImg AsPhoto $phImg [dict get $opts "-chanmap"] {*}[dict get $opts "-valuemap"]
        photoView $phImg -zoom [dict get $opts "-zoom"] -title [dict get $opts "-title"]
    }

    proc GetGreyMatrixBU-byte {} {
        return { \
            { 255 200 150 100 50    0 } \
            {   0  50 100 150 200 255 } \
        }
    }

    proc GetGreyAlphaMatrixBU-byte {} {
        return { \
            { 255 255  200 255  150 255  100 255   50 255    0 255 } \
            {   0   0   50  50  100 100  150 150  200 200  255 255 } \
        }
    }

    proc GetRgbMatrixBU-byte {} {
        return { \
            { 128 128 128     0   0 255     0 255 0   255   0 0   255 255 255     0   0   0 } \
            {   0   0   0   255 255 255   255   0 0     0 255 0     0   0 255   128 128 128 } \
        }
    }

    proc GetRgbAlphaMatrixBU-byte {} {
        return { \
            { 128 128 128 255     0   0 255  255    0 255 0 255  255   0 0 255  255 255 255 255     0   0   0 255 } \
            {   0   0   0   0   255 255 255  50   255   0 0 100    0 255 0 150    0   0 255 200   128 128 128 255 } \
        }
    }

    proc GetGreyMatrixBU-short {} {
        return { \
            { 65535 52000 39000 26000 13000     0 } \
            {     0 13000 26000 39000 52000 65535 } \
        }
    }

    proc GetGreyAlphaMatrixBU-short {} {
        return { \
            { 65535 65535  52000 65535  39000 65535  26000 65535  13000 65535      0 65535 } \
            {     0     0  13000 13000  26000 26000  39000 39000  52000 52000  65535 65535 } \
        }
    }

    proc GetRgbMatrixBU-short {} {
        return { \
            { 30000 30000 30000       0     0 65535       0 65535 0   65535     0 0   65535 65535 65535        0    0     0 } \
            {     0     0     0   65535 65535 65535   65535     0 0       0 65535 0       0     0 65535   30000 30000 30000 } \
        }
    }

    proc GetRgbAlphaMatrixBU-short {} {
        return { \
            { 30000 30000 30000 65535      0     0 65535 65535      0 65535 0 65535  65535     0 0 65535  65535 65535 65535 65535      0    0      0 65535 } \
            {     0     0     0     0  65535 65535 65535 13000  65535     0 0 26000      0 65535 0 39000      0     0 65535 52000  30000 30000 30000 65535 } \
        }
    }

    proc GetGreyMatrixBU-int {} {
        return [GetGreyMatrixBU-short]
    }

    proc GetGreyAlphaMatrixBU-int {} {
        return [GetGreyAlphaMatrixBU-short]
    }

    proc GetRgbMatrixBU-int {} {
        return [GetRgbMatrixBU-short]
    }

    proc GetRgbAlphaMatrixBU-int {} {
        return [GetRgbAlphaMatrixBU-short]
    }

    proc GetGreyMatrixBU-float {} {
        return { \
            { 1.0 0.8 0.6 0.4 0.2 0.0 } \
            { 0.0 0.2 0.4 0.6 0.8 1.0 } \
        }
    }

    proc GetGreyAlphaMatrixBU-float {} {
        return { \
            { 1.0 1.0  0.8 1.0  0.6 1.0  0.4 1.0  0.2 1.0  0.0 1.0 } \
            { 0.0 0.0  0.2 0.2  0.4 0.4  0.6 0.6  0.8 0.8  1.0 1.0 } \
        }
    }

    proc GetRgbMatrixBU-float {} {
        return { \
            { 0.5 0.5 0.5   0.0 0.0 1.0   0.0 1.0 0.0   1.0 0.0 0.0   1.0 1.0 1.0   0.0 0.0 0.0 } \
            { 0.0 0.0 0.0   1.0 1.0 1.0   1.0 0.0 0.0   0.0 1.0 0.0   0.0 0.0 1.0   0.5 0.5 0.5 } \
        }
    }

    proc GetRgbAlphaMatrixBU-float {} {
        return { \
            { 0.5 0.5 0.5 1.0   0.0 0.0 1.0 1.0   0.0 1.0 0.0 1.0   1.0 0.0 0.0 1.0   1.0 1.0 1.0 1.0   0.0 0.0 0.0 1.0 } \
            { 0.0 0.0 0.0 0.0   1.0 1.0 1.0 0.2   1.0 0.0 0.0 0.4   0.0 1.0 0.0 0.6   0.0 0.0 1.0 0.8   0.5 0.5 0.5 1.0 } \
        }
    }

    proc GetGreyMatrixBU-double {} {
        return [GetGreyMatrixBU-float]
    }

    proc GetGreyAlphaMatrixBU-double {} {
        return [GetGreyAlphaMatrixBU-float]
    }

    proc GetRgbMatrixBU-double {} {
        return [GetRgbMatrixBU-float]
    }

    proc GetRgbAlphaMatrixBU-double {} {
        return [GetRgbAlphaMatrixBU-float]
    }

    proc _GetPixelValue-byte { matrix x y numChans } {
        set height  [llength $matrix]
        set rowList [lindex $matrix $y]
        set index1  [expr { $x * $numChans}]
        set index2  [expr { $index1 + $numChans - 1}]
        foreach val [lrange $rowList $index1 $index2] {
            lappend floatList [expr { $val / 255.0}]
        }
        return $floatList
    }

    proc _GetPixelValue-short { matrix x y numChans } {
        set height  [llength $matrix]
        set rowList [lindex $matrix $y]
        set index1  [expr { $x * $numChans}]
        set index2  [expr { $index1 + $numChans - 1}]
        return [lrange $rowList $index1 $index2]
    }

    proc _GetPixelValue-int { matrix x y numChans } {
        set height  [llength $matrix]
        set rowList [lindex $matrix $y]
        set index1  [expr { $x * $numChans}]
        set index2  [expr { $index1 + $numChans - 1}]
        return [lrange $rowList $index1 $index2]
    }

    proc _GetPixelValue-float { matrix x y numChans } {
        set height  [llength $matrix]
        set rowList [lindex $matrix $y]
        set index1  [expr { $x * $numChans}]
        set index2  [expr { $index1 + $numChans - 1}]
        return [lrange $rowList $index1 $index2]
    }

    proc _GetPixelValue-double { matrix x y numChans } {
        set height  [llength $matrix]
        set rowList [lindex $matrix $y]
        set index1  [expr { $x * $numChans}]
        set index2  [expr { $index1 + $numChans - 1}]
        return [lrange $rowList $index1 $index2]
    }

    proc GetGreyPixelValue-byte { x y } {
        set matrix [GetGreyMatrixBU-byte]
        return [_GetPixelValue-byte $matrix $x $y 1]
    }

    proc GetGreyAlphaPixelValue-byte { x y } {
        set matrix [GetGreyAlphaMatrixBU-byte]
        return [_GetPixelValue-byte $matrix $x $y 2]
    }

    proc GetRgbPixelValue-byte { x y } {
        set matrix [GetRgbMatrixBU-byte]
        return [_GetPixelValue-byte $matrix $x $y 3]
    }

    proc GetRgbAlphaPixelValue-byte { x y } {
        set matrix [GetRgbAlphaMatrixBU-byte]
        return [_GetPixelValue-byte $matrix $x $y 4]
    }

    proc GetGreyPixelValue-short { x y } {
        set matrix [GetGreyMatrixBU-short]
        return [_GetPixelValue-short $matrix $x $y 1]
    }

    proc GetGreyAlphaPixelValue-short { x y } {
        set matrix [GetGreyAlphaMatrixBU-short]
        return [_GetPixelValue-short $matrix $x $y 2]
    }

    proc GetRgbPixelValue-short { x y } {
        set matrix [GetRgbMatrixBU-short]
        return [_GetPixelValue-short $matrix $x $y 3]
    }

    proc GetRgbAlphaPixelValue-short { x y } {
        set matrix [GetRgbAlphaMatrixBU-short]
        return [_GetPixelValue-short $matrix $x $y 4]
    }

    proc GetGreyPixelValue-int { x y } {
        set matrix [GetGreyMatrixBU-int]
        return [_GetPixelValue-int $matrix $x $y 1]
    }

    proc GetGreyAlphaPixelValue-int { x y } {
        set matrix [GetGreyAlphaMatrixBU-int]
        return [_GetPixelValue-int $matrix $x $y 2]
    }

    proc GetRgbPixelValue-int { x y } {
        set matrix [GetRgbMatrixBU-int]
        return [_GetPixelValue-int $matrix $x $y 3]
    }

    proc GetRgbAlphaPixelValue-int { x y } {
        set matrix [GetRgbAlphaMatrixBU-int]
        return [_GetPixelValue-int $matrix $x $y 4]
    }

    proc GetGreyPixelValue-float { x y } {
        set matrix [GetGreyMatrixBU-float]
        return [_GetPixelValue-float $matrix $x $y 1]
    }

    proc GetGreyAlphaPixelValue-float { x y } {
        set matrix [GetGreyAlphaMatrixBU-float]
        return [_GetPixelValue-float $matrix $x $y 2]
    }

    proc GetRgbPixelValue-float { x y } {
        set matrix [GetRgbMatrixBU-float]
        return [_GetPixelValue-float $matrix $x $y 3]
    }

    proc GetRgbAlphaPixelValue-float { x y } {
        set matrix [GetRgbAlphaMatrixBU-float]
        return [_GetPixelValue-float $matrix $x $y 4]
    }

    proc GetGreyPixelValue-double { x y } {
        set matrix [GetGreyMatrixBU-double]
        return [_GetPixelValue-double $matrix $x $y 1]
    }

    proc GetGreyAlphaPixelValue-double { x y } {
        set matrix [GetGreyAlphaMatrixBU-double]
        return [_GetPixelValue-double $matrix $x $y 2]
    }

    proc GetRgbPixelValue-double { x y } {
        set matrix [GetRgbMatrixBU-double]
        return [_GetPixelValue-double $matrix $x $y 3]
    }

    proc GetRgbAlphaPixelValue-double { x y } {
        set matrix [GetRgbAlphaMatrixBU-double]
        return [_GetPixelValue-double $matrix $x $y 4]
    }

    proc getByteOrderFromRawFilename { fileName } {
        return "Intel"
    }

    proc getScanOrderFromRawFilename { fileName } {
        set so [lindex [split [file rootname $fileName] "-"] 2]
        if { $so eq "td" } {
            return "TopDown"
        } else {
            return "BottomUp"
        }
    }

    proc getPixelTypeFromRawFilename { fileName } {
        return [lindex [split $fileName "-"] 0]
    }

    proc getFormatFromRawFilename { fileName } {
        set pixelType [getPixelTypeFromRawFilename $fileName]
        if { $pixelType eq "byte" } {
            return $::UBYTE
        } else {
            return $::FLOAT
        }
    }

    proc rawTestImgCheckStats { poImg numChans pixelType } {
        switch -exact -- $numChans {
            1 { set cmdType "GetGreyMatrixBU" }
            2 { set cmdType "GetGreyAlphaMatrixBU" }
            3 { set cmdType "GetRgbMatrixBU" }
            4 { set cmdType "GetRgbAlphaMatrixBU" }
        }
        set cmd "${cmdType}-${pixelType}"
        set matrix [$cmd]

        if { $numChans == 2 || $numChans == 4 } {
            $poImg GetChannelStats $::MATTE -1 -1 -1 -1 mean($numChans) stdDev($numChans) numPix($numChans)
            $poImg GetChannelRange $::MATTE -1 -1 -1 -1 min($numChans) max($numChans)
        }
        if { $numChans == 1 || $numChans == 2 } {
            $poImg GetChannelStats $::BRIGHTNESS -1 -1 -1 -1 mean(1) stdDev(1) numPix(1)
            $poImg GetChannelRange $::BRIGHTNESS -1 -1 -1 -1 min(1) max(1)
        }
        if { $numChans == 3 || $numChans == 4 } {
            $poImg GetChannelStats $::RED   -1 -1 -1 -1 mean(1) stdDev(1) numPix(1)
            $poImg GetChannelStats $::GREEN -1 -1 -1 -1 mean(2) stdDev(2) numPix(2)
            $poImg GetChannelStats $::BLUE  -1 -1 -1 -1 mean(3) stdDev(3) numPix(3)

            $poImg GetChannelRange $::RED   -1 -1 -1 -1 min(1) max(1)
            $poImg GetChannelRange $::GREEN -1 -1 -1 -1 min(2) max(2)
            $poImg GetChannelRange $::BLUE  -1 -1 -1 -1 min(3) max(3)
        }

        set num 0
        # Calculate sum, min and max values.
        foreach rowList $matrix {
            for { set c 0 } { $c < [llength $rowList] } { incr c $numChans } {
                for { set chan 1 } { $chan <= $numChans } { incr chan } {
                    set chanVal [lindex $rowList [expr {$c + $chan - 1}]]
                    if { ! [info exists sum($chan)] } {
                        set sum($chan) $chanVal
                    } else {
                        set sum($chan) [expr { $sum($chan) + $chanVal }]
                    }
                    if { ! [info exists minRef($chan)] } {
                        set minRef($chan) $chanVal
                    } elseif { $chanVal < $minRef($chan) } {
                        set minRef($chan) $chanVal
                    }
                    if { ! [info exists maxRef($chan)] } {
                        set maxRef($chan) $chanVal
                    } elseif { $chanVal > $maxRef($chan) } {
                        set maxRef($chan) $chanVal
                    }
                }
                incr num
            }
        }

        # Calculate mean.
        for { set chan 1 } { $chan <= $numChans } { incr chan } {
            set meanRef($chan) [expr { double($sum($chan)) / double($num) }]
        }

        # Calculate standard deviation. Code taken from tcllib.
        foreach rowList $matrix {
            for { set c 0 } { $c < [llength $rowList] } { incr c $numChans } {
                for { set chan 1 } { $chan <= $numChans } { incr chan } {
                    set chanVal [lindex $rowList [expr {$c + $chan - 1}]]
                    if { ! [info exists sigma_sq($chan)] } {
                        set sigma_sq($chan) [expr { pow(($chanVal-$mean($chan)),2) }]
                    } else {
                        set sigma_sq($chan) [expr { $sigma_sq($chan)+pow(($chanVal-$mean($chan)),2) }]
                    }
                }
            }
        }
        for { set chan 1 } { $chan <= $numChans } { incr chan } {
            set sigma_sq($chan) [expr { $sigma_sq($chan)/double($num-1) }]
            set stdDevRef($chan) [expr { sqrt($sigma_sq($chan)) }]
        }

        for { set chan 1 } { $chan <= $numChans } { incr chan } {
            lappend refList $meanRef($chan) $stdDevRef($chan) $minRef($chan) $maxRef($chan)
            lappend imgList $mean($chan) $stdDev($chan) $min($chan) $max($chan)
        }
        if { ! [floatListsIdent $refList $imgList] } {
            return false
        }
        return true
    }

    proc rawTestImgCheck { poImg numChans withAlpha pixelType expFmt } {
        if { $poImg eq "" } {
            return false
        }
        $poImg GetImageInfo w h
        $poImg GetImageFormat fmt
        if { ($numChans == 2 || $numChans == 4) && !$withAlpha } {
            incr numChans -1
        }
        if { $numChans == 1 } {
            set retVal [expr [lindex $fmt $::BRIGHTNESS] == $expFmt]
        } elseif { $numChans == 2 } {
            set retVal [expr [lindex $fmt $::BRIGHTNESS] == $expFmt && [lindex $fmt $::MATTE] == $expFmt]
        } elseif { $numChans == 3 } {
            set retVal [expr [lindex $fmt $::RED] == $expFmt && [lindex $fmt $::GREEN] == $expFmt && [lindex $fmt $::BLUE] == $expFmt]
        } elseif { $numChans == 4 } {
            set retVal [expr [lindex $fmt $::RED] == $expFmt && [lindex $fmt $::GREEN] == $expFmt && [lindex $fmt $::BLUE] == $expFmt && [lindex $fmt $::MATTE] == $expFmt]
        } else {
            set retVal 0
        }
        if { ! $retVal } {
            return false
        }

        for { set y 0 } { $y < $h } { incr y } {
            for { set x 0 } { $x < $w } { incr x } {
                $poImg GetPixel $x $y colorList
                if { $pixelType eq "byte" } {
                    if { $numChans == 1 } {
                        if { ! [floatListsIdent [GetGreyPixelValue-byte $x $y] [lrange $colorList $::BRIGHTNESS $::BRIGHTNESS]] } {
                            return false
                        }
                    } elseif { $numChans == 2 } {
                        if { ! [floatListsIdent [GetGreyAlphaPixelValue-byte $x $y] [list [lindex $colorList $::BRIGHTNESS] [lindex $colorList $::MATTE]]] } {
                            return false
                        }
                    } elseif { $numChans == 3 } {
                        if { ! [floatListsIdent [GetRgbPixelValue-byte $x $y] [lrange $colorList $::RED $::BLUE]] } {
                            return false
                        }
                    } elseif { $numChans == 4 } {
                        if { ! [floatListsIdent [GetRgbAlphaPixelValue-byte $x $y] [lrange $colorList $::RED $::MATTE]] } {
                            return false
                        }
                    }
                } elseif { $pixelType eq "short" } {
                    if { $numChans == 1 } {
                        if { ! [floatListsIdent [GetGreyPixelValue-short $x $y] [lrange $colorList $::BRIGHTNESS $::BRIGHTNESS]] } {
                            return false
                        }
                    } elseif { $numChans == 2 } {
                        if { ! [floatListsIdent [GetGreyAlphaPixelValue-short $x $y] [list [lindex $colorList $::BRIGHTNESS] [lindex $colorList $::MATTE]]] } {
                            return false
                        }
                    } elseif { $numChans == 3 } {
                        if { ! [floatListsIdent [GetRgbPixelValue-short $x $y] [lrange $colorList $::RED $::BLUE]] } {
                            return false
                        }
                    } elseif { $numChans == 4 } {
                        if { ! [floatListsIdent [GetRgbAlphaPixelValue-short $x $y] [lrange $colorList $::RED $::MATTE]] } {
                            return false
                        }
                    }
                } elseif { $pixelType eq "int" } {
                    if { $numChans == 1 } {
                        if { ! [floatListsIdent [GetGreyPixelValue-int $x $y] [lrange $colorList $::BRIGHTNESS $::BRIGHTNESS]] } {
                            return false
                        }
                    } elseif { $numChans == 2 } {
                        if { ! [floatListsIdent [GetGreyAlphaPixelValue-int $x $y] [list [lindex $colorList $::BRIGHTNESS] [lindex $colorList $::MATTE]]] } {
                            return false
                        }
                    } elseif { $numChans == 3 } {
                        if { ! [floatListsIdent [GetRgbPixelValue-int $x $y] [lrange $colorList $::RED $::BLUE]] } {
                            return false
                        }
                    } elseif { $numChans == 4 } {
                        if { ! [floatListsIdent [GetRgbAlphaPixelValue-int $x $y] [lrange $colorList $::RED $::MATTE]] } {
                            return false
                        }
                    }
                } elseif { $pixelType eq "float" } {
                    if { $numChans == 1 } {
                        if { ! [floatListsIdent [GetGreyPixelValue-float $x $y] [lrange $colorList $::BRIGHTNESS $::BRIGHTNESS]] } {
                            return false
                        }
                    } elseif { $numChans == 2 } {
                        if { ! [floatListsIdent [GetGreyAlphaPixelValue-float $x $y] [list [lindex $colorList $::BRIGHTNESS] [lindex $colorList $::MATTE]]] } {
                            return false
                        }
                    } elseif { $numChans == 3 } {
                        if { ! [floatListsIdent [GetRgbPixelValue-float $x $y] [lrange $colorList $::RED $::BLUE]] } {
                            return false
                        }
                    } elseif { $numChans == 4 } {
                        if { ! [floatListsIdent [GetRgbAlphaPixelValue-float $x $y] [lrange $colorList $::RED $::MATTE]] } {
                            return false
                        }
                    }
                } elseif { $pixelType eq "double" } {
                    if { $numChans == 1 } {
                        if { ! [floatListsIdent [GetGreyPixelValue-double $x $y] [lrange $colorList $::BRIGHTNESS $::BRIGHTNESS]] } {
                            return false
                        }
                    } elseif { $numChans == 2 } {
                        if { ! [floatListsIdent [GetGreyAlphaPixelValue-double $x $y] [list [lindex $colorList $::BRIGHTNESS] [lindex $colorList $::MATTE]]] } {
                            return false
                        }
                    } elseif { $numChans == 3 } {
                        if { ! [floatListsIdent [GetRgbPixelValue-double $x $y] [lrange $colorList $::RED $::BLUE]] } {
                            return false
                        }
                    } elseif { $numChans == 4 } {
                        if { ! [floatListsIdent [GetRgbAlphaPixelValue-double $x $y] [lrange $colorList $::RED $::MATTE]] } {
                            return false
                        }
                    }
                } else {
                    puts "Unknown pixeltype $pixelType"
                    return false
                }
            } 
        }
        return true
    }
}

namespace import -force poImgTest::*

tcltest::testConstraint HaveTk86 [haveTk86]
tcltest::testConstraint HaveTk90 [haveTk90]

tcltest::testConstraint HavePkgImg [havePkgImg]
tcltest::testConstraint HavePkgMem [havePkgMem]

tcltest::testConstraint HavePhotoSupport [havePhotoSupport]
