# Copyright: 2007-2025 Paul Obermeier (obermeier@poSoft.de)
# Distributed under BSD license.

set opts(Verbose) false
set opts(Delete)  true
set opts(Auto)    true
set opts(Compare) true

set patternList  [list]
set testFileList [list]
set useAllFiles  false

set tclsh [info nameofexecutable]
set curDir [file dirname [info script]]

proc PrintUsage { progName { msg "" } } {
    global opts nsAvailList

    puts ""
    if { $msg ne "" } {
        puts "Error: $msg"
    }
    puts ""
    puts "Usage: $progName \[Options\] FilePattern \[FilePattern\]"
    puts ""
    puts "Run the test programs for specified glob-style file pattern(s)."
    puts "Use \"all\" as file pattern to run all available test programs."
    puts ""
    puts "Options:"
    puts "  --help     : Display this usage message and exit."
    puts "  --verbose  : Show the detailed results of the tests. (Default: No)"
    puts "  --nodelete : Do not delete test output files. (Default: Yes)"
    puts "  --noauto   : Do not run tests in automatic mode. (Default: Yes)"
    puts "  --nocompare: Do not compare images with reference images. (Default: Yes)"
    puts ""
}

proc AreImgsIdent { fileName1 fileName2 threshold } {
    package require poImg

    set catchVal [catch {poImage NewImageFromFile $fileName1} img1]
    if { $catchVal } {
        return false
    }
    set catchVal [catch {poImage NewImageFromFile $fileName2} img2]
    if { $catchVal } {
        return false
    }

    set diffImg [$img1 DifferenceImage $img2]
    set markImg [$diffImg MarkNonZeroPixels $threshold numMarked]
    poImgUtil DeleteImage $diffImg
    poImgUtil DeleteImage $markImg
    poImgUtil DeleteImage $img1
    poImgUtil DeleteImage $img2
    if { $numMarked != 0 } {
        return false
    }
    return true
}

proc ArePhotosIdent { fileName1 fileName2 } {
    set phImg1 [image create photo -file $fileName1]
    set phImg2 [image create photo -file $fileName2]

    set w1 [image width  $phImg1]
    set h1 [image height $phImg1]
    set w2 [image width  $phImg2]
    set h2 [image height $phImg2]
    if { $w1 != $w2 && $h1 != $h2 } {
        return false
    }
    for { set y 0 } { $y < $h1 } { incr y } {
        for { set x 0 } { $x < $w1 } { incr x } {
            set left  [$phImg1 get $x $y]
            set right [$phImg2 get $x $y]

            set dr [expr { [lindex $right 0] - [lindex $left 0] }]
            if { $dr != 0 } { return false }

            set dg [expr { [lindex $right 1] - [lindex $left 1] }]
            if { $dg != 0 } { return false }

            set db [expr { [lindex $right 2] - [lindex $left 2] }]
            if { $db != 0 } { return false }
        }
    }
    return true
}

proc RunTest { testFile } {
    global opts

    puts "Running test $testFile with $::tclsh ..."
    if { $opts(Auto) } {
        set catchVal [catch {exec -ignorestderr $::tclsh $testFile auto 2>@1 } retVal optionsDict]
    } else {
        set catchVal [catch {exec -ignorestderr $::tclsh $testFile 2>@1 } retVal optionsDict]
    }
    if { $catchVal || [string match "*Error:*" $retVal] } {
        if { $catchVal } {
            set fullErrorInfo [dict get $optionsDict -errorinfo]
            set msgEndIndex [string first "\n" $fullErrorInfo]
            set msg [string range $fullErrorInfo 0 [expr {$msgEndIndex -1}]]
        } else {
            foreach line [split $retVal "\n"] {
                if { [string match "*Error:*" $line] } {
                    append msg $line
                }
            }
        }
        puts "Test $testFile failed: $msg"
    } else {
        if { $opts(Verbose) } {
            puts $retVal
            puts ""
        }
    }
}

set curArg 0
while { $curArg < $argc } {
    set curParam [lindex $argv $curArg]
    if { [string compare -length 1 $curParam "-"]  == 0 || \
         [string compare -length 2 $curParam "--"] == 0 } {
        set curOpt [string tolower [string trimleft $curParam "-"]]
        if { $curOpt eq "verbose" } {
            set opts(Verbose) true
        } elseif { $curOpt eq "nodelete" } {
            set opts(Delete) false
        } elseif { $curOpt eq "noauto" } {
            set opts(Auto) false
        } elseif { $curOpt eq "nocompare" } {
            set opts(Compare) false
        } elseif { $curOpt eq "help" } {
            PrintUsage $argv0
            exit 1
        } elseif { $curOpt eq "load" } {
            # Added by test target.
        } else {
            PrintUsage $argv0 "Invalid option \"$curParam\" specified."
            exit 1
        }
    } else {
        if { $curParam eq "all" } {
            set useAllFiles true
        } else {
            lappend patternList $curParam
        }
    }
    incr curArg
}

cd $curDir
if { $useAllFiles } {
    set testFileList [lsort [glob -nocomplain "t_*.tcl"]]
} else {
    foreach pattern $patternList {
        lappend testFileList {*}[lsort [glob -nocomplain $pattern]]
    }
}

if { [llength $testFileList] == 0 } {
    PrintUsage $argv0 "No valid file pattern specified."
    exit 1
}

catch { file mkdir testOut }
foreach testFile $testFileList {
    RunTest $testFile
}
if { $opts(Compare) } {
    # As some functionality (warping, anti-aliasing) use floating-point arithmetic,
    # we get slightly different image results depending on the used architecture.
    # For identical results floating-point calculations must be done using IEEE
    # arithmetics, which result in slower execution times.
    # A threshold of 1 pixel value is therefore accepted.
    set refFileList [lsort -dictionary [glob -tails -directory "refOut"  "*"]]
    set imgFileList [lsort -dictionary [glob -tails -directory "testOut" "*"]]
    puts "Comparing generated images with reference images using thresholds 0 and 1 ..."

    foreach refFile $refFileList {
        if { [lsearch $imgFileList $refFile] >= 0 } {
            if { [file extension $refFile] eq ".png" } {
                set res [ArePhotosIdent [file join "refOut" $refFile] [file join "testOut" $refFile]]
                if { ! $res } {
                    puts "ERROR:  Images not identical (Threshold 0): $refFile"
                }
            } else {
                set res0 [AreImgsIdent [file join "refOut" $refFile] [file join "testOut" $refFile] 0]
                if { ! $res0 } {
                    set res1 [AreImgsIdent [file join "refOut" $refFile] [file join "testOut" $refFile] 1]
                    if { ! $res1 } {
                        puts "ERROR:  Images not identical (Threshold 1): $refFile"
                    } else {
                        puts "Notice: Images not identical (Threshold 0): $refFile"
                    }
                }
            }
        } else {
            puts "Error: Reference image $refFile is not available in testOut"
        }
    }
}
if { $opts(Delete) } {
    catch { file delete -force testOut }
}

exit 0
