# Module:         poToolhelp
# Copyright:      Paul Obermeier 2000-2025 / paul@poSoft.de
# First Version:  2000 / 01 / 22
#
# Distributed under BSD license.
#
# Module for handling a toolhelp window. 

namespace eval poToolhelp {
    variable ns [namespace current]

    namespace ensemble create

    namespace export Init
    namespace export ShowToolhelp HideToolhelp
    namespace export AddBinding

    proc Init { tw { bgColor yellow } { fgColor black } } {
        variable pkgInt

        set pkgInt(tw) $tw
        set pkgInt(bg) $bgColor
        set pkgInt(fg) $fgColor
    }
    
    proc ShowToolhelp { w msg } {
        variable pkgInt

        set t $pkgInt(tw)
        catch {destroy $t}
        toplevel $t -bg $pkgInt(bg)
        wm overrideredirect $t yes
        if {[string equal [tk windowingsystem] aqua]}  {
            ::tk::unsupported::MacWindowStyle style $t help none
        }
        label $t.l -text [subst -nocommands $msg] -bg $pkgInt(bg) -fg $pkgInt(fg) -relief ridge -justify left
        pack $t.l -padx 1 -pady 1
        set width  [expr {[winfo reqwidth $t.l]  + 2}]
        set height [expr {[winfo reqheight $t.l] + 2}]
        set xMax   [expr {[winfo screenwidth $w]  - $width}]
        set yMax   [expr {[winfo screenheight $w] - $height}]
        set x      [winfo pointerx $w]
        set y      [expr {[winfo pointery $w] + 20}]
        if {$x > $xMax} then {
            set x $xMax
        }
        if {$y > $yMax} then {
            set y $yMax
        }
        wm geometry $t +$x+$y
        set destroyScript [list destroy $t]
        bind $t <Enter> [list after cancel $destroyScript]
        bind $t <Leave> $destroyScript
    }

    proc HideToolhelp {} {
        variable pkgInt

        if { [info exists pkgInt(tw)] } {
            destroy $pkgInt(tw)
        }
    }

    proc AddBinding { w msg args } {
        variable ns
        variable pkgInt

        if { ! [info exists pkgInt(tw)] } {
            Init .poToolhelp
        }

        array set opt [concat { -tag "" } $args]
        if {$msg ne ""} then {
            set toolTipScript [list ${ns}::ShowToolhelp %W [string map {% %%} $msg]]
            set enterScript [list after 200 $toolTipScript]
            set leaveScript [list after cancel $toolTipScript]
            append leaveScript \n [list after 200 [list destroy $pkgInt(tw)]]
        } else {
            set enterScript {}
            set leaveScript {}
        }
        if {$opt(-tag) ne ""} then {
            switch -- [winfo class $w] {
                Text {
                    $w tag bind $opt(-tag) <Enter> $enterScript
                    $w tag bind $opt(-tag) <Leave> $leaveScript
                }
                Canvas {
                    $w bind $opt(-tag) <Enter> $enterScript
                    $w bind $opt(-tag) <Leave> $leaveScript
                }
                default {
                    bind $w <Enter> $enterScript
                    bind $w <Leave> $leaveScript
                }
            }
        } else {
            bind $w <Enter> $enterScript
            bind $w <Leave> $leaveScript
        }
    }
}

catch {poLog Debug "Loaded Package poTklib (Module [info script])"}
